/*
 * $Id: RemoteFileInputStream.java 114793 2011-01-21 17:21:29Z cassino $
 */

package csbase.logic;

import java.io.IOException;
import java.io.InputStream;

import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;

/**
 * Implementa um leitor (stream) de um arquivo (client-side).
 * 
 * @author Tecgraf/PUC-Rio
 */
public class RemoteFileInputStream extends InputStream {
  /**
   * Identificador do arquivo remoto.
   */
  private byte[] id = null;

  /**
   * Canal de acesso ao arquivo a ler lido.
   */
  private SyncRemoteFileChannel channel = null;

  /**
   * Marca anteriormente registrada para o stream (relativa ao arquivo).
   */
  private long mark = -1;

  /**
   * Posio corrente no arquivo
   */
  private long position = 0;

  /**
   * {@inheritDoc}
   */
  @Override
  public int available() {
    return (int) Math.min(channel.getSize() - position, Integer.MAX_VALUE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws IOException {
    if (channel == null) {
      return;
    }
    try {
      channel.close();
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
    id = null;
    channel = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void mark(int readlimit) {
    mark = position;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean markSupported() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read() throws IOException {
    byte[] dst = new byte[1];
    try {
      int n = channel.syncRead(dst, 0, 1, position);
      if (n == 0) {
        return -1;
      }
      return dst[0];
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst) throws IOException {
    return read(dst, 0, dst.length);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, int off, int len) throws IOException {
    try {
      int n = channel.syncRead(dst, off, len, position);
      if (n == 0) {
        return -1;
      }
      position += n;
      return n;
    }
    catch (FailureException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void reset() throws IOException {
    position = mark;
    mark = -1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long skip(long n) throws IOException {
    n = Math.min(n, channel.getSize() - position);
    position += n;
    return n;
  }

  /**
   * Cria um <code>InputStream</code> a partir de um canal de acesso a um
   * arquivo remoto.
   * 
   * @param info Informaes para conexo do canal.
   * 
   * @throws IOException Em caso de falha na operao.
   */
  public RemoteFileInputStream(RemoteFileChannelInfo info) throws IOException {
    this.id = info.getIdentifier();
    this.channel = new SyncRemoteFileChannel(info);
    try {
      channel.open(true);
    }
    catch (Exception e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * Cria um <code>InputStream</code> a partir de um canal de acesso a um
   * arquivo remoto.
   * 
   * @param id Indentificador do arquivo remoto.
   * @param channel Canal de acesso ao arquivo remoto. Deve ter sido aberto
   *        antes de ser passado a esse construtor.
   */
  public RemoteFileInputStream(byte[] id, SyncRemoteFileChannel channel) {
    this.id = id;
    this.channel = channel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return "RemoteFileInputStream para " + new String(id);
  }
}
