package csbase.logic.algorithms.parameters;

import java.io.Serializable;
import java.util.regex.Pattern;

import csbase.exception.ParseException;
import csbase.logic.ProjectFileType;
import tecgraf.javautils.core.io.FileUtils;

/**
 * <p>
 * Classe que representa uma URL de arquivo.  utilizada como valor para os
 * parmetros do tipo FileParameter e URLParameter.
 * <p>
 * Para FileParameter, apenas os campos path e type so utilizados.
 * <p>
 * Para URLParameter, o protocolo indica se o arquivo est no Projeto, no SGA ou
 * Local (se a busca deve ser local). Caso o protocolo seja SGA, o host deve
 * estar preenchido com o nome do SGA.
 */
public final class FileURLValue implements Comparable<FileURLValue>,
  Serializable, Cloneable {

  /**
   * Separador de arquivo utilizado nos caminhos dos arquivos.
   */
  protected static final String PATH_SEPARATOR = "/";

  /** Caminho abstrato para o arquivo. */
  protected String path;

  /** Tipo do arquivo. */
  protected String type;

  /**
   * Protocolo padro.
   */
  public static final URLProtocol DEFAULT_PROTOCOL = URLProtocol.PROJECT;

  /** Host do arquivo. */
  private String host;

  /** Protocolo da URL **/
  private URLProtocol protocol;

  /**
   * Construtor
   *
   * @param path
   */
  public FileURLValue(String path) {
    this(path, ProjectFileType.UNKNOWN);
  }

  /**
   * Construtor.
   *
   * @param path O caminho (No pode ser {@code null}).
   * @param type O tipo ({@code null} para tipo desconhecido).
   */
  public FileURLValue(String path, String type) {
    this(path, type, URLProtocol.PROJECT);
  }

  /**
   * Construtor.
   *
   * @param path O caminho (No pode ser {@code null}).
   * @param type O tipo ({@code null} para tipo desconhecido).
   * @param protocol O protocolo ({@code null} para protocolo padro).
   */
  public FileURLValue(String path, String type, URLProtocol protocol) {
    this(path, type, protocol, null);
  }

  /**
   * Construtor.
   *
   * @param path O caminho (No pode ser {@code null}).
   * @param type O tipo ({@code null} para tipo desconhecido).
   * @param protocol O protocolo ({@code null} para protocolo padro).
   * @param host O host da URL.
   */
  public FileURLValue(String path, String type, URLProtocol protocol,
    String host) {
    setPath(path);
    setType(type);
    setProtocol(protocol);
    setHost(host);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(FileURLValue file) {
    return getPath().compareToIgnoreCase(file.getPath());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    FileURLValue other = (FileURLValue) obj;

    if (path == null) {
      if (other.path != null) {
        return false;
      }
    }
    else if (!path.equals(other.path)) {
      return false;
    }

    if (type == null) {
      if (other.type != null) {
        return false;
      }
    }
    else if (!type.equals(other.type)) {
      return false;
    }

    if (host == null) {
      if (other.host != null) {
        return false;
      }
    }
    else if (!host.equals(other.host)) {
      return false;
    }
    if (protocol != other.protocol) {
      return false;
    }
    return true;
  }

  /**
   * Obtm o nome do arquivo.
   *
   * @return O nome.
   */
  public String getName() {
    String[] components = getPathAsArray();
    return components[components.length - 1];
  }

  /**
   * Obtm o caminho para o arquivo.
   *
   * @return O caminho.
   */
  public String getPath() {
    return path;
  }

  /**
   * Obtm o caminho para o arquivo com o separador especificado.
   *
   * @param fileSeparator o separador a ser usado.
   *
   * @return O caminho.
   */
  public String getPath(char fileSeparator) {
    return path.replace(PATH_SEPARATOR, String.valueOf(fileSeparator));
  }

  /**
   * Obtm um array representando o caminho para o arquivo.
   *
   * @return um array representando o caminho para o arquivo.
   */
  public String[] getPathAsArray() {
    return FileUtils.splitPath(path, PATH_SEPARATOR);
  }

  /**
   * Obtm o tipo do arquivo.
   *
   * @return O tipo.
   */
  public String getType() {
    return type;
  }

  /**
   * Indica se  um diretrio.
   *
   * @return .
   */
  public boolean isDirectory() {
    return ProjectFileType.getFileType(getType()).isDirectory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((host == null) ? 0 : host.hashCode());
    result = prime * result + ((path == null) ? 0 : path.hashCode());
    result = prime * result + ((protocol == null) ? 0 : protocol.hashCode());
    result = prime * result + ((type == null) ? 0 : type.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    if (path.equals(".")) {
      return PATH_SEPARATOR;
    }
    return path;
    //  return PATH_SEPARATOR + path; (estava assim em FileParameterValue) XXX
  }

  /**
   * Indica se  um arquivo regular.
   *
   * @return .
   */
  public boolean isRegularFile() {
    return !isDirectory();
  }

  /**
   * Indica se o tipo  conhecido.
   *
   * @return .
   */
  public boolean isTypeKnown() {
    return getType().equals(ProjectFileType.UNKNOWN);
  }

  /**
   * Permite renomear o arquivo.
   *
   * @param path String que representa o caminho do arquivo.
   */
  public void setPath(final String path) {
    if (path == null) {
      throw new IllegalArgumentException("O parmetro path est nulo.");
    }

    final String auxPath;
    if (path.length() == 0) {
      auxPath = ".";
    }
    else {
      /*
       * Substitui, o separador do windows - \ - pelo separador universal - /.
       * Desta forma, o caminho passa a ser independente do sistema operacional.
       */
      auxPath = path.replaceAll("\\\\", PATH_SEPARATOR);
    }
    this.path = auxPath;
  }

  /**
   * Atribui o tipo a este arquivo.
   *
   * @param newType o tipo ({@code null} para tipo desconhecido).
   */
  private void setType(final String newType) {
    String auxType = newType;
    if (newType == null || newType.trim().isEmpty()) {
      auxType = ProjectFileType.UNKNOWN;
    }
    this.type = auxType;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileURLValue clone() {
    return new FileURLValue(path, type, protocol, host);
  }

  /**
   * Obtm o protocolo da URL.
   *
   * @return o protocolo.
   */
  public URLProtocol getProtocol() {
    return protocol;
  }

  /**
   * Define o protocolo da URL.
   *
   * @param protocol o protocolo.
   */
  private void setProtocol(URLProtocol protocol) {
    if (protocol == null) {
      this.protocol = DEFAULT_PROTOCOL;
    }
    else {
      this.protocol = protocol;
    }
  }

  /**
   * Obtm o host da URL.
   *
   * @return o host
   */
  public String getHost() {
    return host;
  }

  /**
   * Define o host da URL.
   *
   * @param host o host.
   */
  private void setHost(String host) {
    this.host = host;
  }

  /**
   * Verifica se o valor da URL contm um caminho absoluto Windows.
   *
   * @param urlValue o valor textual da URL.
   * @return <code>true</code> se for um caminho absoluto Windows ou
   *         <code>false</code> caso contrrio.
   */
  public static boolean isWindowsAbsolutePath(String urlValue) {
    if (urlValue == null) {
      return false;
    }
    String windowsAbsolutePathPattern = "^\\w:\\\\.*";
    String windowsAbsolutePathWithUnixSeparatorPattern = "^\\w:/.*";
    boolean isWinPath = Pattern.matches(windowsAbsolutePathPattern, urlValue)
      || Pattern.matches(windowsAbsolutePathWithUnixSeparatorPattern, urlValue);
    return isWinPath;
  }

  /**
   * Verifica se o valor da URL contm um caminho absoluto.
   *
   * @param urlValue o valor textual da URL.
   * @return <code>true</code> se for um caminho absoluto Windows ou
   *         <code>false</code> caso contrrio.
   */
  public static boolean isAbsolutePath(String urlValue) {
    if (urlValue == null) {
      return false;
    }
    return isWindowsAbsolutePath(urlValue) || urlValue.startsWith("/");
  }

  /**
   * Converte o valor textual de uma url em um {@link FileURLValue}.
   *
   * @param mode o modo.
   * @param stringValue o valor textual.
   * @return o {@link FileURLValue}.
   */
  static FileURLValue getURLFromString(FileParameterMode mode,
    String stringValue) {
    if (stringValue == null || stringValue.length() == 0) {
      return null;
    }
    String[] fileValue = stringValue.split(":");
    String path = null, type = null, protocol = null, host = null;
    int groupIndex = 0;
    /*
     * Trata o caso especial de caminhos absolutos Windows que tambm usam o
     * caractere ":".
     */
    if (FileURLValue.isWindowsAbsolutePath(stringValue)) {
      /*
       * Concatena o primeiro e segundo grupos capturados. Exemplo: "c" + ":" +
       * "\teste.txt").
       */
      path = fileValue[groupIndex++] + ":" + fileValue[groupIndex++];
    }
    else {
      /*
       * Caso no seja um caminho windows, o primeiro grupo (se existir)
       * representa o caminho do arquivo.
       */
      if (fileValue.length > groupIndex) {
        path = fileValue[groupIndex++];
      }
    }
    // O grupo seguinte (se existir) representa o tipo do arquivo.
    if (fileValue.length > groupIndex) {
      type = fileValue[groupIndex++];
    }
    // O grupo seguinte (se existir) representa o protocolo do arquivo.
    if (fileValue.length > groupIndex) {
      protocol = fileValue[groupIndex++];
    }
    // O grupo seguinte (se existir) representa o host do arquivo.
    if (fileValue.length > groupIndex) {
      host = fileValue[groupIndex++];
    }
    if (type == null || type.trim().isEmpty()) {
      if (mode == FileParameterMode.DIRECTORY) {
        type = ProjectFileType.DIRECTORY_TYPE;
      }
      else {
        type = ProjectFileType.UNKNOWN;
      }
    }
    URLProtocol urlProtocol = getProtocolFromString(protocol);
    FileURLValue value = new FileURLValue(path, type, urlProtocol, host);
    return value;
  }

  /**
   * Converte um {@link FileURLValue} em sua representao textual.
   *
   * @param file o {@link FileURLValue}.
   * @return valor textual.
   */
  static String getStringValue(FileURLValue file) {
    if (file == null) {
      return null;
    }

    StringBuilder valueAsText = new StringBuilder();
    valueAsText.append(file.getPath());
    valueAsText.append(":");
    valueAsText.append(file.getType());
    URLProtocol protocol = file.getProtocol();
    if (protocol != null) {
      valueAsText.append(":");
      valueAsText.append(protocol.getType());
    }
    String host = file.getHost();
    if (host != null) {
      valueAsText.append(":");
      valueAsText.append(host);
    }
    return valueAsText.toString();
  }

  /**
   * Obtm o protocolo representado pela string.
   *
   * @param protocolString a string.
   * @return o protocolo.
   */
  private static URLProtocol getProtocolFromString(String protocolString) {
    URLProtocol urlProtocol = null;
    URLProtocolConverter converter = new URLProtocolConverter();
    if (protocolString != null) {
      try {
        urlProtocol = converter.valueOf(protocolString);
      }
      catch (ParseException e) {
        throw new IllegalArgumentException("Protocolo de arquivo invlido: "
          + protocolString);
      }
    }
    return urlProtocol;
  }
}
