package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import tecgraf.openbus.algorithmservice.v1_0.parameters.FormulaParameterHelper;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.FormulaParameterValidator;

/**
 * <p>
 * Parmetro do Tipo Frmula: este parmetro permite que o usurio escreva
 * frmulas matemticas.
 * </p>
 * 
 * <p>
 * Este tipo de parmetro gera uma linha de comando codificada para que o shell
 * que executa o comando no tenha problemas com caracteres especiais. O usurio
 * pode informar uma expresso como: x + (y - 3). E o parmetro ir gerar uma
 * linha de comando que no gera problemas no shell. Por exemplo, o valor para o
 * comando para {@code x + (y - 3)} 
 * {@code x__PlusSign__OpenRoundBracket__y__MinusSign__3__CloseRoundBracket__}.
 * </p>
 * 
 * <p>
 * O arquivo de configurao desta classe no seu arquivo de properties
 * (FormulaParameter.properties) precisa definir uma tabela com as
 * substituies. A tabela  formada por 2 vetores de strings. O caractere de
 * usurio  o caractere especial que o usurio pode digitar, mas que 
 * substituido ao gerar a linha de comando. O texto de comando  o texto que
 * substitui o caractere de usurio. O vetor com prefixo
 * {@value #USER_CHARACTER_PROPERTY}  o vetor de caracteres de usurios. O
 * vetor com prefixo {@value #CMD_TEXT_PROPERTY}  o vetor de textos de
 * comandos.
 * </p>
 * 
 * <p>
 * Um trecho do arquivo de configurao: <code> 
 * user_text.1 = <(>
 * cmd_text.1 = OpenRoundBracket
 * user_text.2 = <)>
 * cmd_text.2 = CloseRoundBracket
 * user_text.3 = <[>
 * cmd_text.3 = OpenSquareBracket
 * user_text.4 = <]>
 * cmd_text.4 = CloseSquareBracket
 * </code>
 * 
 * Este trecho indica que se o usurio digitar "(", ")", "[", "]" o valor gerado
 * na linha de comando ser, respectivamente, {@code OpenRoundBracket},
 * {@code CloseRoundBracket}, {@code OpenSquareBracket} e
 * {@code CloseSquareBracket}.
 * </p>
 * 
 * <p>
 * Os caracteres "<" e ">" so utilizados para delimitar o caractere que ir ser
 * utilizado na substituio, portanto apenas o que est dentro deles 
 * utilizado na substituio.
 * </p>
 * 
 * <p>
 * Apenas letras, dgitos, underline e os caracteres de usurio que esto no
 * arquivo de configurao podem ser utilizados pelo usurio. O caracter
 * underline  um caractere de usurio e o seu texto de comando 
 * {@code Underline}.
 * </p>
 * 
 * <p>
 * Apenas letras e dgitos podem ser utilizados no texto de comando.
 * </p>
 * 
 * <p>
 * Para cada caracter trocado, o parmetro colocar 2 underlines ("_") antes e
 * um depois para demarcar a regio que foi substituda. Assim, se o usurio
 * digitar "[", a linha referente a ele ser "__OpenSquareBracket__".
 * </p>
 * 
 * @author lmoreira
 */
public final class FormulaParameter extends SimpleParameter<String> {
  /**
   * A propriedade do arq. de resource que aponta para a lista de strings que
   * iro para a linha de comando.
   */
  private static final String CMD_TEXT_PROPERTY = "cmd_text";
  /**
   * A propriedade do arq. de resource que aponta para a lista de strings que
   * que iro para a linha de comando.
   */
  private static final String USER_CHARACTER_PROPERTY = "user_char";

  /**
   * Textos de comando.
   */
  private List<String> commandTexts;
  /**
   * Caracteres de usurio.
   */
  private List<Character> userCharacters;

  /**
   * Cria o parmetro.
   * 
   * @param name Nome do parmetro (No aceita {@code null}).
   * @param label Rtulo do parmetro (No aceita {@code null}).
   * @param description Descrio do parmetro (Aceita {@code null}).
   * @param isOptional Indica se o parmetro  opcional/obrigatrio.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   */
  public FormulaParameter(String name, String label, String description,
    boolean isOptional, boolean isVisible, String commandLinePattern) {
    super(name, label, description, null, isOptional, isVisible,
      commandLinePattern);

    final Class<? extends FormulaParameter> thisClass = this.getClass();
    final String thisClassName = thisClass.getSimpleName();
    final String resName = thisClassName + ".properties";
    final InputStream inStream = thisClass.getResourceAsStream(resName);
    final Properties properties = new Properties();

    try {
      properties.load(inStream);
    }
    catch (IOException e) {
      throw new IllegalStateException(e);
    }

    this.commandTexts = new ArrayList<String>();
    this.userCharacters = new ArrayList<Character>();
    int i = 1;
    while (true) {
      final String cmdPropName = CMD_TEXT_PROPERTY + "." + i;
      final String usrPropName = USER_CHARACTER_PROPERTY + "." + i;

      if (!properties.containsKey(cmdPropName)) {
        break;
      }

      final String commandText = properties.getProperty(cmdPropName);
      final String cmdRegex = "[A-Za-z0-9_]+";
      final Pattern cmdPattern = Pattern.compile(cmdRegex);
      final Matcher cmdMatcher = cmdPattern.matcher(commandText);
      if (!cmdMatcher.matches()) {
        final String err =
          String.format(
            "O texto de comando %s.%d (%s) est no formato errado. "
              + "Expresso regular: %s.\n", CMD_TEXT_PROPERTY, i, commandText,
            cmdRegex);
        throw new IllegalArgumentException(err);
      }

      commandTexts.add(commandText);

      final String userCharacter = properties.getProperty(usrPropName);
      final String usrRegex = "^<(.)>$";
      final Pattern usrPattern = Pattern.compile(usrRegex);
      final Matcher usrMatcher = usrPattern.matcher(userCharacter);
      if (!usrMatcher.matches()) {
        String errorMessage =
          String.format(
            "O caractere de usurio %s.%d (%s) est no formato errado. "
              + "Expresso regular: %s.\n", USER_CHARACTER_PROPERTY, i,
            userCharacter, usrRegex);
        throw new IllegalArgumentException(errorMessage);
      }

      final String ch = usrMatcher.group(1);
      userCharacters.add(ch.charAt(0));
      i = i + 1;
    }
  }

  /**
   * Obtm o valor da linha de comando para o valor corrente.
   * 
   * @return O valor da linha de comando ({@code null} caso no haja um valor).
   */
  public String getCommandValue() {
    String value = getValue();
    return getEncodedValue(value);
  }

  /**
   * Obtm o valor codificado da frmula (sem o uso de caracteres que possam
   * gerar problemas ao serem passados na linha de comando).
   * 
   * @param value o valor original.
   * @return o valor codificado.
   */
  public String getEncodedValue(String value) {
    if (value == null) {
      return null;
    }
    String encodedValue = value.replace("_", "__Underline__");
    for (int i = 0; i < userCharacters.size(); i++) {
      String userText = Character.toString(userCharacters.get(i));
      String commandText = String.format("__%s__", commandTexts.get(i));
      encodedValue = encodedValue.replace(userText, commandText);
    }
    return encodedValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return "FORMULA";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIDLType() {
    return FormulaParameterHelper.id();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) {
    setValue(parameterValue);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCommandValue(CommandLineContext context) {
    return getCommandValue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FormulaParameterValidator createParameterValidator() {
    return new FormulaParameterValidator(isOptional());
  }
}
