package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;

/**
 * <p>
 * Fbrica de parmetros do tipo enumerao: {@link EnumerationParameter} e
 * {@link EnumerationListParameter}. Como esses dois parmetros podem ser
 * criados a partir de uma mesma tag XML, essa fbrica foi criada para
 * direcionar a anlise para o parser apropriado para cada um dos casos.
 * </p>
 *
 * <p>
 * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
 * elemento {@link EnumerationParameter} ou {@link EnumerationListParameter}.
 * </p>
 */
public class EnumerationParameterFactory implements ParameterFactory {

  /**
   * <p>
   * O elemento {@value #ENUMERATION_PARAMETER_ELEMENT}: descreve as
   * propriedades de um parmetro do tipo enumerao:
   * {@link EnumerationParameter} ou {@link EnumerationListParameter}.
   * </p>
   * <p>
   *  filho do elemento {@link ParameterGroup}.
   * </p>
   */
  public static final String ENUMERATION_PARAMETER_ELEMENT = "enumeracao";

  /**
   * <p>
   * O atributo
   * {@value #ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE} do
   * elemento {@link #ENUMERATION_PARAMETER_ELEMENT}. Indica se o parmetro
   * aceita seleo mltipla. Caso positivo, o parmetro deve ser instanciado
   * como {@link EnumerationListParameter}, caso contrrio, ser um
   * {@link EnumerationParameter}.
   * </p>
   * <p>
   *  opcional, o seu valor-padro 
   * {@link #ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE} e 
   * do tipo booleano.
   * </p>
   */
  protected static final String ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE =
    "selecao_multipla";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE} do
   * elemento {@link #ENUMERATION_PARAMETER_ELEMENT}.
   * </p>
   * <p>
   * O seu valor 
   * {@value #ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE =
    false;

  /**
   * {@inheritDoc}
   */
  @Override
  public Parameter<?> createParameter(XmlParser parser,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    boolean isMultipleSelection = parser.extractAttributeValueAsBoolean(
      ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE,
      ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE);

    AbstractEnumerationParameterParser<?> enumerationParser;
    if (isMultipleSelection) {
      enumerationParser = new EnumerationListParameterParser();
    }
    else {
      enumerationParser = new EnumerationParameterParser();
    }
    return enumerationParser.createParameter(parser, commandLinePattern, group,
      configurator);
  }
}
