/*
 * $Id$
 */
package csbase.logic.algorithms.xml.algorithmspack;

import java.io.IOException;
import java.io.Writer;
import java.util.List;

import tecgraf.javautils.xml.ImprovedXMLListIterator;
import tecgraf.javautils.xml.XMLBasicElement;
import tecgraf.javautils.xml.XMLComment;
import tecgraf.javautils.xml.XMLElementInterface;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmsPack;

/**
 * Esta classe implementa o elemento XML raiz do documento que descreve todas os
 * algoritmos do pacote de algoritmos.
 * 
 */
public class XmlAlgorithmsPackElement extends XMLBasicElement {

  /**
   * Constri o elemento xml que descreve todas o pacote de algoritmos.
   */
  public XmlAlgorithmsPackElement() {
    super();
    setTag(XmlAlgorithmsPackUtils.ALGORITHMS_TAG);
  }

  /**
   * Obtm o pacote de algoritmos que representa o contexto da aplicao.
   * 
   * @return o pacote de algoritmos
   */
  public AlgorithmsPack getAlgorithmsPack() {
    return (AlgorithmsPack) getAppContextObject();
  }

  /**
   * Executado quando a tag de fim do elemento XML  encontrada. Deve-se criar o
   * objeto correspondente na camada da aplicao.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#endTag(java.util.List)
   */
  @Override
  public void endTag(List<XMLElementInterface> childrenList) {
    ImprovedXMLListIterator iterator =
      new ImprovedXMLListIterator(childrenList);
    while (iterator.hasNext()) {
      XMLElementInterface nextElem = iterator.next();
      AlgorithmInfo algoInfo =
        ((XmlAlgorithmElement) nextElem).getAlgorithmInfo();
      if (algoInfo != null) {
        getAlgorithmsPack().addAlgorithm(algoInfo);
      }
    }
  }

  /**
   * Escreve todos os elementos que descrevem os algoritmos.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @throws IOException
   */
  private void writeAllAlgorithmElement(Writer writer, String ident)
    throws IOException {
    String newIdent = XMLBasicElement.getNextIdentation(ident);

    for (AlgorithmInfo algoInfo : getAlgorithmsPack().getAlgorithms()) {
      writeAlgorithmElement(writer, newIdent, algoInfo);
    }
  }

  /**
   * Escreve o elemento que descreve um algoritmo.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @param algoInfo algoritmo a ser escrito
   * @throws IOException
   */
  private void writeAlgorithmElement(final Writer writer, final String ident,
    AlgorithmInfo algoInfo) throws IOException {
    XmlAlgorithmElement xmlalgorithm = new XmlAlgorithmElement(algoInfo);
    xmlalgorithm.write(writer, ident);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(Writer writer, String ident) throws IOException {
    writeStartTagln(writer, ident);
    String comment = "Especificando os algoritmos";
    XMLComment.write(writer, ident, comment);
    writeAllAlgorithmElement(writer, ident);
    writeEndTag(writer, ident);
  }
}
