/*
 * SchedulerServiceInterface.java
 * 
 * $Author: tatimf $ $Date: 2016-08-17 17:18:47 -0300 (Wed, 17 Aug 2016) $
 * $Revision: 175500 $
 */
package csbase.remote;

import java.rmi.RemoteException;
import java.util.Map;
import java.util.Set;

import csbase.logic.CommandInfo;
import csbase.logic.CommandSubmission;
import csbase.logic.Priority;
import csbase.logic.algorithms.ExecutionType;

/**
 * Interface do servio de escalonamento de execuo de algoritmos.
 * 
 * @author Bruno Oliveira Silvestre
 * 
 */
public interface SchedulerServiceInterface extends ServiceInterface,
  RemoteObservable {

  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "SchedulerService";

  /**
   * Submete um ou mais comandos para a fila de execuo.
   * 
   * @param commandSubmission informaes sobre a submisso do comando.
   * @return Comandos gerados.
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public Set<CommandInfo> submitCommand(CommandSubmission commandSubmission)
    throws RemoteException;

  /**
   * Submete um ou mais comandos para a fila de execuo cadastrando
   * observadores remotos.
   * 
   * @param submission informaes sobre a submisso do comando.
   * @param observers Observadores Remotos
   * @return Comandos gerados.
   * @throws RemoteException Se houver falha na chamada remota.
   * 
   * @deprecated Utilizar o mtodo {@link #submitCommand(CommandSubmission)}
   *             para submeter comandos. Os observadores no so notificados
   *             sobre mudanas nos comandos submetidos.
   */
  @Deprecated
  public Set<CommandInfo> submitCommand(CommandSubmission submission,
    RemoteObserver... observers) throws RemoteException;

  /**
   * Altera a posio do comando na fila de execuo.
   * 
   * @param commandId Identificador do comando.
   * @param position Nova posio na fila.
   * 
   * @return <code>true</code> se as alterao foram realizadas com sucesso.
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public boolean setPosition(Object commandId, int position)
    throws RemoteException;

  /**
   * Altera a prioridade do comando na fila de espera.
   * 
   * @param commandId Identificador do comando.
   * @param priority Nova prioridade para o comando.
   * 
   * @return <code>true</code> se as alterao foram realizadas com sucesso.
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public boolean setPriority(Object commandId, Priority priority)
    throws RemoteException;

  /**
   * Atualiza a descrio de um comando no agendador.
   * 
   * @param commandId identificador do comando.
   * @param description nova descrio do comando.
   * 
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public void updateCommandDescription(String commandId, String description)
    throws RemoteException;

  /**
   * Remove o comando da fila de espera.
   * 
   * @param commandId Identificador do comando.
   * 
   * @return <code>true</code> caso o comando tenha sido removido.
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public boolean removeCommand(Object commandId) throws RemoteException;

  /**
   * Obtm todos os comandos esperando para execuo. Caso seja o administrador,
   * retorna os comandos de todos os usurios.
   * 
   * @return Os comandos que esto na fila de espera.
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public CommandInfo[] getQueuedCommands() throws RemoteException;

  /**
   * Altera o estado da fila para bloqueada ou desbloqueada. Quando bloqueada,
   * todos os commandos permanecem na fila at que ela seja desbloqueada.
   * 
   * @param blocked Indica o bloqueio (true) ou desbloqueio (false).
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public void setBlocked(boolean blocked) throws RemoteException;

  /**
   * Indica se a fila est bloqueada.
   * 
   * @return true se a fila estiver bloqueada, ou false caso contrrio
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public boolean isBlocked() throws RemoteException;

  /**
   * Obtm um mapa com as estatsticas de submisso de comandos por tipo de
   * comando.
   * 
   * @return mapa (no-modificvel) com as estatsticas de submisso de comandos
   *         por tipo de comando
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public Map<ExecutionType, Integer> getExeTypeStats() throws RemoteException;

  /**
   * Obtm um mapa com as estatsticas de submisso de comandos por algoritmo.
   * 
   * @param flowAlgorithms <code>true</code> para obter estatsticas de fluxos,
   *        <code>false</code> para obter das execues simples
   * 
   * @return mapa (no-modificvel) com as estatsticas de submisso de comandos
   *         por algoritmo
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public Map<String, Integer> getAlgoStats(boolean flowAlgorithms)
    throws RemoteException;

  /**
   * Obtm o nmero de execues de fluxos.
   * 
   * @return nmero de execues de fluxos
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public int getFlowExecutionStats() throws RemoteException;

  /**
   * Obtm o nmero de execues por usurio, independente do tipo.
   * 
   * @return nmero de execues por usurio.
   * @throws RemoteException Se houver falha na chamada remota.
   */
  public Map<String, Integer> getUserStats() throws RemoteException;

}
