package csbase.logic.algorithms.parsers.triggers;

import java.util.HashSet;
import java.util.Set;

import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.conditions.AndOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.Condition;
import csbase.logic.algorithms.parameters.conditions.GenericCondition;
import csbase.logic.algorithms.parameters.conditions.NotOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.OrOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.SimpleCondition;
import csbase.logic.algorithms.parameters.triggers.Trigger;
import csbase.logic.algorithms.parsers.XmlParser;

/**
 * <p>
 * Classe abstrata para simplificar a criao de {@link TriggerFactory fbricas
 * de gatilhos}.
 * </p>
 * 
 * <p>
 * Ela implementa os mtodos requiridos deixando mtodos abstratos nos pontos
 * que so especficos de acordo com o tipo de fbrica.
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class AbstracTriggerFactory implements TriggerFactory {
  /**
   * O nome do elemento que define as propriedades do gatilho.
   */
  private String elementName;

  /**
   * O nome do atributo que indica o {@link Parameter parmetro} manipulado pela
   * {@link Trigger}.
   */
  private String parameterAttributeName;

  /**
   * O nome do elemento que indica o {@link Parameter parmetro} manipulado pela
   * {@link Trigger}.
   */
  private String parameterElementName;

  /**
   * O nome do atributo que indica o nome ou identificador do {@link Parameter
   * parmetro} manipulado pela {@link Trigger}.
   */
  private String parameterNameAttributeName;

  /**
   * Indica se essa fbrica  capaz de criar mltiplas instncias do mesmo
   * {@link Trigger gatilhos}.
   */
  private boolean allowMultiples;

  /**
   * Cria a fbrica.
   * 
   * @param elementName O nome do elemento que define as propriedades do
   *        gatilho.
   * @param parameterElementName O nome do elemento que indica o
   *        {@link Parameter parmetro} manipulado pela {@link Trigger}.
   * @param parameterNameAttributeName O nome do atributo que indica o nome ou
   *        identificador do {@link Parameter parmetro} manipulado pela
   *        {@link Trigger}.
   * @param parameterAttributeName O nome do atributo que indica o
   *        {@link Parameter parmetro} manipulado pela {@link Trigger}.
   * @param allowMultiples Indica se essa fbrica  capaz de criar mltiplas
   *        instncias do mesmo {@link Trigger gatilhos}.
   */
  protected AbstracTriggerFactory(String elementName,
    String parameterElementName, String parameterNameAttributeName,
    String parameterAttributeName, boolean allowMultiples) {
    this.elementName = elementName;
    this.parameterElementName = parameterElementName;
    this.parameterNameAttributeName = parameterNameAttributeName;
    this.parameterAttributeName = parameterAttributeName;
    this.allowMultiples = allowMultiples;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getElementName() {
    return elementName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Set<Trigger<?>> createTriggers(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    Condition condition = loadTriggerCondition(parser, configurator);
    Set<Trigger<?>> triggers = new HashSet<Trigger<?>>();
    Set<Parameter<?>> parameters = loadTriggerParameters(parser, configurator);
    for (Parameter<?> parameter : parameters) {
      triggers.add(doCreateTrigger(parser, condition, parameter, configurator));
    }
    parser.checkAttributes();
    return triggers;
  }

  /**
   * Cria o {@link Trigger gatilho}.
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param condition A condio do {@link Trigger gatilho} (No aceita
   *        {@code null}).
   * @param parameter O parmetro que ser manipula pelo {@link Trigger gatilho}
   *        (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O gatilho.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  protected abstract Trigger<?> doCreateTrigger(XmlParser parser,
    Condition condition, Parameter<?> parameter,
    SimpleAlgorithmConfigurator configurator) throws ParseException;

  /**
   * Cria o {@link Trigger gatilho}.
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * 
   * @return O parmetro ou {@code null} se no houver parmetros.
   * 
   * @throws OperationFailureException Se forem encontrado 2 ou mais parmetros
   *         com o mesmo nome.
   */
  protected abstract Parameter<?> findParameter(
    SimpleAlgorithmConfigurator configurator, String parameterName)
    throws OperationFailureException;

  /**
   * Carrega os parmetros referenciados por um gatilho.
   * 
   * @param parser O analisador (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return Os parmetros.
   * 
   * @throws ParseException Se houver algum erro no XML, inclusive se no houver
   *         indicao de que parmetros utilizar.
   */
  private Set<Parameter<?>> loadTriggerParameters(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String elementName = parser.getElementName();
    Set<Parameter<?>> parameters = new HashSet<Parameter<?>>();
    String parameterName =
      parser.extractAttributeValue(parameterAttributeName, null);
    if (parameterName != null) {
      Parameter<?> parameter;
      try {
        parameter = findParameter(configurator, parameterName);
      }
      catch (OperationFailureException e) {
        throw new ParseException(e,
          "Falha ao tentar localizar o parmetro {0}", parameterName);
      }
      if (parameter == null) {
        throw new ParseException(
          "O parmetro {0} que est referenciado no elemento {1} "
            + "atributo {2} no est definido.", parameterName, elementName,
          parameterAttributeName);
      }
      parameters.add(parameter);
    }
    if (parser.goToFirstChild(parameterElementName)) {
      do {
        parameterName = parser.getAttributeValue(parameterNameAttributeName);
        Parameter<?> parameter;
        try {
          parameter = findParameter(configurator, parameterName);
        }
        catch (OperationFailureException e) {
          throw new ParseException(e,
            "Falha ao tentar localizar o parmetro {0}", parameterName);
        }
        if (parameter == null) {
          throw new ParseException(
            "O parmetro {0} que est referenciado no elemento {1} "
              + "atributo {2} no est definido.", parameterName,
            parameterElementName, parameterAttributeName);
        }
        parameters.add(parameter);
      } while (parser.goToNextSibling(parameterElementName));
      parser.goToParent();
    }
    if (parameters.isEmpty()) {
      throw new ParseException(String.format(
        "Os parmetros associados ao gatilho representado pelo elemento %s "
          + "no foram informados.", elementName));
    }
    if (!allowMultiples && parameters.size() > 1) {
      throw new ParseException(String.format(
        "O gatilho representado pelo elemento %s no pode ser "
          + "utilizado para mais do que 1 parmetro.", elementName));
    }
    return parameters;
  }

  /**
   * <p>
   * Carrega uma condio, que pode ser:
   * <ul>
   * <li>{@link SimpleCondition}</li>
   * <li>{@link GenericCondition}</li>
   * <li>{@link AndOperatorCondition}</li>
   * <li>{@link OrOperatorCondition}</li>
   * <li>{@link NotOperatorCondition}</li>
   * </ul>
   * </p>
   * 
   * <p>
   * O elemento corrente do aponta para o gatilho.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O operador.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Condition loadTriggerCondition(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    Condition condition = null;
    if (parser.goToFirstChild(ConditionParser.CONDITION_ELEMENT)
      || parser.goToFirstChild(ConditionParser.OR_ELEMENT)
      || parser.goToFirstChild(ConditionParser.AND_ELEMENT)
      || parser.goToFirstChild(ConditionParser.NOT_ELEMENT)) {
      ConditionParser conditionParser = new ConditionParser();
      condition = conditionParser.loadCondition(parser, configurator);
      parser.goToParent();
      return condition;
    }
    throw new ParseException("A condio de uma restrio no est definida.");
  }
}
