/*
 * $Id: AdministrationServiceInterface.java 176366 2016-09-29 21:59:06Z isabella $
 */
package csbase.remote;

import java.io.File;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Locale;

import javax.swing.ImageIcon;

import csbase.exception.administration.AdministrationDeleteException;
import csbase.logic.Permission;
import csbase.logic.Platform;
import csbase.logic.PlatformInfo;
import csbase.logic.Role;
import csbase.logic.RoleInfo;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.logic.UserGroupInfo;
import csbase.logic.UserInfo;
import csbase.logic.UserOutline;
import csbase.logic.UserUpdate;

/**
 * Define a interface remota do servio de acesso a usurios, grupos de
 * usurios, perfis e plataformas.
 * 
 * @author Tecgraf/PUC-Rio
 */
public interface AdministrationServiceInterface extends TransactionInterface,
  ServiceInterface, RemoteObservable {

  /**
   * Nome do servio para o {@code ServiceManager}
   */
  public static final String SERVICE_NAME = "AdministrationService";

  /**
   * Obtm um vetor de todos os usurios do sistema. Essas informaes so
   * representadas por objetos da classe {@link csbase.logic.User User}.
   * 
   * @return um vetor com todos os usurios
   * @throws RemoteException em caso de erro.
   */
  public List<User> getAllUsers() throws RemoteException;

  /**
   * Obtm uma lista de informaes sumarizadas necessrias para fazer a
   * atualizao de uma cpia do repositrio de usurios. Essas informaes so
   * representadas por objetos da classe {@link UserUpdate}.
   * 
   * @return um vetor com os resumos de atualizao de todos os usurios
   * @throws RemoteException em caso de erro.
   */
  public List<UserUpdate> getAllUserUpdates() throws RemoteException;

  /**
   * Obtm uma lista de informaes sumarizadas de todos os usurios do sistema.
   * 
   * @return um vetor com informaes de usurios
   * @throws RemoteException em caso de erro.
   */
  public List<UserOutline> getAllUserOutlines() throws RemoteException;

  /**
   * Obtm um usurio que possui uma determinada identificao.
   * 
   * @param id a identificao de um usurio
   * 
   * @return o usurio que possui a identificao requisitada ou null caso esse
   *         usurio no exista
   * @throws RemoteException em caso de erro.
   */
  public User getUser(Object id) throws RemoteException;

  /**
   * Cria um novo usurio no sistema. No podem existir dois usurios com o
   * mesmo login.
   * 
   * @param info os dados do usurio a ser includo
   * 
   * @return o novo usurio
   * @throws RemoteException em caso de erro.
   */
  public User createUser(UserInfo info) throws RemoteException;

  /**
   * Busca a foto de um usurio.
   * 
   * @param id a identificao do usurio a ser modificado
   * @return a imagem
   * @throws RemoteException em caso de erro.
   */
  public ImageIcon getPhoto(Object id) throws RemoteException;

  /**
   * Retorna o arquivo com a foto de um usurio.
   *
   * @param id a identificao do usurio a ser modificado
   * @return o arquivo ou null caso no exista
   * @throws RemoteException em caso de erro.
   */
  public File getPhotoFile(Object id) throws RemoteException;

  /**
   * Modifica um usurio no sistema. No  permitido mudar o login de um
   * usurio.
   * 
   * @param id a identificao do usurio a ser modificado
   * @param info o dados a serem usuados na modificao desse usurio
   * 
   * @return o usurio aps a modificao
   * @throws RemoteException em caso de erro.
   */
  public User modifyUser(Object id, UserInfo info) throws RemoteException;

  /**
   * Remove um usurio do sistema.
   * 
   * @param id o identificador do usurio a ser removido do sistema.
   * @throws RemoteException falha de rmi
   * @throws AdministrationDeleteException se o usurio possuir projetos
   */
  public void deleteUser(Object id) throws RemoteException,
    AdministrationDeleteException;

  /**
   * Obtm uma lista de todas os UserGroups do sistema.
   * 
   * @return um vetor com os UserGroups
   * @throws RemoteException em caso de erro.
   */
  public List<UserGroup> getAllUserGroups() throws RemoteException;

  /**
   * Obtm um UserGroup que possui uma determinada identificao.
   * 
   * @param id a identificao de um UserGroup
   * 
   * @return o UserGroup que possui a identificao requisitada ou null caso ele
   *         no exista
   * @throws RemoteException em caso de erro.
   */
  public UserGroup getUserGroup(Object id) throws RemoteException;

  /**
   * Cria um novo UserGroup no sistema. No podem existir dois UserGroups com o
   * mesmo nome.
   * 
   * @param info os dados do grupo de usurios a ser includo
   * 
   * @return o grupo de usurios criado
   * @throws RemoteException em caso de erro.
   */
  public UserGroup createUserGroup(UserGroupInfo info) throws RemoteException;

  /**
   * Modifica um UserGroup no sistema. No  possvel mudar o nome de um
   * UserGroup se j existir outro com esse mesmo nome.
   * 
   * @param id o identificador do UserGroup a ser modificado
   * @param info os dados da grupo de usurios a ser modificado
   * 
   * @return o grupo de usurios modificado
   * @throws RemoteException em caso de erro.
   */
  public UserGroup modifyUserGroup(Object id, UserGroupInfo info)
    throws RemoteException;

  /**
   * Remove um UserGroup do sistema.
   * 
   * @param id a identificao do grupo de usurios a ser removido do sistema.
   * 
   * @throws AdministrationDeleteException erro levantado quando existe usuario
   *         associado com o UserGroup.
   * @throws RemoteException em caso de erro.
   */
  public void deleteUserGroup(Object id) throws RemoteException,
    AdministrationDeleteException;

  /**
   * Obtm todos os perfis cadastrados no sistema.
   * 
   * @return vetor com os perfis
   * @throws RemoteException em caso de erro.
   */
  public List<Role> getAllRoles() throws RemoteException;

  /**
   * Obtm um perfil que possui uma determinada identificao.
   * 
   * @param id a identificao de um perfil
   * 
   * @return o perfil que possui a identificao requisitada
   * @throws RemoteException em caso de erro.
   */
  public Role getRole(Object id) throws RemoteException;

  /**
   * Cria um novo perfil no sistema.
   * 
   * @param info as informaes do novo perfil.
   * 
   * @return o perfil criado
   * @throws RemoteException em caso de erro.
   */
  public Role createRole(RoleInfo info) throws RemoteException;

  /**
   * Modifica um perfil no sistema.
   * 
   * @param id - identificador do perfil a ser modificado
   * @param info as informaes a serem usadas na modificao do perfil.
   * 
   * @return o perfil modificado
   * @throws RemoteException em caso de erro.
   */
  public Role modifyRole(Object id, RoleInfo info) throws RemoteException;

  /**
   * Remove um perfil do sistema.
   * 
   * @param id o identificador do perfil a ser removido do sistema.
   * 
   * @throws AdministrationDeleteException erro levantado quando existe usuario
   *         associado com o perfil.
   * @throws RemoteException em caso de erro.
   */
  public void deleteRole(Object id) throws RemoteException,
    AdministrationDeleteException;

  /**
   * Obtm os nomes de todas as classes concretas de permisses utilizadas no
   * sistema.
   * 
   * @return vetor com os nomes das classes de permisses
   * @throws RemoteException em caso de erro.
   */
  public List<String> getPermissionClasses() throws RemoteException;

  /**
   * Obtm todas as permisses cadastradas no sistema.
   * 
   * @return vetor com as permisses
   * @throws RemoteException em caso de erro.
   */
  public List<Permission> getAllPermissions() throws RemoteException;

  /**
   * Obtm todas as permisses cadastradas no sistema, com eventuais senhas que
   * elas possuam. Este mtodo  utilizado por servidores locais, logados como
   * admin, para fazer a sincronizao das bases de permisses. Todas as
   * permisses do tipo UserPasswordPermission retornadas por este mtodo tm a
   * senha guardada no campo password.
   * 
   * @return vetor com as permisses
   * @throws RemoteException em caso de erro.
   */
  public List<Permission> getAllPermissionsWithPasswords()
    throws RemoteException;

  /**
   * Obtm uma permisso que possui uma determinada identificao.
   * 
   * @param id a identificao de uma permisso
   * 
   * @return a permisso que possui a identificao requisitada
   * @throws RemoteException em caso de erro.
   */
  public Permission getPermission(Object id) throws RemoteException;

  /**
   * Obtm uma permisso que possui uma determinada identificao. Este mtodo 
   * utilizado por servidores locais, logados como admin, para fazer a
   * sincronizao das bases de permisses. Toda permisso do tipo
   * UserPasswordPermission retornada por este mtodo tem a senha guardada no
   * campo password.
   * 
   * @param id a identificao de uma permisso
   * 
   * @return a permisso que possui a identificao requisitada, com senha
   * @throws RemoteException em caso de erro.
   */
  public Permission getPermissionWithPassword(Object id) throws RemoteException;

  /**
   * Cria uma nova permisso no sistema.
   * 
   * @param permission a nova permisso.
   * 
   * @return a permisso criada
   * @throws RemoteException em caso de erro.
   */
  public Permission createPermission(Permission permission)
    throws RemoteException;

  /**
   * Modifica uma permisso no sistema.
   * 
   * @param id o identificador da permisso a ser alterada.
   * @param permission a nova verso da permisso.
   * 
   * @return a permisso modificada
   * @throws RemoteException em caso de erro.
   */
  public Permission modifyPermission(Object id, Permission permission)
    throws RemoteException;

  /**
   * Remove uma permisso do sistema.
   * 
   * @param id o identificador da permisso a ser removida do sistema.
   * 
   * @throws AdministrationDeleteException erro levantado quando existe usuario
   *         ou um perfil utilizando a permisso.
   * @throws RemoteException em caso de erro.
   */
  public void deletePermission(Object id) throws RemoteException,
    AdministrationDeleteException;

  /**
   * Obtm todas as plataformas cadastradas no sistema.
   * 
   * @return vetor com as plataformas
   * @throws RemoteException em caso de erro.
   */
  public List<Platform> getAllPlatforms() throws RemoteException;

  /**
   * Obtm a plataforma que possui o identificador especificado.
   * 
   * @param id identificador da plataforma
   * 
   * @return plataforma que possui o identificador especificado ou null caso
   *         essa plataforma no exista
   * @throws RemoteException em caso de erro.
   */
  public Platform getPlatform(Object id) throws RemoteException;

  /**
   * Cria uma nova plataforma no sistema.
   * 
   * @param info as informaes da nova plataforma
   * 
   * @return plataforma criada
   * @throws RemoteException em caso de erro.
   */
  public Platform createPlatform(PlatformInfo info) throws RemoteException;

  /**
   * Modifica uma plataforma no sistema.
   * 
   * @param id identificador da plataforma a ser modificada
   * @param info as novas informaes da plataforma
   * 
   * @return plataforma modificada
   * @throws RemoteException em caso de erro.
   */
  public Platform modifyPlatform(Object id, PlatformInfo info)
    throws RemoteException;

  /**
   * Remove uma plataforma
   * 
   * @param id identificador da plataforma a ser removida
   * @throws RemoteException em caso de erro.
   */
  public void deletePlatform(Object id) throws RemoteException;

  /**
   * Atribui um locale a thread corrente. .
   *
   * @param locale o locale para um idioma.
   */
  public void setLocale(Locale locale);

  /**
   * Obtem o locale configurado no sistema
   *
   * @return o locale do sistema
   */
  public Locale getCurrentLocale();
}
