package csbase.logic.algorithms.parsers;

import static csbase.logic.algorithms.parsers.EnumerationItemFactory.DESCRIPTION_ATTRIBUTE_NAME;
import static csbase.logic.algorithms.parsers.EnumerationItemFactory.ELEMENT_NAME;
import static csbase.logic.algorithms.parsers.EnumerationItemFactory.ID_ATTRIBUTE_NAME;
import static csbase.logic.algorithms.parsers.EnumerationItemFactory.LABEL_ATTRIBUTE_NAME;
import static csbase.logic.algorithms.parsers.EnumerationItemFactory.VALUE_ATTRIBUTE_NAME;
import static csbase.logic.algorithms.parsers.EnumerationParameterFactory.ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE;
import static csbase.logic.algorithms.parsers.EnumerationParameterFactory.ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE;


import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.DefaultEnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationColumn;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parsers.elements.attributes.BooleanAttribute;
import csbase.logic.algorithms.parsers.elements.attributes.StringAttribute;
import csbase.logic.algorithms.parsers.elements.ParsedElement;
import csbase.logic.algorithms.parsers.elements.ElementStructure;
import csbase.logic.algorithms.parsers.elements.ParameterStructure;
import csbase.logic.algorithms.parsers.elements.ParsedSimpleParameter;
import csbase.logic.algorithms.parsers.elements.SimpleParameterStructure;

/**
 * Classe abstrata que serve de base para analisadores de parmetros {@code <T>}
 * com valores do tipo enumerao como {@link EnumerationParameter} e
 * {@link EnumerationListParameter}.
 *
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class AbstractEnumerationParameterParser <T extends
  SimpleParameter<?>> extends SimpleParameterParser<T> {

  /**
   * Nome do parmetro.
   */
  public static final String ENUMERATION_PARAMETER_ELEMENT = "enumeracao";

  /**
   * <p>
   * O atributo {@value #ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE} dos elementos:
   * </p>
   * <ul>
   * <li>{@link EnumerationColumn} de um {@link TableParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * </ul>
   * <p>
   * Indica se o itens de enumerao devem ser ordenados.  opcional, o seu
   * valor-padro  {@link #ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE} e o seu
   * tipo  booleano.
   * </p>
   */
  public static final String ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE =
    "ordenar";


  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE} dos elementos:
   * </p>
   * <ul>
   * <li>{@link EnumerationColumn} de um {@link TableParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * </ul>
   * <p>
   * O seu valor  {@value #ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE}.
   * </p>
   */
  public static final boolean ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE =
    true;

  /**
   * A fbrica de itens de enumerao.
   */
  private EnumerationItemFactory enumerationItemFactory;

  /**
   * Cria o parser.
   */
  protected AbstractEnumerationParameterParser() {
    enumerationItemFactory = new EnumerationItemFactory();

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public T createSimpleParameter(XmlParser parser,
    ParsedSimpleParameter definition, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {

    String defaultItemId = definition.getAttributeValue(
      SimpleParameterStructure.PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE);
    boolean mustSort =
      definition.getAttributeValue(ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE);
    List<ParsedElement> children =
      definition.getChildren(ELEMENT_NAME);
    List<EnumerationItem> items;
    if(children != null) {
      items =
      loadEnumerationItems(children, definition.getName());
    } else {
      items = new ArrayList<>();
    }
    return createEnumerationParameter(parser, definition.getName(),
      definition.getLabel(), definition.getDescription(),
      definition.isOptional(), definition.isVisible(),
      definition.getCommandLinePattern(), mustSort, items, defaultItemId);
  }

  /**
   * <p>
   * Carrega os {@link DefaultEnumerationItem}.
   * </p>
   *
   * @param definitions a lista de definies dos itens de enumrao.
   * @param parameterName nome do parmetro.
   * @return Os items (Se no houver itens a lista estar vazia).
   * @throws ParseException Em caso de erro no XML.
   */
  public List<EnumerationItem> loadEnumerationItems(
    List<ParsedElement> definitions,
    String parameterName) throws ParseException {
    List<EnumerationItem> items = new LinkedList<>();
    for (ParsedElement definition : definitions) {
      String id = definition.getAttributeValue(ID_ATTRIBUTE_NAME);
      String label = definition.getAttributeValue(LABEL_ATTRIBUTE_NAME);
      String value = definition.getAttributeValue(VALUE_ATTRIBUTE_NAME);
      String description =
        definition.getAttributeValue(DESCRIPTION_ATTRIBUTE_NAME);
      EnumerationItem enumerationItem =
        new DefaultEnumerationItem(id, label, value, description);
      if (items.contains(enumerationItem)) {
        throw new ParseException("O item {0} est duplicado na enumerao {1}.",
          enumerationItem.getId(), parameterName);
      }
      for (EnumerationItem item : items) {
        if (enumerationItem.getLabel().equals(item.getLabel())) {
          throw new ParseException(
            "O item {0} est duplicado na enumerao {1}.",
            enumerationItem.getLabel(), parameterName);
        }
      }
      items.add(enumerationItem);
    }
    return items;
  }

  /**
   * Cria uma instncia do parmetro de tipo {@code <T>}, a partir dos atributos
   * bsicos de um parmetro do tipo enumerao. As subclasses devem implementar
   * esse mtodo, fazendo a extrao dos demais atributos do parmetro.
   *
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param mustSort Indica se a enumerao deve ser ordenada.
   * @param items Os itens da enumerao, o valor-padro deve ser um dos itens.
   * @param defaultItemId O valor-padro do parmetro.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   * preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   * parmetro com os atributos especificados.
   */
  protected abstract T createEnumerationParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, boolean mustSort, List<EnumerationItem> items,
    String defaultItemId) throws ParseException;

  /**
   * Obtm a fbrica de itens de enumerao.
   *
   * @return .
   */
  protected final EnumerationItemFactory getEnumerationItemFactory() {
    return enumerationItemFactory;
  }

  /**
   * Cria a estrutura base de parmetros de enumerao.
   *
   * @param tag nome do parmetro.
   * @param parameterClass classe do parmetro.
   * @return a estrutura do parmetro.
   */
  protected ParameterStructure<T> getEnumerationStructure(String tag,
    Class<T> parameterClass) {
    ParameterStructure<T> structure =
      new SimpleParameterStructure<>(tag, parameterClass);
    structure.addAttribute(new BooleanAttribute(
      ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_ATTRIBUTE,
      ENUMERATION_PARAMETER_ELEMENT_MULTIPLE_SELECTION_DEFAULT_VALUE));
    structure.addAttribute(
      new BooleanAttribute(ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE,
        ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE));
    structure.addAttribute(new StringAttribute(
      SimpleParameterStructure.PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE,
      null));
    ElementStructure<EnumerationItem> elementItemStructure =
      new ElementStructure<>(ELEMENT_NAME, EnumerationItem.class);
    elementItemStructure.addAttribute(
      new StringAttribute(EnumerationItemFactory.ID_ATTRIBUTE_NAME));
    elementItemStructure.addAttribute(
      new StringAttribute(EnumerationItemFactory.LABEL_ATTRIBUTE_NAME));
    elementItemStructure.addAttribute(
      new StringAttribute(EnumerationItemFactory.VALUE_ATTRIBUTE_NAME));
    elementItemStructure.addAttribute(
      new StringAttribute(EnumerationItemFactory.DESCRIPTION_ATTRIBUTE_NAME,
        null));
    structure.addChild(elementItemStructure);
    return structure;
  }

}
