/**
 * $Id: RemoteObjectObservable.java 149541 2014-02-11 12:51:59Z pietroguedes $
 */

package csbase.remote;

import java.util.Locale;

import csbase.logic.ObserverData;

/**
 * A classe <code>RemoteObjectObservable</code> representa um objeto que pode
 * ser observado remotamente. Essa classe pode ser estendida para representar um
 * objeto remoto que a aplicao precisa observar.
 * 
 * <p>
 * Um objeto pode ser observado por diversos observadores remotos. Um observador
 * remoto deve implementar a interface {@link csbase.remote.RemoteObserver
 * RemoteObserver}. Quando um objeto observado sofre alguma mudana, informada
 * pelo mtodo <code>setChanged</code>, todos os observadores so notificados
 * atravs do seu mtodo <code>update</code>.
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class RemoteObjectObservable implements RemoteObservable,
  RemoteObserverNotifierInterface {
  /**
   * Observadores remotos. Usamos um conjunto pois a ordem no  relevante.
   */
  protected RemoteObserversNotificationManager observerManager;

  /**
   * Inclui um observador remoto no conjunto de observadores. Um observador
   * remoto deve implementar a interface {@link csbase.remote.RemoteObserver
   * RemoteObserver}. Se esse observador j pertencer ao conjunto, a nova
   * referncia substitui a anterior.
   * 
   * @param o um observador remoto que vai ser includo
   * @param arg argumento do observador
   */
  @Override
  public void addObserver(RemoteObserver o, Object arg) {
    observerManager.addObserver(new ObserverData(o, arg));
  }

  /**
   * Remove um observador remoto do conjunto de observadores desse objeto.
   * 
   * @param o o obsevador remoto a ser removido
   * @param arg o argumento do observador
   * @return <code>true</code> se o observador foi removido
   */
  @Override
  public boolean deleteObserver(RemoteObserver o, Object arg) {
    return observerManager.deleteObserver(new ObserverData(o, arg));
  }

  /**
   * Remove um observador remoto do conjunto de observadores desse objeto.
   * 
   * @param od dados do observador remoto (observador + argumento)
   * @return <code>true</code> se o observador foi removido
   */
  public boolean deleteObserver(ObserverData od) {
    return observerManager.deleteObserver(od);
  }

  /**
   * Envia uma notificao para todos os observadores registrados quando o
   * objeto tiver alguma mudana indicada pelo mtodo <code>hasChanged</code>.
   * Aps a notificao ser enviada, invoca o mtodo <code>clearChanged</code>
   * para indicar que a mudana j foi notificada.
   * 
   * <p>
   * Cada observador  notificado atravs da chamada ao mtodo
   * <code>update</code> que recebe como parmetro um argumento para indicar
   * qual a mudana ocorrida.
   * </p>
   * 
   * @param event um argumento qualquer para a notificao de mudana
   * 
   * @see csbase.remote.RemoteObserver#update(RemoteEvent)
   */
  @Override
  public synchronized void notifyObservers(final RemoteEvent event) {
    new Thread(new Runnable() {
      @Override
      public void run() {
        observerManager.notifyObservers(event);
      }
    }, "RemoteObjectObservable.notify").start();
  }

  /**
   * Mtodo para indicar se o observador deve ser notificado.
   * 
   * @param arg Argumento do Observador
   * @param event Evento gerado para notificar o observador
   * 
   * @return True quando o observador deve ser notificado. False caso contrrio
   */
  protected abstract boolean has2Update(Object arg, Object event);

  /**
   * Remove todos os observadores registrados nesse objeto.
   */
  @Override
  public void deleteObservers() {
    observerManager.deleteObservers();
  }

  /**
   * Retorna o nmero de observadores registrados nesse objeto.
   * 
   * @return o nmero de observadores
   */
  @Override
  public int countObservers() {
    return observerManager.numObservers();
  }

  /**
   * Constri um objeto observvel
   * 
   * @param locale Locale.
   */
  public RemoteObjectObservable(Locale locale) {
    observerManager = new RemoteObserversNotificationManager(this, locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean has2update(ObserverData observerData, Object event) {
    return has2Update(observerData.arg, event);
  }
}
