package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Collections;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import csbase.exception.algorithms.ExpressionFunctionExecutionException;
import csbase.exception.algorithms.ExpressionFunctionNotFoundException;

/**
 * Classe de execuo de expresses
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ExpressionFunctionExecutor {
  /**
   * Instncia
   */
  private static ExpressionFunctionExecutor instance;

  /**
   * Conjunto de funes.
   */
  private Set<ExpressionFunction> expressionFunctions;

  /**
   * Construtor
   */
  private ExpressionFunctionExecutor() {
    this.expressionFunctions = new HashSet<ExpressionFunction>();
    this.createExpressionFunctions();
  }

  /**
   * Consulta do singleton
   * 
   * @return instncia
   */
  public static ExpressionFunctionExecutor getInstance() {
    if (instance == null) {
      instance = new ExpressionFunctionExecutor();
    }
    return instance;
  }

  /**
   * Cria e carrega instncias de ExpressionFunction a partir de configuraes. <br>
   * As configuraes devem estar de acordo com o formato abaixo:
   * <ul>
   * <li>Para criar instncias de ReflectionExpressionFunction, as propriedades
   * devem estar de acordo com o formato:<br>
   * class.[0..n] = &lt;nome da classe cujos mtodos estticos, pblicos e no
   * abstratos devem ser encapsulados por uma ReflectionExpressionFunction&gt;<br>
   * Ex.:<br>
   * class.0 = {@link Math}</li>
   * <li>Para criar instncias de classes que estendem ExpressionFuncion, as
   * propriedades devem estar no formato:<br>
   * function.[0..n] = &lt;nome da classe que estende ExpressionFunction&gt;<br>
   * Ex.:<br>
   * function.0 = {@link CheckListExpressionFunction}</li>
   * </ul>
   * 
   */
  @SuppressWarnings("unchecked")
  private void createExpressionFunctions() {
    final String thisClassName = this.getClass().getSimpleName();
    final String resName = thisClassName + ".properties";
    final InputStream inStream = this.getClass().getResourceAsStream(resName);
    final Properties properties = new Properties();

    try {
      properties.load(inStream);
    }
    catch (IOException ioe) {
      throw new IllegalStateException(ioe);
    }

    int ix = 0;
    while (true) {
      final String classPropName = "class." + ix;
      final String className = properties.getProperty(classPropName);
      if (className == null) {
        break;
      }
      final Class<?> functionsClass;
      try {
        functionsClass = Class.forName(className);
      }
      catch (ClassNotFoundException cnfe) {
        throw new IllegalStateException(cnfe);
      }
      if (functionsClass == null) {
        throw new IllegalStateException("Null functionClass! (class.ix)");
      }
      createReflexiveExpressionFunctions(functionsClass);
      ix++;
    }

    ix = 0;
    while (true) {
      final String funcPropName = "function." + ix;
      final String className = properties.getProperty(funcPropName);
      if (className == null) {
        break;
      }

      final Class<ExpressionFunction> functionClass;
      try {
        functionClass = (Class<ExpressionFunction>) Class.forName(className);
      }
      catch (ClassNotFoundException cnfe) {
        throw new IllegalStateException(cnfe);
      }
      if (functionClass == null) {
        throw new IllegalStateException("Null functionClass! (function.ix)");
      }
      try {
        Constructor<ExpressionFunction> constructor =
          functionClass.getConstructor(new Class[] {});
        ExpressionFunction function = constructor.newInstance(new Object[] {});
        addExpressionFunction(function);
      }
      catch (Exception e) {
        throw new IllegalStateException(e);
      }
      ix++;
    }
  }

  /**
   * Adiciona funes da classe
   * 
   * @param clazz a classe.
   */
  public void createReflexiveExpressionFunctions(Class<?> clazz) {
    for (Method method : clazz.getMethods()) {
      if (Modifier.isPublic(method.getModifiers())
        && Modifier.isStatic(method.getModifiers())
        && !Modifier.isAbstract(method.getModifiers())) {
        addExpressionFunction(new ReflectionExpressionFunction(method));
      }
    }
  }

  /**
   * Adiciona a expresso ao executor.
   * 
   * @param expressionFunction funo
   * @return indicativo
   */
  public boolean addExpressionFunction(ExpressionFunction expressionFunction) {
    if (expressionFunction == null) {
      throw new IllegalArgumentException(
        "O parmetro expressionFunction est nulo.");
    }
    return this.expressionFunctions.add(expressionFunction);
  }

  /**
   * Adiciona lista de expresses ao executor.
   * 
   * @param expFunctions a lista
   * @return indicativo
   */
  public boolean addAllExpressionFunction(Set<ExpressionFunction> expFunctions) {
    if (expFunctions == null) {
      throw new IllegalArgumentException(
        "O parmetro expressionFunctions est nulo.");
    }
    return this.expressionFunctions.addAll(expFunctions);
  }

  /**
   * Executa expresso.
   * 
   * @param functionName nome da funo
   * @param configurator configurador
   * @param parameters par metors
   * @return o objeto
   * @throws ExpressionFunctionExecutionException em caso de falha de execuo.
   * @throws ExpressionFunctionNotFoundException em caso de expresso
   *         inexistente.
   */
  public Object execute(String functionName,
    SimpleAlgorithmConfigurator configurator, Object... parameters)
    throws ExpressionFunctionExecutionException,
    ExpressionFunctionNotFoundException {
    if (functionName == null) {
      throw new IllegalArgumentException(
        "O parmetro functionName no foi encontrado.");
    }
    if (configurator == null) {
      throw new IllegalArgumentException("O parmetro configurator est nulo.");
    }
    if (parameters == null) {
      throw new IllegalArgumentException(
        "O parmetro parameters no foi encontrado.");
    }
    for (ExpressionFunction expressionFunction : this.expressionFunctions) {
      if (expressionFunction.getName().equals(functionName)
        && expressionFunction.acceptParameters(parameters)) {
        return expressionFunction.execute(configurator, parameters);
      }
    }
    Class<?>[] parameterTypes = new Class[parameters.length];
    for (int i = 0; i < parameters.length; i++) {
      parameterTypes[i] = parameters[i].getClass();
    }
    throw new ExpressionFunctionNotFoundException(functionName, parameterTypes);
  }

  /**
   * Retorna uma viso imutvel da lista das possveis funes a serem
   * executadas.
   * 
   * @return uma viso imutvel da lista das possveis funes a serem
   *         executadas.
   */
  public Set<ExpressionFunction> getExpressionFunctions() {
    return Collections.unmodifiableSet(this.expressionFunctions);
  }

  /**
   * Apaga todas as funes da lista.
   */
  public void clear() {
    this.expressionFunctions.clear();
  }
}