package csbase.logic.algorithms.parsers;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parsers.elements.IElementStructure;
import csbase.logic.algorithms.parsers.elements.ParameterStructure;
import csbase.logic.algorithms.parsers.elements.ParsedElement;
import csbase.logic.algorithms.parsers.elements.ParsedSimpleParameter;
import csbase.logic.algorithms.parsers.elements.attributes.IElementAttribute;

/**
 * Classe abstrata que serve de base para parsers de parmetros do tipo
 * {@link SimpleParameterParser}.
 *
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class SimpleParameterParser <T extends SimpleParameter<?>>
  implements ParameterFactory {

  /**
   * Cria uma instncia do parmetro de tipo {@code <T>}, a partir dos atributos
   * bsicos do parmetro. As subclasses devem implementar esse mtodo, fazendo
   * a extrao dos demais atributos do parmetro.
   *
   * @param parser Parser xml do configurador.
   * @param definition definio do parmetro.
   * @param group grupo do parmetro.
   * @param configurator Configurador de algoritmo.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   * preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   * parmetro com os atributos especificados.
   */
  public abstract T createSimpleParameter(XmlParser parser,
    ParsedSimpleParameter definition, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException;

  /**
   * {@inheritDoc}
   */
  @Override
  public T createParameter(final XmlParser parser,
    final String defaultCommandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {

    Map<IElementAttribute<?>, Object> attributeValues =
      parser.extractAttributes(getParameterStructure());

    Map<IElementStructure<?>, List<ParsedElement>> children =
      parser.extractChildren(getParameterStructure().getChildElements());

    ParsedSimpleParameter definition =
      new ParsedSimpleParameter(attributeValues, children,
        defaultCommandLinePattern);

    final T parameter =
      createSimpleParameter(parser, definition, group, configurator);

    parameter.setIgnoreIfDisabled(definition.isIgnoreIfDisabled());
    parameter.setIgnoreIfInvisible(definition.isIgnoreIfInvisible());
    return parameter;
  }



  /**
   * Obtm a estrutura do parmetro.
   *
   * @return a estrutura.
   */
  public abstract ParameterStructure<T> getParameterStructure();

  /**
   * {@inheritDoc}
   */
  @Override
  public final List<ParameterStructure<?>> getParameterStructures() {
    return Collections.singletonList(getParameterStructure());
  }
}
