/*
 * $Id$
 */

package csbase.logic.algorithms;

import java.io.Serializable;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

/**
 * Representa o identificador de uma verso de algoritmo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class AlgorithmVersionId implements Serializable,
  Comparable<AlgorithmVersionId>, Cloneable {

  /**
   * Separador dos nmeros da verso
   */
  private static final String SEPARATOR = ".";

  /**
   * Major
   */
  final private int major;

  /**
   * Minor
   */
  final private int minor;

  /**
   * Patch
   */
  final private int patch;

  /**
   * Uma string que respresenta a verso.
   */
  final private String stringValue;

  /**
   * Cria um identificador de verso.
   * 
   * @param major maior
   * @param minor menor
   * @param patch nmero de patch
   */
  public AlgorithmVersionId(final int major, final int minor, final int patch) {
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    final String sep = SEPARATOR;
    final StringBuffer buf = new StringBuffer(String.valueOf(major));
    buf.append(sep);
    buf.append(String.valueOf(minor));
    buf.append(sep);
    buf.append(String.valueOf(patch));
    this.stringValue = buf.toString();
  }

  /**
   * Cria um identificador de verso a partir de um texto.
   * 
   * @param versionText A verso (No aceita {@code null}).
   * 
   * @return A verso ou {@code null} se o texto no representar um
   *         identificador de verso.
   */
  public static AlgorithmVersionId create(final String versionText) {
    if (versionText == null) {
      final String msg = "O parmetro versionText est nulo.";
      throw new IllegalArgumentException(msg);
    }
    final StringTokenizer tokenizer =
      new StringTokenizer(versionText, SEPARATOR);
    try {
      final String majorText = tokenizer.nextToken();
      final String minorText = tokenizer.nextToken();
      final String patchText = tokenizer.nextToken();
      final int major = Integer.parseInt(majorText);
      final int minor = Integer.parseInt(minorText);
      final int patch = Integer.parseInt(patchText);
      return new AlgorithmVersionId(major, minor, patch);
    }
    catch (final NoSuchElementException e) {
      return null;
    }
    catch (final NumberFormatException e) {
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AlgorithmVersionId clone() {
    try {
      return (AlgorithmVersionId) super.clone();
    }
    catch (final CloneNotSupportedException e) {
      final IllegalStateException ise =
        new IllegalStateException("A verso deveria ser clonvel.");
      ise.initCause(e);
      throw ise;
    }
  }

  /**
   * {@inheritDoc}
   */
  public int compareTo(final AlgorithmVersionId other) {
    if (this.major < other.major) {
      return -1;
    }
    if (this.major > other.major) {
      return 1;
    }
    if (this.minor < other.minor) {
      return -1;
    }
    if (this.minor > other.minor) {
      return 1;
    }
    if (this.patch < other.patch) {
      return -1;
    }
    if (this.patch > other.patch) {
      return 1;
    }
    return 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object o) {
    if (o == null) {
      return false;
    }
    if (!getClass().equals(o.getClass())) {
      return false;
    }
    final AlgorithmVersionId other = (AlgorithmVersionId) o;
    return ((this.major == other.major) && (this.minor == other.minor) && (this.patch == other.patch));
  }

  /**
   * Obtm o major.
   * 
   * @return O major.
   */
  public int getMajor() {
    return this.major;
  }

  /**
   * Obtm o minor.
   * 
   * @return O minor.
   */
  public int getMinor() {
    return this.minor;
  }

  /**
   * Obtm o patch.
   * 
   * @return O patch.
   */
  public int getPatch() {
    return this.patch;
  }

  /**
   * Obtm os nmeros da verso.
   * 
   * @return um array com os trs nmeros (major, minor, patch)
   */
  public int[] getNumbers() {
    return new int[] { this.major, this.minor, this.patch };
  }

  /**
   * {@inheritDoc}
   * 
   * Retorna um hash code para o identificador. O hash code retornado  obtido
   * da representao do identificador como String, para permitir que consultas
   * a Hashtables em localizaes diferentes (podem usar objetos serializados)
   * possam ser feitas.
   */
  @Override
  public int hashCode() {
    return this.stringValue.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.stringValue;
  }
}
