package csbase.logic.algorithms.parameters;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import csbase.logic.algorithms.AlgorithmVersionId;

/**
 * Carregador de valores de um grupo de parmetros.
 */
public class ParameterLoader implements Serializable {

  /** Conjunto de nomes dos parmetros de entrada. */
  private Set<String> inputParameterNames;

  /** O nome do parmetro que mostrar a sada do carregador. */
  private String outputParameterName;

  /** O nome do algoritmo que carregar os valores. */
  private String algorithmName;

  /** Identificador da verso do algoritmo que carregar os valores. */
  private AlgorithmVersionId algorithmVersionId;

  /**
   * Indica que durante o processo de validao, deve ser verificado se os dados
   * carregados foram originados dos {@link #inputParameterNames parmetros de
   * entrada}.
   */
  private boolean validateInput;

  /**
   * Construtor.
   * 
   * @param algorithmName o nome do algoritmo que carregar os valores.
   * @param algorithmVersionId verso do algoritmo que carregar os valores.
   * @param outputParameterName parmetro que mostrar a sada do algoritmo.
   * @param validateInput Indica que durante o processo de validao, deve ser
   *        verificado se os dados carregados foram originados dos parmetros de
   *        entrada.
   * @see #addInputParameterName(String)
   */
  public ParameterLoader(String algorithmName,
    AlgorithmVersionId algorithmVersionId, String outputParameterName,
    boolean validateInput) {
    this.inputParameterNames = new HashSet<String>();
    this.outputParameterName = outputParameterName;
    this.algorithmName = algorithmName;
    this.algorithmVersionId = algorithmVersionId;
    this.validateInput = validateInput;
  }

  /**
   * Adiciona um novo parmetro de entrada ao carregador.
   * 
   * @param inputParameterName o nome do parmetro.
   * @return verdadeiro, se bem sucedido ou falso, caso contrrio.
   */
  public boolean addInputParameterName(String inputParameterName) {
    return this.inputParameterNames.add(inputParameterName);
  }

  /**
   * Obtm o conjunto de nomes dos parmetros de entrada do carregador.
   * 
   * @return input parameter names
   */
  public Set<String> getInputParameterNames() {
    return Collections.unmodifiableSet(inputParameterNames);
  }

  /**
   * Obtm o nome do parmetro de sada do carregador.
   * 
   * @return output o nome do parmetro de sada.
   */
  public String getOutputParameterName() {
    return outputParameterName;
  }

  /**
   * Obtm o nome do algoritmo que carregar os valores.
   * 
   * @return o nome do algoritmo.
   */
  public String getAlgorithmName() {
    return algorithmName;
  }

  /**
   * Obtm o identificador do algoritmo que carregar os valores.
   * 
   * @return o identificador do algoritmo.
   */
  public AlgorithmVersionId getAlgorithmVersionId() {
    return algorithmVersionId;
  }

  /**
   * <p>
   * Indica se a validao dos parmetros de entrada est ativa.
   * </p>
   * <p>
   * A validao dos parmetros de entrada ocorre durante o processo de
   * validao da viso do carregador de parmetros. Ela verifica se os arquivo
   * carregados originaram dos parmetros de entrada.
   * </p>
   * 
   * @return <tt>true</tt> se a validao dos parmetros de entrada est ativa.
   */
  public boolean isInputValidationEnabled() {
    return validateInput;
  }
}
