package csbase.logic.algorithms.parameters.triggers;

import java.text.MessageFormat;

import csbase.logic.algorithms.parameters.DoubleListParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.conditions.Condition;

/**
 * Gatilho Trocar Mnimo para Parmetros com Nmeros Reais: Modifica o valor
 * mnimo para parmetro do tipo {@link DoubleParameter Real},
 * {@link DoubleListParameter Lista de Reais} quando a {@link Condition
 * condio} for aceita.
 * 
 * @author lmoreira
 */
public final class ChangeMinimumForDoublesTrigger extends
  Trigger<SimpleParameter<?>> {
  /**
   * Indica se o valor mnimo deve ser considerado vlido.
   */
  private boolean isMinimumIncluded;

  /**
   * Mnimo ou {@code null} se ele no existir.
   */
  private Double minimum;

  /**
   * Cria o gatilho.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param condition A condio (No aceita {@code null}).
   * @param minimum O mnimo (Aceita {@code null}).
   * @param isMinimumIncluded Indica se o valor mnimo deve ser considerado
   *        vlido.
   */
  public ChangeMinimumForDoublesTrigger(DoubleParameter parameter,
    Condition condition, Double minimum, boolean isMinimumIncluded) {
    super(parameter, condition);
    this.isMinimumIncluded = isMinimumIncluded;
    this.minimum = minimum;
  }

  /**
   * Cria o gatilho.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param condition A condio (No aceita {@code null}).
   * @param minimum O mnimo (Aceita {@code null}).
   * @param isMinimumIncluded Indica se o valor mnimo deve ser considerado
   *        vlido.
   */
  public ChangeMinimumForDoublesTrigger(DoubleListParameter parameter,
    Condition condition, Double minimum, boolean isMinimumIncluded) {
    super(parameter, condition);
    this.isMinimumIncluded = isMinimumIncluded;
    this.minimum = minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (!super.equals(obj)) {
      return false;
    }
    ChangeMinimumForDoublesTrigger trigger =
      (ChangeMinimumForDoublesTrigger) obj;
    if (this.isMinimumIncluded != trigger.isMinimumIncluded) {
      return false;
    }
    if (this.minimum == null && trigger.minimum == null) {
      return true;
    }
    return (this.minimum != null && this.minimum.equals(trigger.minimum));
  }

  /**
   * Obtm o mnimo.
   * 
   * @return O mnimo ou {@code null} se ele no existir.
   */
  public Double getMinimum() {
    return this.minimum;
  }

  /**
   * Indica se o valor mnimo deve ser considerado um dos valores vlidos.
   * 
   * @return .
   */
  public boolean isMinimumIncluded() {
    return this.isMinimumIncluded;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void doAction() {
    SimpleParameter<?> parameter = getParameter();
    if (parameter instanceof DoubleListParameter) {
      DoubleListParameter doubleListParameter = (DoubleListParameter) parameter;
      doubleListParameter.setMinimum(getMinimum(), isMinimumIncluded());
    }
    else if (parameter instanceof DoubleParameter) {
      DoubleParameter doubleParameter = (DoubleParameter) parameter;
      doubleParameter.setMinimum(getMinimum(), isMinimumIncluded());
    }
    else {
      throw new IllegalStateException(MessageFormat.format(
        "O parmetro {0} no  de um tipo vlido.\n"
          + "Tipos permitidos:\n{1};\n{2}.", new Object[] {
            parameter.getLabel(), "real", "lista_de_reais" }));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void undoAction() {
    // @todo Auto-generated method stub
  }
}
