/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters.validators;

import java.rmi.RemoteException;

import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * Validador do {@link csbase.logic.algorithms.parameters.IntegerParameter
 * Parmetro do Tipo Inteiro}.
 *
 * @author lmoreira
 */
public final class IntegerParameterValidator extends
  SimpleParameterValidator<Integer> {
  /**
   * O valor mximo considerado vlido.
   */
  private Integer maximum;
  /**
   * O valor mnimo considerado vlido.
   */
  private Integer minimum;

  /**
   * Cria um validador.
   *
   * @param isOptional Indica se  o valor  opcional/obrigatrio.
   * @param maximum O mximo (Aceita {@code null}; se o mnimo existir ele tem
   *        que ser maior ou igual a ele).
   * @param minimum O mnimo (Aceita {@code null}; se o mximo existir ele tem
   *        que ser menor ou igual a ele).
   */
  public IntegerParameterValidator(boolean isOptional, Integer maximum,
    Integer minimum) {
    super(isOptional);
    setMaximum(maximum);
    setMinimum(minimum);
  }

  /**
   * Obtm o valor mximo.
   *
   * @return O valor mximo ou {@code null} se ele no existir.
   */
  public Integer getMaximum() {
    return this.maximum;
  }

  /**
   * Obtm o valor mnimo.
   *
   * @return O valor mnimo ou {@code null} se ele no existir.
   */
  public Integer getMinimum() {
    return this.minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object convert(Object value) throws Exception {
    if (Integer.class.isInstance(value)) {
      return value;
    }
    if (Long.class.isInstance(value)) {
      long l = ((Long) value).longValue();
      if (l > Integer.MAX_VALUE || l < Integer.MIN_VALUE) {
        throw new Exception("Value out of range.");
      }
    }
    if (Double.class.isInstance(value)) {
      throw new Exception("Double/float values are not permitted for integer.");
    }
    final String className = value.getClass().getName();
    throw new Exception("Bad conversion from " + className + ".");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validateValue(SimpleParameter<?> parameter, Integer value,
    ValidationContext context) throws RemoteException {
    Validation validationResult = super.validateValue(parameter, value,
      context);
    if (!validationResult.isWellSucceeded()) {
      return validationResult;
    }
    if (parameter.isEnabled() && parameter.isEnabled()) {
      if (value != null) {
        if (value.equals(IntegerParameter.ERROR_VALUE)) {
          return new ValidationError(new LocalizedMessage(
            IntegerParameterValidator.class, "error_value", new Object[] {
                parameter.getLabel() }));
        }
        if (this.minimum != null) {
          if (value.compareTo(this.minimum) < 0) {
            LocalizedMessage message = new LocalizedMessage(
              IntegerParameterValidator.class, "bellow_minimum", new Object[] {
                  parameter.getLabel(), value, this.minimum });
            return new ValidationError(message);
          }
        }
        if (this.maximum != null) {
          if (value.compareTo(this.maximum) > 0) {
            LocalizedMessage message = new LocalizedMessage(
              IntegerParameterValidator.class, "above_maximum", new Object[] {
                  parameter.getLabel(), value, this.maximum });
            return new ValidationError(message);
          }
        }
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Atribui o mximo a este parmetro.
   *
   * @param maximum O mximo (Aceita {@code null}; se o mnimo existir ele tem
   *        que ser maior ou igual a ele).
   */
  public void setMaximum(Integer maximum) {
    if (this.minimum != null) {
      if (maximum != null && maximum.compareTo(this.minimum) < 0) {
        throw new IllegalArgumentException(
          "O valor mximo no pode ser menor do que o valor mnimo.");
      }
    }
    this.maximum = maximum;
  }

  /**
   * Atribui o mnimo a este parmetro.
   *
   * @param minimum O mnimo (Aceita {@code null}; se o mximo existir ele tem
   *        que ser menor ou igual a ele).
   */
  public void setMinimum(Integer minimum) {
    if (this.maximum != null) {
      if (minimum != null && minimum.compareTo(this.maximum) > 0) {
        throw new IllegalArgumentException(
          "O valor mnimo no pode ser maior do que o valor mximo.");
      }
    }
    this.minimum = minimum;
  }
}
