package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.ParameterLoader;

/**
 * <p>
 * Analisador de {@link ParameterLoader}.
 * </p>
 * 
 * <p>
 * Este parser l os atributos que representam um carregador de parmetros. O
 * elemento corrente do {@link XmlParser analisador de XML} precisa ser um
 * elemento {@link #PARAMETER_LOADER_ELEMENT}.
 * </p>
 */
public class ParameterLoaderParser {

  /**
   * <p>
   * O elemento {@value #PARAMETER_LOADER_ELEMENT}: descreve as propriedades de
   * um {@link ParameterLoader carregador de parmetros}.
   * </p>
   * <p>
   *  elemento-filho do elemento {@link ParameterGroup}.
   * </p>
   */
  static final String PARAMETER_LOADER_ELEMENT = "carregar_parametros";

  /**
   * <p>
   * O elemento {@value #INPUT_PARAMETER_ELEMENT}: informa um parmetro de
   * entrada para o {@link ParameterLoader Carregador de Parmetros}.
   * </p>
   * <p>
   *  filho do elemento {@link #PARAMETER_LOADER_ELEMENT}.
   * </p>
   */
  static final String INPUT_PARAMETER_ELEMENT = "parametro_de_entrada";

  /**
   * O atributo {@value #PARAMETER_LOADER_ELEMENT_ALGORITHM_ATTRIBUTE} do
   * elemento {@link #PARAMETER_LOADER_ELEMENT}. Indica o nome do algoritmo
   * utilizado pelo {@link ParameterLoader carregador de parmetros}. 
   * obrigatrio e o seu tipo  string.
   */
  private static final String PARAMETER_LOADER_ELEMENT_ALGORITHM_ATTRIBUTE =
    "algoritmo";

  /**
   * O atributo {@value #PARAMETER_LOADER_ELEMENT_OUTPUT_ATTRIBUTE} do elemento
   * {@link #PARAMETER_LOADER_ELEMENT}. Indica o nome do parmetro que  a sada
   * do algoritmo utilizado pelo {@link ParameterLoader carregador de
   * parmetros}.  obrigatrio e o seu tipo  string.
   */
  private static final String PARAMETER_LOADER_ELEMENT_OUTPUT_ATTRIBUTE =
    "saida";

  /**
   * O atributo {@value #PARAMETER_LOADER_ELEMENT_VERSION_ATTRIBUTE} do elemento
   * {@link #PARAMETER_LOADER_ELEMENT}. Indica o identificador da verso do
   * algoritmo utilizado pelo {@link ParameterLoader carregador de parmetros}.
   *  obrigatrio e o seu tipo  string.
   */
  private static final String PARAMETER_LOADER_ELEMENT_VERSION_ATTRIBUTE =
    "versao";

  /**
   * O atributo {@value #PARAMETER_LOADER_ELEMENT_VALIDATION_ATTRIBUTE} do
   * elemento {@link #PARAMETER_LOADER_ELEMENT} indica se durante o processo de
   * validao do {@link ParameterLoader carregador de parmetros} deve-se
   * verificar se os dados carregados originaram da entrada escolhida.  do tipo
   * boolean,  opcional e seu padro  sim, validar se os dados foram
   * carregados da entrada selecionada.
   */
  private static final String PARAMETER_LOADER_ELEMENT_VALIDATION_ATTRIBUTE =
    "validar_entrada";

  /**
   * O atributo {@value #INPUT_PARAMETER_ELEMENT_NAME_ATTRIBUTE} do elemento
   * {@link #INPUT_PARAMETER_ELEMENT}: indica o nome do parmetro de entrada, 
   * obrigatrio e  do tipo string.
   */
  private static final String INPUT_PARAMETER_ELEMENT_NAME_ATTRIBUTE = "nome";

  /**
   * <p>
   * L os parmetros do carregador de parmetros do algoritmo preenchedor.
   * </p>
   * <p>
   * Um parmetro vlido deve estar no formato: {@code
   * <carregar_parametros algoritmo="algoritmo_preenchedor" versao="1.0.0"
   *  saida="ARQ_SAI">
   *   <parametro_de_entrada nome="PARAM_ENTRADA"/>
   * </carregar_parametros>
   * }
   * <p>
   * No verificamos se o algoritmo preenchedor  vlidos e nem se ele existe.
   * </p>
   * <p>
   * O parametro_de_entrada  opcional e por enquanto, s aceitamos um. Esse
   * parmetro deve estar definido no algoritmo principal.
   * </p>
   * 
   * @param parser analisador. Deve estar apontando para o elemento
   *        {@link #PARAMETER_LOADER_ELEMENT}.
   * @param group um grupo de parmetros do algoritmo principal
   * 
   * @throws ParseException Formato de arquivo invlido.
   */
  public void loadParameterLoader(XmlParser parser, ParameterGroup group)
    throws ParseException {
    String algorithmName = parser.extractAttributeValue(
      PARAMETER_LOADER_ELEMENT_ALGORITHM_ATTRIBUTE);
    String algorithmVersionIdStr = parser.extractAttributeValue(
      PARAMETER_LOADER_ELEMENT_VERSION_ATTRIBUTE);
    AlgorithmVersionId algorithmVersionId = AlgorithmVersionId.create(
      algorithmVersionIdStr);
    if (algorithmVersionId == null) {
      throw new ParseException(
        "Verso invlida do atributo {0} do elemento {1}.",
        PARAMETER_LOADER_ELEMENT_VERSION_ATTRIBUTE, PARAMETER_LOADER_ELEMENT);
    }
    String outputParameterName = parser.extractAttributeValue(
      PARAMETER_LOADER_ELEMENT_OUTPUT_ATTRIBUTE);
    boolean validate_input = parser.extractAttributeValueAsBoolean(
      PARAMETER_LOADER_ELEMENT_VALIDATION_ATTRIBUTE, true);
    ParameterLoader parameterLoader = new ParameterLoader(algorithmName,
      algorithmVersionId, outputParameterName, validate_input);
    if (parser.goToFirstChild()) {
      do {
        parser.ensureElementName(INPUT_PARAMETER_ELEMENT);
        String inputParameterName = null;
        inputParameterName = parser.extractAttributeValue(
          INPUT_PARAMETER_ELEMENT_NAME_ATTRIBUTE);
        if (!parameterLoader.addInputParameterName(inputParameterName)) {
          throw new ParseException(
            "O parmetro {0} referenciado pelo elemento {1} j existe.",
            inputParameterName, INPUT_PARAMETER_ELEMENT);
        }
        parser.checkAttributes();
      } while (parser.goToNextSibling());
      parser.goToParent();
    }
    parser.checkAttributes();
    if (!group.setParameterLoader(parameterLoader)) {
      throw new ParseException(
        "J existe um carregador de parmetros para o grupo {0}.", group);
    }
  }

}
