package csbase.logic.algorithms.parsers.elements.attributes;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parsers.elements.ParsedElement;

/**
 * Classe base para atributos de um elemento.
 */
public abstract class AbstractElementAttribute <T> implements
  IElementAttribute<T> {

  /**
   * Nome do atributo.
   */
  private final String name;

  /**
   * Tipo do atributo.
   */
  private final Class<T> type;

  /**
   * Indicativo do atributo opcional.
   */
  private final boolean optional;

  /**
   * Atributo com valor padro.
   */
  private IElementAttribute<T> defaultValueAttribute;

  /**
   * Valor padro do atributo.
   */
  private T defaultValue;

  /**
   * Construtor.
   *
   * @param name o nome do atributo.
   * @param type o tipo do atributo.
   * @param optional indicativo de atributo opcional.
   * @param defaultValue valor padro.
   * @param defaultValueAttribute atributo com valor padro.
   */
  protected AbstractElementAttribute(String name, Class<T> type, boolean optional,
    T defaultValue, IElementAttribute<T> defaultValueAttribute) {
    this.name = name;
    this.type = type;
    this.optional = optional;
    this.defaultValue = defaultValue;
    this.defaultValueAttribute = defaultValueAttribute;
  }

  /**
   * Define um parmetro que vai determinar valor padro deste atributo.
   *
   * @param defaultValueAttribute o parmetro.
   */
  public void setDefaultValueAttribute(
    IElementAttribute<T> defaultValueAttribute) {
    this.defaultValueAttribute = defaultValueAttribute;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IElementAttribute<T> getDefaultValueAttribute() {
    return defaultValueAttribute;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return name;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<T> getType() {
    return type;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isOptional() {
    return optional;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public T getDefaultValue() {
    return defaultValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean validate(ParsedElement definition) throws ParseException {
    Object attributeValue = definition.getAttributeValue(name);
    if (attributeValue == null) {
      if (!isOptional()) {
        if (getDefaultValue() != null) {
          IElementAttribute<T> defaultValueParameter =
            getDefaultValueAttribute();
          if (defaultValueParameter != null) {
            Object value =
              definition.getAttributeValue(defaultValueParameter.getName());
            if (value != null) {
              throw new ParseException(
                "O atributo obrigatrio {1} no foi encontrado.", name);
            }
          }
          else {
            throw new ParseException(
              "O atributo obrigatrio {1} no foi encontrado.", name);
          }
        }
      }
    }
    return true;
  }

}
