package csbase.logic.algorithms.parsers.elements.attributes;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.TextParameter;
import csbase.logic.algorithms.parsers.BooleanParameterFactory;
import csbase.logic.algorithms.parsers.DoubleParameterFactory;
import csbase.logic.algorithms.parsers.EnumerationParameterFactory;
import csbase.logic.algorithms.parsers.IntegerParameterFactory;
import csbase.logic.algorithms.parsers.TextParameterFactory;

/**
 * Atributo cujo tipo depende do parmetro especificado.
 */
public class DerivedTypeAttribute extends AbstractElementAttribute
  implements StringToValueConverter<Object> {

  /**
   * O parmetro.
   */
  private final Parameter<?> parameter;

  /**
   * Construtor.
   * @param name o nome do atributo.
   * @param optional indicativo de atributo opcional.
   * @param defaultValue valor padro.
   * @param parameter o parmetro.
   */
  DerivedTypeAttribute(String name, boolean optional, Object defaultValue,
    Parameter<?> parameter) {
    super(name, Object.class, optional, defaultValue, null);
    this.parameter = parameter;
  }

  /**
   * Construtor para atributos opcionais.
   *
   * @param name o nome do atributo.
   * @param defaultValue valor padro.
   * @param parameter o parmetro.
   */
  public DerivedTypeAttribute(String name, Object defaultValue,
    Parameter<?> parameter) {
    this(name, true, defaultValue, parameter);
  }

  /**
   * Construtor para atributos obrigatrios.
   *
   * @param name nome do atributo.
   * @param parameter o parmetro.
   */
  public DerivedTypeAttribute(String name, Parameter<?> parameter) {
    this(name, false, null, parameter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public StringToValueConverter<?> getValueConverter() {
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object valueOf(String string) throws ParseException {
    if (parameter instanceof BooleanParameter) {
      BooleanAttribute booleanAttribute =
        new BooleanAttribute(getName(), isOptional(),
          (Boolean) getDefaultValue(), null);
      StringToValueConverter<Boolean> valueConverter =
        booleanAttribute.getValueConverter();
      return valueConverter.valueOf(string);
    }
    else if (parameter instanceof TextParameter) {
      StringAttribute stringAttribute =
        new StringAttribute(getName(), isOptional(),
          (String) getDefaultValue(), null);
      StringToValueConverter<String> valueConverter =
        stringAttribute.getValueConverter();
      return valueConverter.valueOf(string);
    }
    else if (parameter instanceof DoubleParameter) {
      DoubleAttribute attribute =
        new DoubleAttribute(getName(), isOptional(), (Double) getDefaultValue(),
          null, null, null, null, null);
      StringToValueConverter<Double> valueConverter =
        attribute.getValueConverter();
      return valueConverter.valueOf(string);
    }
    else if (parameter instanceof IntegerParameter) {
      IntegerAttribute attribute = new IntegerAttribute(getName(), isOptional(),
        (Integer) getDefaultValue(), null, null, null, null, null);
      StringToValueConverter<Integer> valueConverter =
        attribute.getValueConverter();
      return valueConverter.valueOf(string);
    }
    else if (parameter instanceof EnumerationParameter) {
      StringAttribute stringAttribute = new StringAttribute(getName());
      StringToValueConverter<String> valueConverter =
        stringAttribute.getValueConverter();
      String itemId = valueConverter.valueOf(string);
      EnumerationParameter enumerationParameter =
        (EnumerationParameter) parameter;
      EnumerationItem item = enumerationParameter.getItem(itemId);
      if (item == null) {
        throw new ParseException(
          "O item {1} do parmetro {0} referenciado em uma condio no foi "
            + "definido.",
          parameter, itemId);
      }
      return item;
    }
    else {
      throw new ParseException(
        "O parmetro {0} no  aceito.\n" + "Apenas os" + " tipos {1}, {2}, "
          + "{3}, {4} e {5} so permitidos.",
        parameter, BooleanParameterFactory.BOOLEAN_PARAMETER_ELEMENT,
        TextParameterFactory.TEXT_PARAMETER_ELEMENT,
        DoubleParameterFactory.DOUBLE_PARAMETER_ELEMENT,
        IntegerParameterFactory.INTEGER_PARAMETER_ELEMENT,
        EnumerationParameterFactory.ENUMERATION_PARAMETER_ELEMENT);
    }
  }

}