package csbase.logic.algorithms;

import java.io.Serializable;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

import csbase.logic.AlgoEvent;
import csbase.logic.IdFactory;
import csbase.logic.IdInterface;

/**
 * Essa classe representa um conjunto de categorias de algoritmos.
 * 
 */
public class CategorySet implements Serializable {

  /** Categorias de algoritmos */
  private SortedSet<Category> categories;

  /** O gerador de identificadores das categorias */
  private IdFactory idFactory;

  /** Flag que indica se esse conjunto de categorias foi ou no salvo */
  private boolean savedFlag;

  /** Objeto que permite a observao local desse conjunto de categorias */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Constri o conjunto de categorias vazio.
   * 
   */
  public CategorySet() {
    this.categories = Collections.synchronizedSortedSet(
      new TreeSet<Category>());
    idFactory = new IdFactory(convertToList(getAllCategories()));
  }

  /**
   * Converte um conjunto ordenado de categorias em uma lista de categorias.
   * 
   * @param categorySet conjunto ordenado de categorias
   * @return a lista de categorias correspondente
   */
  private List<? extends IdInterface> convertToList(
    SortedSet<Category> categorySet) {
    List<Category> categoryList = new Vector<Category>();
    categoryList.addAll(getAllCategories());
    return categoryList;
  }

  /**
   * Constri o conjunto de categorias a partir das categorias disponveis no
   * sistema.
   * 
   * @param categories categorias disponveis no sistema
   */
  public CategorySet(SortedSet<Category> categories) {
    idFactory = new IdFactory(convertToList(getAllCategories()));
    this.categories = categories;
  }

  /**
   * Obtm as categorias de algoritmos disponveis no conjunto.
   * 
   * @return as categorias de algoritmos
   */
  public SortedSet<Category> getCategories() {
    return categories;
  }

  /**
   * Estabelece as categorias de algoritmos disponveis no conjunto.
   * 
   * @param categories as categorias de algoritmos
   */
  public void setCategories(SortedSet<Category> categories) {
    this.categories = categories;
  }

  /**
   * Adiciona uma categoria ao conjunto de categorias.
   * 
   * @param category categoria a ser adicionada
   */
  public void addCategory(Category category) {
    categories.add(category);
  }

  /**
   * Cria uma nova categoria, a partir do nome especificado, na categoria pai.
   * Se a categoria pai for nula, a categoria  criada diretamente no conjunto,
   * caso contrrio, ela  adicionada  categoria pai.
   * 
   * @param parentCategory categoria pai
   * @param name nome da categoria sendo criada
   * 
   * @return a categoria criada
   */
  public Category createCategory(Category parentCategory, String name) {
    Category category = new Category(parentCategory, getNextCategoryId(), name);
    if (parentCategory != null) {
      parentCategory.addCategory(category);
    }
    else {
      addCategory(category);
    }
    return category;
  }

  /**
   * Atualiza uma categoria a partir dos dados da categoria especificada.
   * 
   * @param category categoria cujos dados sero copiados
   */
  public void updateCategory(Category category) {
    Category currentCategory = getCategory(category.getId());
    if (currentCategory != null) {
      currentCategory.updateCategory(category);
    }
  }

  /**
   * Remove uma determinada categoria desse conjunto de categorias.
   * 
   * @param id identificador da categoria a ser removida
   * 
   * @return a categoria removida
   */
  public Category removeCategory(String id) {
    return removeCategory(id, categories.iterator());
  }

  /**
   * Remove uma categoria de um iterador sobre uma coleo de categorias.
   * 
   * @param id identificador da categoria a ser removida
   * @param catIterator iterador sobre a coleo a ser pesquisada
   * 
   * @return a categoria removida
   */
  private Category removeCategory(String id, Iterator<Category> catIterator) {
    while (catIterator.hasNext()) {
      Category category = catIterator.next();
      if (category.getId().equals(id)) {
        freeAllSubCategories(category);
        catIterator.remove();
        idFactory.free(id);
        return category;
      }
      Category removedCategory = removeCategory(id, category.getCategories()
        .iterator());
      if (removedCategory != null) {
        return removedCategory;
      }
    }
    return null;
  }

  /**
   * Libera todos os ids das sub-categorias que fazem parte da categoria
   * especificada.
   * 
   * @param category categoria que est sendo removida
   */
  private void freeAllSubCategories(Category category) {
    for (Category subCat : category.getCategories()) {
      idFactory.free(subCat.getId());
      // System.out.println("liberando id da subcategoria removida: "
      // + subCat.getId());
      freeAllSubCategories(subCat);
    }
  }

  /**
   * Remove todas as categorias do conjunto de categorias.
   */
  public void removeAllCategories() {
    categories.clear();
  }

  /**
   * Obtm uma determinada categoria nesse conjunto de categorias.
   * 
   * @param id identificador da categoria procurada
   * @return a categoria procurada, caso contrrio, retorna null
   */
  public Category getCategory(String id) {
    if (id == null) {
      return null;
    }
    return getCategory(id, categories);
  }

  /**
   * Obtm uma determinada categoria que esteja na raiz do conjunto de
   * categorias, ou seja, no deve buscar nas sub-categorias da categoria.
   * 
   * @param categoryId identificador da categoria procurada
   * @return retorna true se a categoria procurada est na raiz do conjunto de
   *         categorias, caso contrrio, retorna null
   */
  public Category getRootCategory(String categoryId) {
    if (categoryId == null) {
      return null;
    }
    for (Category category : categories) {
      if (category.getId().equals(categoryId)) {
        return category;
      }
    }
    return null;
  }

  /**
   * Obtm uma determinada categoria na lista de categorias especificada. Essa
   * categoria pode estar no primeiro nvel ou ser uma sub-categoria.
   * 
   * @param id identificador da categoria procurada
   * @param categories conjunto de categorias onde procurar a categoria
   * @return a categoria procurada, caso contrrio, retorna null
   */
  private Category getCategory(String id, SortedSet<Category> categories) {
    for (Category category : categories) {
      if (category.getId().equals(id)) {
        return category;
      }
      Category subCategory = getCategory(id, category.getCategories());
      if (subCategory != null) {
        return subCategory;
      }
    }
    return null;
  }

  /**
   * Adiciona os algoritmos especificados em uma determinada categoria.
   * 
   * @param category categoria onde os algoritmos sero adicionados
   * @param algorithms algoritmos a serem adicionados
   */
  public void addAlgorithms(Category category, List<AlgorithmInfo> algorithms) {
    for (AlgorithmInfo algoInfo : algorithms) {
      category.addAlgorithm(algoInfo);
    }
  }

  /**
   * Adiciona os algoritmos especificados em uma determinada categoria.
   * 
   * @param categories categorias onde sero adicionados os algoritmos
   * @param algorithms algoritmos a serem adicionados
   */
  public void addAlgorithmsToCategories(SortedSet<Category> categories,
    List<AlgorithmInfo> algorithms) {
    for (Category category : categories) {
      addAlgorithms(category, algorithms);
    }
  }

  /**
   * Remove os algoritmos especificados em uma determinada categoria.
   * 
   * @param category categoria em que os algoritmos sero removidos
   * @param algorithms algoritmos a serem removidos
   */
  public void removeAlgorithms(Category category,
    List<AlgorithmInfo> algorithms) {
    for (AlgorithmInfo algoInfo : algorithms) {
      category.removeAlgorithm(algoInfo);
    }
  }

  /**
   * Remove os algoritmos especificados das categorias especificadas.
   * 
   * @param categories categorias em que sero removidos os algoritmos
   * @param algorithms algoritmos a serem removidos
   */
  public void removeAlgorithmsFromCategories(SortedSet<Category> categories,
    List<AlgorithmInfo> algorithms) {
    for (Category removedCategory : categories) {
      removeAlgorithms(getCategory(removedCategory.getId()), algorithms);
    }
  }

  /**
   * Verifica se o conjunto de categorias est vazio.
   * 
   * @return retorna true se no tiver categorias no conjunto, caso contrrio,
   *         retorna false
   */
  public boolean isEmpty() {
    return categories.isEmpty();
  }

  /**
   * Obtm o prximo identificador nico para uma nova categoria.
   * 
   * @return um identificador nico pra categoria
   */
  private String getNextCategoryId() {
    return String.valueOf(idFactory.next());
  }

  /**
   * Esse mtodo  chamado quando o servio de algoritmos sofre alguma alterao
   * relativa a categoria de algoritmos.
   * 
   * Sempre que a classe  avisada pelo servio de algoritmos de que alguma
   * mudana ocorreu, os observadores locais tambm so notificados.
   * 
   * @param event o evento que ocorreu no servio de algoritmo
   * 
   * @throws Exception Em caso de erro.
   */
  public static void update(AlgoEvent event) throws Exception {
    observable.notifyObservers(event);
  }

  /**
   * Adiciona um observador local da classe.
   * 
   * @param o um observador local
   */
  public static void addObserver(Observer o) {
    observable.addObserver(o);
  }

  /**
   * Remove um observador local da classe.
   * 
   * @param o o observador a ser removido
   */
  public static void deleteObserver(Observer o) {
    observable.deleteObserver(o);
  }

  /**
   * Obtm o nmero de categorias que existem na raiz desse conjunto.
   * 
   * @return o nmero de categorias que existem na raiz desse conjunto
   */
  public int getSize() {
    return categories.size();
  }

  /**
   * Obter todas as categorias disponveis no servidor, inclusive as
   * subcategorias.
   * 
   * @return as categorias disponveis no servidor, inclusive as subcategorias
   */
  public SortedSet<Category> getAllCategories() {
    return getAllCategories(categories);
  }

  /**
   * Obter todas as categorias disponveis no servidor, inclusive as
   * subcategorias.
   * 
   * @param rootCategories categorias raiz, cujas sub-categorias sero buscadas
   * 
   * @return as categorias disponveis no servidor, inclusive as subcategorias
   */
  private SortedSet<Category> getAllCategories(
    SortedSet<Category> rootCategories) {
    SortedSet<Category> allCategories = new TreeSet<Category>();
    for (Category category : rootCategories) {
      allCategories.add(category);
      SortedSet<Category> subCategories = getAllCategories(category
        .getCategories());
      if (!subCategories.isEmpty()) {
        allCategories.addAll(subCategories);
      }
    }
    return allCategories;
  }

  /**
   * Estabelece um novo estado para o flag que indica se esse conjunto de
   * categorias foi ou no salvo
   * 
   * @param state se true, indica que o conjunto de categorias foi salvo, caso
   *        contrrio, ocorreu um erro durante a persistncia e o conjunto no
   *        foi salvo
   */
  public void setCategorySetSavedFlag(boolean state) {
    savedFlag = state;
  }

  /**
   * Verifica se esse conjunto de categorias foi ou no salvo com sucesso.
   * 
   * @return retorna true, se o conjunto de categorias foi salvo, caso
   *         contrrio, retorna false (quando ocorre um erro durante a
   *         persistncia)
   */
  public boolean isCategorySetSaved() {
    return savedFlag;
  }

  /**
   * Remove todos os algoritmos de todas as categorias especificadas.
   * 
   * @param categories categorias em que sero removidos os algoritmos
   */
  public void removeAllAlgorithmsFromCategories(
    SortedSet<Category> categories) {
    for (Category category : categories) {
      category.removeAllAlgorithms();
    }
  }

  /**
   * Obtm a primeira categoria do conjunto.
   * 
   * @return a primeira categoria do conjunto
   */
  public Category getFirstCategory() {
    return categories.first();
  }

  /**
   * Modifica os identificadores lidos para esse conjunto de categorias, para
   * obter os nmeros sequencialmente a partir da estrutura de controle de ids.
   */
  public void changeCategoryIds() {
    SortedSet<Category> allCategories = getCategories();
    for (Category category : allCategories) {
      generateCategoryId(category, null);
    }
  }

  /**
   * Gera um id para a categoria especificada e passa o seu novo valor para as
   * sub-categorias alterarem tambm o id da categoria pai.
   * 
   * @param category categoria que est sendo gerado um novo id
   * @param parentCategory categoria pai da categoria
   */
  private void generateCategoryId(Category category, Category parentCategory) {
    String catId = getNextCategoryId();
    // System.out.println("CategorySet: mudando o id da categoria "
    // + category.getName() + " de [" + category.getId() + "] para o id ["
    // + catId + "]");

    category.setId(catId);
    category.setParentCategory(parentCategory);
    SortedSet<Category> subCategories = category.getCategories();
    for (Category subCat : subCategories) {
      generateCategoryId(subCat, category);
    }
  }

  /**
   * Obtm os nomes completos de todas as categorias que contm o algoritmo
   * especificado.
   * 
   * @param algoInfo informaes do algoritmo procurado
   * @return uma lista com os nomes completos de todas as categorias que contm
   *         o algoritmo
   */
  public List<String> getAlgorithmCategoriesFullNames(AlgorithmInfo algoInfo) {
    List<String> categoriesPath = new Vector<String>();
    for (Category category : getAllCategories()) {
      if (category.containsAlgorithm(algoInfo)) {
        categoriesPath.add(category.getFullName());
      }
    }
    return categoriesPath;
  }

  /**
   * Obtm os nomes completos de todas as categorias que contm o algoritmo
   * especificado.
   * 
   * @param algoName informaes do algoritmo procurado
   * @return uma lista com os nomes completos de todas as categorias que contm
   *         o algoritmo
   */
  public List<String> getAlgorithmCategoriesFullNames(String algoName) {
    List<String> categoriesPath = new Vector<String>();
    for (Category category : getAllCategories()) {
      if (category.containsAlgorithm(algoName)) {
        categoriesPath.add(category.getFullName());
      }
    }
    return categoriesPath;
  }

  /**
   * Obtm uma lista de categorias por atravs de uma busca feita a partir de
   * uma lista de nomes de categorias recebida como parmetro.
   * 
   * @param categoryFullNames lista de nome de categorias.
   * @return lista de categorias.
   */
  public List<Category> getCategoriesFromFullNames(
    List<String> categoryFullNames) {
    List<Category> categories = new Vector<Category>();
    for (Category category : getAllCategories()) {
      if (categoryFullNames.contains(category.getFullName())) {
        categories.add(category);
      }
    }
    return categories;
  }

  /**
   * Obtm uma categoria, dado o nome da categoria pai e o nome da categoria a
   * ser obtida.
   * 
   * @param parentCategoryFullName nome da categoria pai.
   * @param catFullName nome da categoria.
   * @return a categoria.
   */
  public Category getCategory(String parentCategoryFullName,
    String catFullName) {
    if (parentCategoryFullName == null) {
      //Busca s nas categorias da raiz para otimizar
      SortedSet<Category> rootCategories = getCategories();
      for (Category category : rootCategories) {
        if (category.getFullName().equals(catFullName)) {
          return category;
        }
      }
    }
    else {
      SortedSet<Category> allCategories = getAllCategories();
      for (Category category : allCategories) {
        Category parentCategory = category.getParentCategory();
        String currentParentFullName = (parentCategory != null) ? parentCategory
          .getFullName() : null;
        String currentCatName = category.getFullName();
        if (currentParentFullName != null && currentParentFullName.equals(
          parentCategoryFullName) && currentCatName.equals(catFullName)) {
          return category;
        }
      }
    }
    return null;
  }
}
