/*
 * $Id$
 */
package csbase.logic;

import java.lang.reflect.Field;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.atomic.AtomicBoolean;

import csbase.exception.CSBaseException;
import csbase.exception.IncompatibleVersionException;
import csbase.remote.RemoteObservable;
import csbase.remote.RemoteObserver;
import csbase.remote.ServerEntryPoint;
import csbase.remote.ServiceInterface;
import csbase.util.rmi.Pingable;
import tecgraf.javautils.core.lng.LNG;

/**
 * Guarda os dados referentes a monitorao de um servidor remoto. Mantm a
 * sesso, os ouvintes, observadores e a referncia para o servios exportados.
 * O servidor  considerado vivo se est alcanvel e o login foi realizado com
 * sucesso.
 * 
 * @author Tecgraf/PUC-Rio
 * 
 */
public abstract class MonitoredServer implements ServerMonitorListener {

  /** Lista dos ouvintes */
  protected List<MonitoredServerListener> listeners;

  /** Mapa dos observadores cadastrados */
  protected Map<String, List<ObserverData>> observers;

  /** Mapa dos servios exportados */
  protected Map<String, ServiceInterface> services;

  /** O monitor do servidor */
  protected ServerMonitor monitor = null;

  /** A sesso do usurio logado */
  protected Session session = null;

  /**
   * Estado do servidor. Considerado vivo (true) se est alcanvel e o usurio
   * logado, false caso contrrio.
   */
  protected AtomicBoolean alive = new AtomicBoolean(false);

  /** Classe ou interface que mantm as referncias para os objetos remotos. */
  protected Class<?> locator;

  /** Indica se esse servidor monitorado  o default */
  protected AtomicBoolean defaultServer = new AtomicBoolean(false);

  /** Nome do sistema a ser definido no servidor */
  protected String systemName;

  /** Usurio a ser delegado no login */
  protected String delegatedLogin = null;

  /** TimeZone para usa no login */
  protected TimeZone timeZone = TimeZone.getDefault();

  /** Thread do relogin automtico */
  private Thread reloginThread;

  /**
   * Conjunto com os nomes dos servios a serem recuperados do servidor. Usado
   * no caso onde o locator no  necessrio e funcionaria apenas como
   * repositrio para descoberta dos servios disponveis
   */
  protected Set<String> serviceNames;

  /** Indica se a verificao de verso entre cliente e servidor ser ignorada */
  protected boolean ignoreVersion = false;

  /**
   * Indica se mensagens informativas devem ser exibidas. Se for
   * <code>false</code>, apenas erros e alertas sero exibidos.
   */
  private final boolean verbose;

  /** Valor mximo da janela de backoff. */
  private int windowSize;

  /**
   * Valor default para o tamanho mximo da janela, caso nenhum valor esteja
   * configurado no Client.properties
   */
  public static final int RELOAD_TIME = 5;

  /**
   * Constri uma monitorao.
   * 
   * @param serverURI A URI do servidor a ser monitorado
   * @param locator A classe do locator
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        e servidor ser ignorada
   * @param windowSize Valor mximo da janela de backoff.
   */
  protected MonitoredServer(ServerURI serverURI, Class<?> locator,
    boolean ignoreVersion, int windowSize) {
    this(serverURI, locator, ignoreVersion, true, windowSize);
  }

  /**
   * Constri uma monitorao.
   * 
   * @param serverURI A URI do servidor a ser monitorado
   * @param locator A classe do locator
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        e servidor ser ignorada
   * @param verbose <code>true</code> se informaes devem ser exibidas,
   *        <code>false</code> se apenas erros e alertas devem ser exibidos
   * @param windowSize Valor mximo da janela de backoff.
   */
  protected MonitoredServer(ServerURI serverURI, Class<?> locator,
    boolean ignoreVersion, boolean verbose, int windowSize) {
    if (serverURI == null) {
      throw new IllegalStateException("serverURI == null");
    }

    if (windowSize <= 0) {
      this.windowSize = RELOAD_TIME;
    }
    else {
      this.windowSize = windowSize;
    }

    this.verbose = verbose;
    this.locator = locator;
    this.monitor = new ServerMonitor(serverURI, 0, verbose);
    this.monitor.addListener(this);
    this.listeners = new ArrayList<MonitoredServerListener>();
    this.observers = new Hashtable<String, List<ObserverData>>();
    this.services = new Hashtable<String, ServiceInterface>();
    this.ignoreVersion = ignoreVersion;
  }

  /**
   * Constri uma monitorao.
   * 
   * @param serverURI A URI do servidor a ser monitorado
   * @param locator A classe do locator
   * @param windowSize Valor mximo da janela de backoff.
   */
  protected MonitoredServer(ServerURI serverURI, Class<?> locator,
    int windowSize) {
    this(serverURI, locator, false, windowSize);
  }

  /**
   * Constri uma monitorao.
   * 
   * @param serverURI A URI do servidor a ser monitorado
   * @param serviceNames Conjunto com os nomes dos servios a serem
   *        disponibilizados
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        e servidor ser ignorada
   * @param windowSize Valor mximo da janela de backoff.
   */
  protected MonitoredServer(ServerURI serverURI, Set<String> serviceNames,
    boolean ignoreVersion, int windowSize) {
    this(serverURI, (Class<?>) null, ignoreVersion, windowSize);
    this.serviceNames = serviceNames;
  }

  /**
   * Constri uma monitorao.
   * 
   * @param serverURI A URI do servidor a ser monitorado
   * @param serviceNames Conjunto com os nomes dos servios a serem
   *        disponibilizados
   * @param windowSize Valor mximo da janela de backoff.
   */
  protected MonitoredServer(ServerURI serverURI, Set<String> serviceNames,
    int windowSize) {
    this(serverURI, (Class<?>) null, false, windowSize);
    this.serviceNames = serviceNames;
  }

  /**
   * Marca esta monitorao como sendo a do servidor default
   *
   * @param isDefaultServer verdadeiro se a monitorao  a do servidor
   * default ou falso, caso contrrio.
   */
  protected final void setAsDefaultServer(boolean isDefaultServer) {
    if (this.defaultServer.get() != isDefaultServer) {
      this.defaultServer.set(isDefaultServer);
      if (this.isDefault() && this.isAlive()) {
        this.fillLocatorFields();
      }
    }
  }

  /**
   * Verifica se um determinado servidor consegue acessar o host a partir do
   * qual este mtodo foi chamado, em uma determinada porta.
   * 
   * @param uri A URI que india o servidor.
   * @param port Porta a ser testada.
   * @return <tt>True</tt> se o servidor consegue acessar o host a partir do
   *         qual este mtodo foi chamado, em uma dada porta.
   * 
   * @throws RemoteException Em caso de falha na comunicao com o servidor.
   */
  public boolean canReach(ServerURI uri, int port) throws RemoteException {
    Pingable pingable = new Pingable();
    UnicastRemoteObject.exportObject(pingable, port);
    try {
      return monitor.getServer().canReach(pingable);
    }
    finally {
      UnicastRemoteObject.unexportObject(pingable, true);
    }
  }

  /**
   * Inicia a thread de relogin automtico. Se o relogin falhar novas tentativas
   * sero feitas em intervalos definidos pelo atributo RELOGIN_TIME.
   * 
   * 
   * @param serverURI A URI do servidor para relogin
   */
  private void startReloginThread(final ServerURI serverURI) {

    /** Tamanho mximo para a janela de backoff para relogin **/
    final int maxWinSize = windowSize;

    if (this.reloginThread == null || !this.reloginThread.isAlive()) {
      this.reloginThread = new Thread("reloginThread-" + serverURI) {

        @Override
        public void run() {

          int winSize = 1;
          while (true) {
            try {
              Thread.sleep(getNextTime(winSize));

              if (validate()) {
                notifyConnectionReestablished(serverURI);
                break;
              }

              // Verifica se a janela j est em seu tamanho mximo
              if (winSize < maxWinSize) {
                winSize++;
              }
            }
            catch (RemoteException e) {
              System.err.println(LNG.get(
            		  "csbase.logic.CommunicationFail"));
            }
            catch (Throwable t) {
              System.err.println(LNG.get(
            		  "csbase.logic.ReloginFail"));
              t.printStackTrace();
              break;
            }
          }
        }
      };
      this.reloginThread.start();
    }
  }

  /**
   * Retorna o prximo tempo de espera para se relogar
   * 
   * @param winSize
   * @return Tempo de espera em milisegundos
   */
  private long getNextTime(int winSize) {
    return (long) ((1 + Math.random() * Math.pow(2, winSize)) * 1000);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void notifyServerReachable(ServerURI serverURI) {
    this.startReloginThread(serverURI);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void notifyServerUnreachable(ServerURI serverURI) {
    this.alive.set(false);
    this.notifyConnectionLost(serverURI);
  }

  /**
   * Notifica os ouvintes que o servidor est vivo novamente.
   * 
   * @param serverURI a URI do servidor em questo.
   */
  private void notifyConnectionReestablished(ServerURI serverURI) {
    synchronized (this.listeners) {
      for (MonitoredServerListener l : this.listeners) {
        l.notifyConnectionReestablished(serverURI);
      }
    }
  }

  /**
   * Notifica os ouvintes perda da comunicao com o servidor.
   * 
   * @param serverURI a URI do servidor em questo.
   */
  private void notifyConnectionLost(ServerURI serverURI) {
    synchronized (this.listeners) {
      for (MonitoredServerListener l : this.listeners) {
        l.notifyConnectionLost(serverURI);
      }
    }
  }

  /**
   * Notifica os ouvintes o login bem sucedido
   * 
   * @param serverURI a URI do servidor em questo.
   */
  private void notifyLoggedIn(ServerURI serverURI) {
    synchronized (this.listeners) {
      for (MonitoredServerListener l : this.listeners) {
        l.notifyLoggedIn(serverURI);
      }
    }
  }

  /**
   * Notifica os ouvintes o logout bem sucedido
   * 
   * @param serverURI a URI do servidor em questo.
   */
  private void notifyLoggedOut(ServerURI serverURI) {
    synchronized (this.listeners) {
      for (MonitoredServerListener l : this.listeners) {
        l.notifyLoggedOut(serverURI);
      }
    }
  }

  /**
   * Adiciona um ouvinte.
   * 
   * @param listener A instncia de um ouvinte
   */
  protected final void addListener(MonitoredServerListener listener) {
    synchronized (this.listeners) {
      if (!this.listeners.contains(listener)) {
        this.listeners.add(listener);
      }
    }
  }

  /**
   * Remove um ouvinte
   * 
   * @param listener A Instncia de um ouvinte
   */
  protected final void deleteListener(MonitoredServerListener listener) {
    synchronized (this.listeners) {
      this.listeners.remove(listener);
    }
  }

  /**
   * Adiciona o observador ao servio observado.Caso no haja comunicao
   * estabelecida com o servidor, o observador  guardado para envio posterior.
   * 
   * @param serviceName Nome do servio em que o observador  adicionado.
   * @param observer Observador remoto.
   * @param arg Argumento associado ao observador.
   * 
   * @return true, caso tenha conseguido adicionar o observador, ou false, caso
   *         contrrio.
   */
  public final synchronized boolean addObserver(String serviceName,
    RemoteObserver observer, Object arg) {
    if (!this.observers.containsKey(serviceName)) {
      this.observers.put(serviceName, new LinkedList<ObserverData>());
    }
    List<ObserverData> observerList = this.observers.get(serviceName);
    observerList.add(new ObserverData(observer, arg));
    try {
      if (!this.isAlive()) {
        return false;
      }
      RemoteObservable remoteObservable =
        (RemoteObservable) this.services.get(serviceName);
      if (remoteObservable == null) {
        System.err.println(serviceName + LNG.get( 
        		"csbase.logic.ObserverAddFail"));
        return false;
      }
      remoteObservable.addObserver(observer, arg);
      return true;
    }
    catch (RemoteException ex) {
      this.monitor.invalidate();
      System.err.println(ex);
    }
    return false;
  }

  /**
   * Remove o par (observador, arg) do servio dado.
   * 
   * @param serviceName Nome do servio do qual o observador ser retirado.
   * @param observer Observador remoto.
   * @param arg Argumento associado ao observador.
   */
  public final synchronized void deleteObserver(String serviceName,
    RemoteObserver observer, Object arg) {
    if (!this.observers.containsKey(serviceName)) {
      return;
    }
    List<ObserverData> observerList = this.observers.get(serviceName);
    ObserverData obsData = new ObserverData(observer, arg);
    observerList.remove(obsData);
    if (observerList.size() == 0) {
      this.observers.remove(serviceName);
    }
    removeObserver(serviceName, observer, arg);
  }

  /**
   * Remove todos os observadores que este cliente cadastrou.
   * 
   * @param serviceName O nome do servio
   * @param observer A instncia do observador remoto
   * @param arg O argumento do observador remoto
   */
  private synchronized void removeObserver(String serviceName,
    RemoteObserver observer, Object arg) {
    if (!this.monitor.isReachable()) {
      System.err.println(LNG.get(
    		  "csbase.logic.OffLineWhenRemoving") + observer);
      return;
    }
    try {
      RemoteObservable remoteObservable =
        (RemoteObservable) this.services.get(serviceName);
      remoteObservable.deleteObserver(observer, arg);
    }
    catch (RemoteException e) {
      this.monitor.invalidate();
      System.err.println(e);
    }
  }

  /**
   * Exibe uma mensagem informativa (se {@link #verbose} = <code>true</code>).
   * 
   * @param msg mensgem
   */
  private void printInfo(String msg) {
    if (verbose) {
      System.out.println(msg);
    }
  }

  /**
   * Envia os observadores nos servios correspondentes.
   */
  protected synchronized final void sendObservers() {
    if (observers.isEmpty()) {
      // Mensagem para verificar problema no envio de notificaes.
      printInfo(LNG.get("csbase.logic.EmptyObserverList"));
      return;
    }

    Iterator<String> serviceNameIterator = observers.keySet().iterator();
    while (serviceNameIterator.hasNext()) {
      String serviceName = serviceNameIterator.next();
      // Mensagem para verificar problema no envio de notificaes.
      printInfo(LNG.get("csbase.logic.SendObserver") + serviceName);
      List<ObserverData> observerList = observers.get(serviceName);
      RemoteObservable remoteObservable =
        (RemoteObservable) services.get(serviceName);
      if (remoteObservable == null) {
        String msg = serviceName + LNG.get("csbase.logic.ObserverAddFail");
        System.err.println(msg);
      }
      else {
        // Mensagem para verificar problema no envio de notificaes.
        printInfo("MonitoredServer: " + observerList.size() +
        		LNG.get("csbase.logic.Observers"));
        for (ObserverData obsDat : observerList) {
          try {
            remoteObservable.addObserver(obsDat.observer, obsDat.arg);
          }
          catch (Exception ex) {
            this.monitor.invalidate();
            System.err.println(ex);
          }
        }
      }
    }
  }

  /**
   * Fora revalidar a conexo com o servidor e a sesso.<br>
   * Se o servidor estiver fora, tenta reconectar. <br>
   * Se tiver acesso ao servidor mas a sesso caiu, desloga e loga novamente.<br>
   */
  public void invalidate() {
    if (monitor == null) {
      return;
    }
    if (!isAlive()) {
      return;
    }

    if (monitor.tryReaching()) {
      // Obtm um servio qualquer para testar se a sesso ainda  vlida.
      Iterator<ServiceInterface> iter = services.values().iterator();
      if (!iter.hasNext()) {
        // Ou o usurio deslogou aps o teste do isAlive ou ainda no carregou
        // os servios.
        return;
      }
      ServiceInterface anyService = iter.next();

      try {
        // Chama um mtodo dummy de um servio qualquer.
        anyService.getName();
      }
      catch (RemoteException e) {
        /*
         * Se lanar RemoteException significa que a sesso no  mais vlida.
         * 
         * Isso costuma acontecer quando o cliente est rodando em um laptop que
         *  posto para dormir. Aps um tempo neste estado, o mtodo
         * ClientConnectionSpy.unrefered()  chamado dando a entender ao
         * servidor que o cliente caiu. Ao voltar do sleep, o cliente mantm
         * suas referncias ao servidor e aos servios, entretanto os servios
         * no constam mais na tabela RMI do servidor.
         */
        this.alive.set(false);
        this.notifyConnectionLost(monitor.getURI());
        this.startReloginThread(monitor.getURI());
      }
    }
    else {
      /*
       * Se o servidor no pode ser alcanado, deve-se requisitar que o monitor
       * continue tentando.
       * 
       * No precisa se preocupar cm fazer logout, pois no primeiro isso foi
       * feito no primeiro teste em que houve falha ao tentar conectar o
       * servidor. Naquele momento o monitor lanou um evento indicando que o
       * servidor no pode ser alcanado, fazendo com que essa instncia fizesse
       * logout.
       */
      monitor.invalidate();
    }
  }

  /**
   * Valida o servidor. Faz o login com o servidor, recupera os servios
   * exportados, envia os observadores, atribui o nome do sistema e em caso de
   * sucesso muda o estado do servidor para vivo.
   * 
   * @return true se validado com sucesso, false caso contrrio
   * @throws CSBaseException Em caso de erro durante a validao
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  protected boolean validate() throws CSBaseException, RemoteException {

    this.session = this.performLogin();

    if (this.session != null) {
      if (this.fetchServices()) {
        this.alive.set(true);
        if (this.isDefault()) {
          this.fillLocatorFields();
        }
        this.sendObservers();
        this.setSystemNameOnServer();
        printInfo(String.format(LNG.get("csbase.logic.LoggedIn"),
        		this.monitor.getURI()));
        this.postLogin();
      }
      else {
        this.session = null;
        this.alive.set(false);
        System.err.println(String.format(LNG.get("csbase.logic.LogInFail"),
        		this.monitor.getURI()) );
      }
    }
    else {
      this.alive.set(false);
    }

    return this.session != null;
  }

  /**
   * Define o nome do sistema no servidor
   * 
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  private void setSystemNameOnServer() throws RemoteException {
    if (this.systemName != null) {
      this.monitor.getServer().setSystemName(this.session.getKey(),
        this.systemName);
    }
  }

  /**
   * Define o nome do sistema a ser usado
   * 
   * @param systemName nome do sistema.
   */
  protected void setSystemName(String systemName) {
    this.systemName = systemName;
  }

  /**
   * Faz o login com o servidor.
   * 
   * @return A sesso do usurio logado ou null em caso de login/senha invlido.
   * 
   * @throws CSBaseException Em caso de erro durante o login
   * @throws RemoteException Em caso de erro na comunicao com o servidor
   */
  protected abstract Session performLogin() throws CSBaseException,
    RemoteException;

  /**
   * Executado aps login com sucesso.
   * 
   * @throws CSBaseException em caso de erro durante o login.
   */
  protected abstract void postLogin() throws CSBaseException;

  /**
   * Mtodo usado para traduo de termos internacionalizados.
   * 
   * @param key A chave do do termo
   * @return O valor traduzido
   */
  protected abstract String lng(String key);

  /**
   * Executa procedimentos de finalizao. Pra o monitor, remove os
   * observadores, executa o logout e remove os ponteiros para os listeners e
   * servios.
   * 
   */
  protected final void flush() {
    try {
      this.logout();
      this.monitor = null;
      this.listeners.clear();
      this.observers.clear();
      this.listeners = null;
      this.observers = null;
      this.services = null;
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Pra a monitorao, remove observadores e executa o logout no servidor.
   * 
   */
  protected final void logout() {
    try {
      if (this.isAlive()) {
        this.removeObservers();
        this.monitor.getServer().logout(this.session.getKey());
        this.notifyLoggedOut(this.getURI());
        printInfo(String.format(LNG.get("csbase.logic.LoggedOut"),
        		this.getURI()));
      }
    }
    catch (Exception e) {
      System.err.println(LNG.get("csbase.logic.LogOutError"));
      e.printStackTrace();
    }
    finally {
      this.monitor.stopMonitoring();
      this.session = null;
      this.alive.set(false);
    }
  }

  /**
   * Executa o login no servidor. Se a monitorao no estiver sendo executa
   * ser iniciada. Lana {@link IncompatibleVersionException} caso o servidor
   * no seja compatvel com o cliente.
   * 
   * @return true se login bem sucedido, false caso contrrio
   * @throws CSBaseException Em caso de falha durante o processo de login
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  protected final boolean login() throws CSBaseException, RemoteException {

    if (!this.isAlive()) {
      if (!this.monitor.lookup()) {
        String m =
          MessageFormat.format(lng("MonitoredServer.server_down"),
            this.getURI());
        throw new CSBaseException(m);
      }
    }

    if (this.validate()) {
      this.notifyLoggedIn(this.getURI());
      this.monitor.startMonitoring();
      return true;
    }
    else {
      return false;
    }

  }

  /**
   * Recupera as referncias para os servios exportados pelo servidor
   * 
   * @return true em caso de sucesso na recuperao dos servios exportados,
   *         false caso contrrio.
   * @throws CSBaseException Em caso de falha na recuperao do servios
   *         exportados
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  protected boolean fetchServices() throws CSBaseException, RemoteException {
    if (this.session == null) {
      throw new CSBaseException(MessageFormat.format(
        lng("MonitoredServer.nullsession"), this.getURI()));
    }

    this.services =
      this.getMonitor().getServer()
        .fetchServices(this.session.getKey(), this.getServicesNames());
    if (this.services != null && this.services.size() > 0) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Remove todos os observadores que este cliente cadastrou.
   */
  protected final synchronized void removeObservers() {
    Iterator<String> serviceNameIterator = observers.keySet().iterator();
    while (serviceNameIterator.hasNext()) {
      String serviceName = serviceNameIterator.next();
      List<ObserverData> observerList = observers.get(serviceName);
      RemoteObservable remoteObservable =
        (RemoteObservable) services.get(serviceName);
      for (ObserverData oa : observerList) {
        try {
          remoteObservable.deleteObserver(oa.observer, oa.arg);
        }
        catch (RemoteException e) {
          this.monitor.invalidate();
          System.err.println(e);
        }
      }
    }
    observers.clear();
  }

  /**
   * Consulta o estado do servidor. O servidor  considerado vivo se est
   * alcanvel e o login foi bem sucedido.
   * 
   * @return true para servidor vivo, false caso contrrio.
   */
  protected final boolean isAlive() {
    return this.alive.get();
  }

  /**
   * @return true se este servidor monitorado  o default, false caso contrrio
   */
  protected final boolean isDefault() {
    return this.defaultServer.get();
  }

  /**
   * @return Retorna a instncia do monitor ({@link ServerMonitor}).
   */
  protected final ServerMonitor getMonitor() {
    return this.monitor;
  }

  /**
   * @return A instncia da sesso
   */
  protected final Session getSession() {
    return this.session;
  }

  /**
   * @return Retorna a URI do servidor.
   */
  protected final ServerURI getURI() {
    return this.getMonitor().getURI();
  }

  /**
   * Preenche os campos do locator com as referncias para os servios.
   */
  protected final void fillLocatorFields() {

    if (this.locator == null) {
      return;
    }

    this.setValueOnLocatorField(ServerEntryPoint.class,
      this.monitor.getServer());
    Iterator<String> serviceNameIterator = this.services.keySet().iterator();
    while (serviceNameIterator.hasNext()) {
      String serviceName = serviceNameIterator.next();
      ServiceInterface service = this.services.get(serviceName);
      Class<?> serviceInterface =
        this.getServiceInterface(service, serviceName);
      this.setValueOnLocatorField(serviceInterface, service);
    }
  }

  /**
   * <p>
   * Obtm a interface de um servio.
   * </p>
   * <p>
   * A interface do servio  identificada por ter um campo com o nome
   * SERVICE_NAME.
   * </p>
   * 
   * @param service O servio.
   * @param serviceName O nome do servio.
   * 
   * @return A classe da interface do servio, ou null, caso no seja
   *         encontrada.
   */
  private Class<?> getServiceInterface(Object service, String serviceName) {
    Class<?>[] interfaces = service.getClass().getInterfaces();
    final String fieldName = "SERVICE_NAME";
    for (int i = 0; i < interfaces.length; i++) {
      Field f = null;
      try {
        f = interfaces[i].getField(fieldName);
        if (serviceName.equals(f.get(null))) {
          return interfaces[i];
        }
      }
      catch (NoSuchFieldException e) {
        /*
         * Caso o campo no exista, continua a busca.
         */
      }
      catch (IllegalAccessException e) {
        IllegalStateException ise =
          new IllegalStateException(MessageFormat.format(
    		  LNG.get("csbase.logic.FailFieldAccess"),
    		  new Object[] { fieldName, interfaces[i].getName() }));
        ise.initCause(e);
        throw ise;
      }
    }
    return null;
  }

  /**
   * Insere um valor num campo da classe que contm as referncias para os
   * objetos remotos ({@link #locator}).
   * 
   * @param fieldType A classe que representa o tipo do campo a ser preenchido.
   * @param value O valor que ser inserido no campo.
   * 
   * @return true, caso o valor seja inserido, ou false, caso contrrio.
   */
  private boolean setValueOnLocatorField(Class<?> fieldType, Object value) {
    Field[] fields = this.locator.getFields();
    for (int i = 0; i < fields.length; i++) {
      if (fields[i].getType().isAssignableFrom(fieldType)) {
        try {
          fields[i].set(null, value);
          return true;
        }
        catch (IllegalAccessException e) {
          IllegalStateException ise =
            new IllegalStateException(MessageFormat.format(
        		LNG.get("csbase.logic.FailFieldAccess"),
        		new Object[] { fields[i].getName(), this.locator.getName() }));
          ise.initCause(e);
          throw ise;
        }
      }
    }
    return false;
  }

  /**
   * @return O conjunto de strings com os nomes dos servios a serem recuperados
   *         do servidor. Usa o {#link ClientRemoteLocator} para descoberta dos
   *         servios ou atravs do atributo serviceNames
   */
  protected Set<String> getServicesNames() {

    if (this.locator == null) {
      return this.serviceNames;
    }

    Set<String> servicesNames = new HashSet<String>();
    Field[] fields = this.locator.getFields();
    for (int i = 0; i < fields.length; i++) {
      if (ServiceInterface.class.isAssignableFrom(fields[i].getType())) {
        try {
          Class<?> aServiceInterface = fields[i].getType();
          String serviceName =
            (String) aServiceInterface.getField("SERVICE_NAME").get(null);
          servicesNames.add(serviceName);
        }
        catch (NoSuchFieldException e) {
          // continua a busca
        }
        catch (IllegalAccessException e) {
          // continua a busca
        }
      }
    }
    return servicesNames;
  }

  /**
   * Define o usurio a ser delegado no login
   * 
   * @param delegatedLogin login do usurio a ser delegado
   */
  public void setDelegatedLogin(String delegatedLogin) {
    this.delegatedLogin = delegatedLogin;
  }

  /**
   * Define o time zone usado no login do cliente
   * 
   * @param timeZone A instncia de TimeZone a ser usado, null usa TimeZone
   *        default
   */
  public void setTimeZone(TimeZone timeZone) {
    if (timeZone != null) {
      this.timeZone = timeZone;
    }
  }
}
