/*
 * $Id$
 */
package csbase.logic;

import java.net.MalformedURLException;
import java.rmi.Naming;
import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.rmi.UnmarshalException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import csbase.remote.ServerEntryPoint;
import tecgraf.javautils.core.lng.LNG;

/**
 * Monitora uma conexo com um servidor CSBase identificado por uma
 * {@link ServerURI}. Somente guarda a referncia para o ponto de entrada do
 * servidor aps um lookup bem sucedido. Notifica atravs de ouvintes do tipo
 * {@link ServerMonitorListener} o estado do servidor que pode ser alcanvel ou
 * no alcanvel.
 * 
 * @author Tecgraf/PUC-Rio
 * 
 */
public class ServerMonitor {

  /** Ponto de entrada para comunicao com o servidor. */
  private ServerEntryPoint server;

  /** Tempo entre verificaes quando o servidor est vivo. */
  private final long OK_SLEEP_TIME = 60000; // 1 min

  /** Tempo entre verificaes quando o servidor est fora. */
  private final long NOK_SLEEP_TIME = 10000; // 10 seg

  /** URI do servidor. */
  private final ServerURI serverURI;

  /** Indica se o servidor est vivo. */
  private AtomicBoolean serverReachable = new AtomicBoolean(false);

  /** Indica se  para continuar monitorando o servidor. */
  private AtomicBoolean mustContinue = new AtomicBoolean(false);

  /** Os ouvintes deste monitor */
  private List<ServerMonitorListener> listeners;

  /**
   * Nmero mximo de tentativas para reestabelecer conexo. Zero significa
   * tentar indefinidamente
   */
  private int maxRetries;

  /**
   * Monitora a comunicao com o servidor. Caso haja mudana no estado dessa
   * comunicao, avisa aos observadores.
   */
  private Thread monitor;

  /** A thread de lookup do servidor */
  private Thread serverLookupThread;

  /**
   * Indica se mensagens informativas devem ser exibidas. Se for
   * <code>false</code>, apenas erros e alertas sero exibidos.
   */
  private boolean verbose = true;

  /**
   * Constri um monitor que ser monitorado indefinidamente.
   * 
   * @param serverURI A URI do servidor a ser monitorado.
   */
  public ServerMonitor(ServerURI serverURI) {
    this(serverURI, 0);
  }

  /**
   * Constri um monitor.
   * 
   * @param serverURI A URI do servidor a ser monitorado
   * @param maxRetries O nmero mximo de tentativas de reconexo. Zero
   *        significa tentar indefinidamente
   */
  public ServerMonitor(ServerURI serverURI, int maxRetries) {
    this(serverURI, maxRetries, true);
  }

  /**
   * Constri um monitor.
   * 
   * @param serverURI A URI do servidor a ser monitorado
   * @param maxRetries O nmero mximo de tentativas de reconexo. Zero
   *        significa tentar indefinidamente
   * @param verbose <code>true</code> se informaes devem ser exibidas,
   *        <code>false</code> se apenas erros e alertas devem ser exibidos
   */
  public ServerMonitor(ServerURI serverURI, int maxRetries, boolean verbose) {
    this.verbose = verbose;
    if (maxRetries < 0) {
      throw new IllegalArgumentException(LNG.get(
		  "csbase.logic.MaxRetriesPositive"));
    }
    this.maxRetries = maxRetries;
    this.serverURI = serverURI;
    this.listeners = new ArrayList<ServerMonitorListener>();
    this.createLookupthread();
  }

  /**
   * Cria a thread de monitorao
   */
  private final void createMonitorThread() {
    this.monitor = new Thread(new Runnable() {
      @Override
      public void run() {
    	    int retries = 0;
    	    while (mustContinue.get()) {
    	    	
    	      if (ServerMonitor.this.tryReaching()) {
      	        retries = 0;
    	      }
    	      else {
    	    	  if (maxRetries > 0 && ++retries >= maxRetries
    	    			  && !isReachable()) {
    	    		  logWarning(MessageFormat.format(
    	    				  LNG.get("csbase.logic.ReachedMaxRetries"),
    	    				  new Object[] { serverURI, maxRetries } ));
    	    		  stopMonitoring();
    	    	  }
    	      }

    	      try {
    	        synchronized (ServerMonitor.this) {
    	          ServerMonitor.this.wait(isReachable() ? OK_SLEEP_TIME
    	            : NOK_SLEEP_TIME);
    	        }
    	      }
    	      catch (InterruptedException e) {
    	        /**
    	         * No tem problema se a thread for interrompida.
    	         */
    	    	  e.printStackTrace();
    	      }
    	    }
      }
    }, "serverMonitorThread-" + this.getURI());
    try {
      this.monitor.setDaemon(true);
    }
    catch (SecurityException e) {
      /** Applet no pode colocar a thread rodando como daemon. */
    }
  }

  /**
   * Escreve no console a mensagem de erro e a exceo lanada
   * 
   * @param msg a mensagem
   * @param t a exceo lanada
   */
  protected void logError(String msg, Throwable t) {
    System.err.println(msg);
    System.err.println(t);
  }

  /**
   * Escreve no console uma mensagem de aviso
   * 
   * @param msg a mensagem
   */
  protected void logWarning(String msg) {
    System.out.println(msg);
  }

  /**
   * Escreve no console uma mensagem informativa
   * 
   * @param msg a mensagem
   */
  protected void logInfo(String msg) {
    if (verbose) {
      System.out.println(msg);
    }
  }

  /**
   * Faz a carga do entry point.
   */
  private void loadServerEntryPoint() {
    String serverPath =
      "rmi://" + this.getURI().getHostAndPort() + "/" + ServerEntryPoint.LOOKUP;
    try {
      this.logInfo(LNG.get("csbase.logic.ContactingServer") + this.serverURI);
      this.server = null;
      this.server = (ServerEntryPoint) Naming.lookup(serverPath);
      this.serverReachable.set(true);
      this.logInfo(MessageFormat.format(LNG.get("csbase.logic.ServerReached"),
		  new Object[] { this.serverURI } ));
    }
    catch (UnmarshalException e) {
      this.logError(LNG.get("csbase.logic.IncompatibleVersion"), e);
    }
    catch (RemoteException e) {
      this.logInfo(MessageFormat.format(LNG.get("csbase.logic.OfflineServer"),
    		  new Object[] { this.serverURI }));
    }
    catch (NotBoundException e) { // Quando o servidor CSBase cai.
      this.logError(LNG.get("csbase.logic.MissingEntryPoint")
        + this.serverURI, e);
    }
    catch (MalformedURLException e) {
      this.logError(MessageFormat.format(LNG.get(
    		  "csbase.logic.InvalidURL"), serverPath), e);
    }
  }

  /**
   * Executa um lookup para obter a referncia do ponto de entrada do servidor.
   * 
   * @return true se lookup bem sucedido ou false caso contrrio.
   */
  public final boolean lookup() {
    try {
      if (!this.isReachable() && !this.getServerLookupThread().isAlive()) {
        this.createLookupthread().start();
      }
      this.getServerLookupThread().join();
      if (this.server == null) {
        this.serverReachable.set(false);
        return false;
      }
      this.serverReachable.set(true);
    }
    catch (InterruptedException e1) {
      logWarning(LNG.get("csbase.logic.InterruptedLookupThread"));
    }
    return true;
  }

  /**
   * <p>
   * Valida se consegue fazer contato com o servidor.
   * </p>
   * <p>
   * Retorna {@code true} Se for possvel contactar o servidor. Caso contrrio,
   * se antes o servidor estava inalcanavel, notifica seus ouvintes da mudana
   * de estado. Tenta contactar o servidor novamente. Se for possvel, notifica
   * os ouvintes e retorna {@code true}, caso contrrio retorna {@code false}.
   * </p>
   * 
   * @return {@code true} se foi possvel contactar o servidor.
   */
  protected boolean tryReaching() {
    if (!ping()) {
      if (serverReachable.compareAndSet(true, false)) {
        this.notifyServerUnreachable();
      }
      if (lookup()) {
        this.notifyServerReachable();
        return true;
      }
      return false;
    }
    return true;
  }

  /**
   * Fora uma verificao assncrona do estado do servidor.
   */
  public final synchronized void invalidate() {
    /** Apenas interrompe a espera do monitor. */
    notifyAll();
  }

  /**
   * Testa se o servidor est alcanvel.
   * 
   * @return true, caso o servidor esteja alcanvel, ou false, caso contrrio.
   */
  public final boolean ping() {
    if (this.server == null || !this.isReachable()) {
      return false;
    }
    try {
      this.server.ping();
      return true;
    }
    catch (RemoteException e) {
      this.server = null;
      this.invalidate();
    }
    return false;
  }

  /**
   * Indica se o servidor est alcanvel.
   * 
   * @return true, caso o servidor esteja alcanvel, ou false, caso contrrio.
   */
  public final boolean isReachable() {
    return this.serverReachable.get();
  }

  /**
   * Inicia a thread de monitorao do servidor.
   */
  public final void startMonitoring() {
    if (this.mustContinue.get()) {
      return;
    }
    if (this.monitor == null || !this.monitor.isAlive()) {
      this.mustContinue.set(true);
      this.createMonitorThread();
      this.monitor.start();
    }
  }

  /**
   * Interrompe o monitor.
   */
  public final void stopMonitoring() {
    this.mustContinue.set(false);
    this.serverReachable.set(false);
    invalidate();
  }

  /**
   * Indica se a monitorao est sendo executada
   * 
   * @return true se a thread de monitorao est ativa, false caso contrrio
   */
  public final boolean isMonitoring() {
    return this.mustContinue.get();
  }

  /**
   * Notifica os ouvintes que o servidor est inalcanvel
   */
  private final void notifyServerUnreachable() {
    for (ServerMonitorListener listener : listeners) {
      listener.notifyServerUnreachable(this.serverURI);
    }
  }

  /**
   * Notifica os ouvintes que o servidor est alcanvel
   */
  private final void notifyServerReachable() {
    for (ServerMonitorListener listener : listeners) {
      listener.notifyServerReachable(this.serverURI);
    }
  }

  /**
   * Adiciona o listener a lista de listeners que so notificados quando a
   * conexo  perdida ou restabelecida.
   * 
   * @param listener Listener adicionado na lista
   */
  public final void addListener(ServerMonitorListener listener) {
    if (!listeners.contains(listener)) {
      listeners.add(listener);
    }
  }

  /**
   * Remove o listener da lista de listeners que so notificados quando a
   * conexo  perdida ou restabelecida.
   * 
   * @param listener Listener a ser removido na lista
   */
  public final void deleteListener(ServerMonitorListener listener) {
    listeners.remove(listener);
  }

  /**
   * Retorna a URI do servidor que est sendo monitorado.
   * 
   * @return O nome do servidor.
   */
  public final ServerURI getURI() {
    return this.serverURI;
  }

  /**
   * Retorna a thread de lookup com o servidor.
   * 
   * @return A thread de lookup
   */
  public final Thread getServerLookupThread() {
    return this.serverLookupThread;
  }

  /**
   * @return O ponto de entrada do servidor ou null caso esteja fora do ar
   */
  public final ServerEntryPoint getServer() {
    if (!this.isReachable()) {
      return null;
    }
    return this.server;
  }

  /**
   * Cria a thread de lookup com o servidor
   * 
   * @return A instncia da thread de lookup
   */
  private Thread createLookupthread() {
    serverLookupThread = new Thread() {
      @Override
      public void run() {
        loadServerEntryPoint();
      }
    };
    return serverLookupThread;
  }

  /**
   * Define se mensagens informativas devem ser exibidas.
   * 
   * @param verbose <code>true</code> se informaes devem ser exibidas,
   *        <code>false</code> se apenas erros e alertas devem ser exibidos
   */
  public void setVerbose(boolean verbose) {
    this.verbose = verbose;
  }
}