package csbase.logic.algorithms.flows;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import csbase.logic.algorithms.parameters.FileURLValue;

/**
 * <p>
 * Fluxo de algoritmos
 * </p>
 *
 * @author lmoreira
 */
public final class Flow implements Serializable, Cloneable {
  /**
   * O nome do fluxo.
   */
  private String name;

  /**
   * A descrio do fluxo.
   */
  private String description;

  /**
   * As conexes.
   */
  private Set<FlowLink> links;

  /**
   * Os ns.
   */
  private Set<FlowNode> nodes;

  /**
   * Cria um fluxo de algoritmos.
   *
   * @param name O nome do fluxo (Aceita {@code null}).
   * @param description A descrio do fluxo (Aceita {@code null}).
   * @param nodes Os ns (No aceita {@code null}).
   * @param links As conexes (No aceita {@code null}).
   */
  public Flow(String name, String description, Set<FlowNode> nodes,
    Set<FlowLink> links) {
    this.name = name;
    this.description = description;
    setLinks(links);
    setNodes(nodes);
  }

  /**
   * Formata o nome do arquivo levando em considerao o identificador do n do
   * fluxo. Ex.: [path][filename].[ext] -> [path][finename][nodeId].[ext].
   *
   * @param file caminho para o arquivo
   * @param nodeId identificador do n
   * @return caminho para o arquivo com o identificador do n
   */
  private String formatFilePathForNode(String file, int nodeId) {
    String extension = FileUtils.getFileExtension(file);
    String filePath = FileUtils.getFilePath(file);
    String formattedFilename;
    if (extension != null) {
      String fileWithExtension = FileUtils.getFileName(file);
      String filename =
        fileWithExtension.substring(0,
          fileWithExtension.lastIndexOf(extension) - 1);
      formattedFilename =
        String.format("%s/%s%s.%s", filePath, filename, nodeId, extension);
    }
    else {
      formattedFilename = String.format("%s%s", file, nodeId);
    }
    return formattedFilename;
  }

  /**
   * <p>
   * Atribui o valor do arquivo que deve receber a sada padro da execuo de
   * um algoritmo representado por este configurador.
   * </p>
   * <p>
   * Este valor ser repassado para todos os ns deste fluxo.
   * </p>
   *
   * @param standardOutputFile arquivo de sada padro
   */
  public void setStandardOutputFile(FileURLValue standardOutputFile) {

    String stdOutPath = standardOutputFile.getPath();
    String type = standardOutputFile.getType();
    for (FlowNode node : nodes) {
      int nodeId = node.getId();
      String nodeStdOutPath = formatFilePathForNode(stdOutPath, nodeId);
      FileURLValue nodeStdOut = new FileURLValue(nodeStdOutPath, type);

      node.setStandardOutputFile(nodeStdOut);
    }
  }

  /**
   * <p>
   * Obtm um conjunto imutvel contendo os arquivos de sada padro de cada n
   * que sero utilizadas durante a execuo de um fluxo.
   * </p>
   *
   * @return um conjunto imutvel contendo os arquivos de sada padro de cada
   * n que sero utilizadas durante a execuo de um fluxo.
   */
  public Set<FileURLValue> getStandardOutputFiles() {
    Set<FileURLValue> stdouts = new HashSet<FileURLValue>();
    for (FlowNode node : nodes) {
      if (!node.isBypassed() && null != node.getStandardOutputFile()) {
        stdouts.add(node.getStandardOutputFile());
      }
    }
    return Collections.unmodifiableSet(stdouts);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Flow clone() {
    try {
      Flow clone = (Flow) super.clone();
      clone.name = this.name;
      clone.description = this.description;
      clone.links = new HashSet<FlowLink>();
      for (FlowLink flowLink : this.links) {
        clone.links.add(flowLink.clone());
      }
      clone.nodes = new HashSet<FlowNode>();
      for (FlowNode flowNode : this.nodes) {
        clone.nodes.add(flowNode.clone());
      }
      return clone;
    }
    catch (CloneNotSupportedException e) {
      throw new IllegalStateException(e);
    }
  }

  /**
   * <p>
   * Obtm as conexes deste fluxo.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return As conexes, caso no haja conexes, retorna um conjunto vazio.
   */
  public Set<FlowLink> getLinks() {
    return Collections.unmodifiableSet(this.links);
  }

  /**
   * Obtm o nome do fluxo.
   *
   * @return o nome do fluxo ou {@code null} se ele no tiver nome.
   */
  public String getName() {
    return name;
  }

  /**
   * Obtm um n dado o identificador do n.
   *
   * @param nodeId O identificador do n.
   * @return O n ou {@code null} se ele no existir.
   */
  public FlowNode getNode(int nodeId) {
    for (FlowNode node : nodes) {
      if (node.getId() == nodeId) {
        return node;
      }
    }
    return null;
  }

  /**
   * <p>
   * Obtm os ns deste fluxo.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return As conexes, caso no haja ns, retorna um conjunto vazio.
   */
  public Set<FlowNode> getNodes() {
    return Collections.unmodifiableSet(this.nodes);
  }

  /**
   * <p>
   * Atribui conexes a este fluxo.
   * </p>
   *
   * @param links O conjunto de conexes (No aceita {@code null}).
   */
  private void setLinks(Set<FlowLink> links) {
    if (links == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "links"));
    }
    this.links = new HashSet<FlowLink>(links);
  }

  /**
   * <p>
   * Atribui um nome a este fluxo.
   * </p>
   *
   * @param name O nome do fluxo (No aceita {@code null}).
   */
  public void setName(String name) {
    if (name == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "name"));
    }
    this.name = name;
  }

  /**
   * <p>
   * Atribui ns a este fluxo.
   * </p>
   *
   * @param nodes O conjunto de ns (No aceita {@code null}).
   */
  private void setNodes(Set<FlowNode> nodes) {
    if (nodes == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "nodes"));
    }
    this.nodes = new HashSet<FlowNode>(nodes);
  }

  /**
   * <p>
   * Atribui o valor do arquivo que deve receber o cdigo de sada de execuo
   * de um algoritmo representado por este configurador.
   * </p>
   * <p>
   * Este valor ser repassado para todos os ns deste fluxo.
   * </p>
   *
   * @param exitCodeLogFile arquivo com o cdigo de sada.
   */
  public void setExitCodeLogFile(FileURLValue exitCodeLogFile) {
    String exitCodeFilePath = exitCodeLogFile.getPath();
    String type = exitCodeLogFile.getType();
    for (FlowNode node : nodes) {
      int nodeId = node.getId();
      String nodeExitCodeFilePath =
        formatFilePathForNode(exitCodeFilePath, nodeId);
      FileURLValue nodeExitCodeFile =
        new FileURLValue(nodeExitCodeFilePath, type);
      if (node.hasExitCode()) {
        node.setExitCodeLogFile(nodeExitCodeFile);
      }
    }
  }

  /**
   * <p>
   * Atribui o valor do arquivo que deve receber alertas de execuo
   * de um algoritmo representado por este configurador.
   * </p>
   * <p>
   * Este valor ser repassado para todos os ns deste fluxo.
   * </p>
   *
   * @param warningsFile arquivo de alertas.
   */
  public void setWarningsFile(FileURLValue warningsFile) {
    String warningsFilePath = warningsFile.getPath();
    String type = warningsFile.getType();
    for (FlowNode node : nodes) {
      int nodeId = node.getId();
      String nodeWarningsFilePath =
        formatFilePathForNode(warningsFilePath, nodeId);
      FileURLValue nodeWarningsFile =
        new FileURLValue(nodeWarningsFilePath, type);
      node.setWarningsFile(nodeWarningsFile);
    }
  }

  /**
   * <p>
   * Obtm um conjunto imutvel contendo os arquivos de alerta de execuo de
   * cada n durante a execuo de um fluxo.
   * </p>
   *
   * @return um conjunto imutvel contendo os arquivos de alerta dos ns do
   * fluxo.
   */
  public Set<FileURLValue> getWarningsFiles() {
    Set<FileURLValue> files = new HashSet<>();
    for (FlowNode node : nodes) {
      if (!node.isBypassed() && null != node.getWarningsFile()) {
        files.add(node.getWarningsFile());
      }
    }
    return Collections.unmodifiableSet(files);
  }

  /**
   * Retorna a descrio do fluxo.
   *
   * @return description a descrio.
   */
  public String getDescription() {
    return description;
  }

  /**
   * Atribui a descrio do fluxo.
   *
   * @param description a descrio.
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * <p>
   * Obtm um conjunto imutvel contendo que devem receber o cdigo de sada de
   * execuo de cada n durante a execuo de um fluxo.
   * </p>
   *
   * @return um conjunto imutvel contendo os arquivos de cdigo de sada dos
   * ns do fluxo.
   */
  public Set<FileURLValue> getExitCodeLogFiles() {
    Set<FileURLValue> exitCodeFiles = new HashSet<FileURLValue>();
    for (FlowNode node : nodes) {
      if (node.hasExitCode() && node.getExitCodeLogFile() != null) {
        exitCodeFiles.add(node.getExitCodeLogFile());
      }
    }
    return Collections.unmodifiableSet(exitCodeFiles);
  }

  /**
   * Indica se pelo menos um n do fluxo prov cdigo de sada durante sua
   * execuo.
   *
   * @return Verdadeiro se pelo menos um n prov cdigo de sada, ou falso,
   * caso contrrio.
   */
  public boolean hasExitCode() {
    for (FlowNode node : nodes) {
      if (node.hasExitCode()) {
        return true;
      }
    }
    return false;
  }
}
