package csbase.console;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

/**
 * Tabela com informaes no formato "chave : valor". Os valores so alinhados
 * de acordo com a maior chave, e as chaves podem ser exibidas em ordem
 * alfabtica ou por ordem de entrada.
 * <p>
 * A tabela pode ser associada a um status indicando erro; neste caso isto ser
 * sinalizado ao lado do seu cabealho.
 * 
 * @author Tecgraf
 */
class InfoTable {
  /**
   * Status da tabela (representa a validade dos dados como um todo).
   */
  private boolean isOK = true;
  /**
   * Cabealho da tabela.
   */
  private String header;
  /**
   * Mapa que armazena os pares (chave, valor).
   */
  private Map<String, Object> map;

  /**
   * Construtor.
   * 
   * @param header cabealho
   * @param sorted <code>true</code> para exibir os dados ordenados pelas
   *        chaves, <code>false</code> para exibi-los pela ordem de entrada
   */
  InfoTable(String header, boolean sorted) {
    this.header = header;
    if (sorted) {
      map = new TreeMap<String, Object>();
    }
    else {
      map = new LinkedHashMap<String, Object>();
    }
  }

  /**
   * Adiciona um novo par (chave, valor)  tabela.
   * 
   * @param label chave
   * @param value valor
   */
  void add(String label, Object value) {
    map.put(label, value);
  }

  /**
   * Adiciona um novo par (chave, valor)  tabela, opcionalmente invalidando a
   * mesma.
   * 
   * @param label chave
   * @param value valor
   * @param isOK <code>false</code> para invalidar a tabela
   */
  void add(String label, Object value, boolean isOK) {
    add(label, value);
    setOK(isOK);
  }

  /**
   * Altera o status da tabela.
   * 
   * @param isOK <code>true</code> para manter o status corrente,
   *        <code>false</code> para invalidar a tabela (este valor  combinado
   *        via AND com o valor corrente)
   */
  void setOK(boolean isOK) {
    this.isOK &= isOK;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append(String.format("%-50s%s\n", header + ':', isOK ? ""
      : "** ERRO **"));
    int maxLen = Diagnostics.getMaxStrLen(map.keySet());
    String format = Diagnostics.IDENT + "%-" + maxLen + "s : %s\n";
    for (Entry<String, Object> entry : map.entrySet()) {
      builder.append(String.format(format, entry.getKey(), entry.getValue()));
    }
    return builder.toString();
  }

  /**
   * Obtm o status da tabela.
   * 
   * @return <code>true</code> se todos os dados esto vlidos,
   *         <code>false</code> se pelo menos um  invlido
   */
  public boolean isOK() {
    return isOK;
  }
}