package csbase.rest.adapter.drmaa2.v1;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;
import java.util.concurrent.Executors;

import csbase.logic.CommandInfo;
import csbase.logic.User;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.sgaservice.SGAService;
import ibase.rest.api.drmaa2.v1.adapter.JobDAO;
import ibase.rest.api.drmaa2.v1.adapter.JobMonitorListener;
import ibase.rest.model.drmaa2.v1.Job;

/**
 * Monitora alteraes de informaes de comando (CommandInfo).
 */
public class JobInfoMonitor {
  private JobDAO jobDAO = null;
  private Collection<JobMonitorListener> jobsListener =
    Collections.synchronizedList(new ArrayList<JobMonitorListener>());
  private final Object listenerLock = new Object();

  /**
   * Construtor
   *
   * @param jobDAO DAO de job
   */
  public JobInfoMonitor(JobDAO jobDAO) {
    this.jobDAO = jobDAO;
    Executors.newSingleThreadExecutor().execute(UpdateJobInfoThread);
  }

  /**
   * Adiciona um listener
   *
   * @param listener o listener
   */
  public void addJobMonitorListener(JobMonitorListener listener) {
    synchronized (listenerLock) {
      jobsListener.add(listener);
      listenerLock.notify();
    }
  }

  /**
   * Remove um listener
   *
   * @param listener o listener
   */
  public void removeJobMonitorListener(JobMonitorListener listener) {
    synchronized (listenerLock) {
      jobsListener.remove(listener);
    }
  }

  /**
   * Thread de aquisio das informaes de todos os comandos em execuo.
   * Somentes as novas informaes -- comparadas com as informaes no banco --
   * so notificadass aos listeners.
   */
  private Runnable UpdateJobInfoThread = new Runnable() {
    @Override
    public void run() {
      while (true) {
        synchronized (listenerLock) {
          while (jobsListener.isEmpty()) {
            try {
              listenerLock.wait();
            } catch (InterruptedException e) {
              Server.logSevereMessage("Erro no wait", e);
            }
          }
        }

        Service.setUserId(User.getAdminId());
        Vector<CommandInfo> infos = SGAService.getInstance().getAllSGACommands();
        List<Job> jobs = new LinkedList<>();
        for (CommandInfo info : infos) {
          if (info.getProgressData() != null) {
            Job job = jobDAO.findJobById(info.getId());
            if (job != null &&
              !info.getProgressData().getDescription().equals(job.getProgressInfo())) {
              job.setProgressInfo(info.getProgressData().getDescription());
              job.setLastModifiedTime(LocalDateTime.now().toString());
              jobDAO.updateJob(job);
              jobs.add(job);
            }
          }
        }


        if (!jobs.isEmpty()) {

          List<JobMonitorListener> copy = new ArrayList<>();
          synchronized (listenerLock) {
            copy.addAll(jobsListener);
          }

          // Faz uma cpia da lista de listeners, pois os prprios listeners
          // se retiram da lista, causando a exceo ConcurrentModificationException
          for (JobMonitorListener l : copy) {
            try {
              l.infoChanged(jobs);
            }
            catch (Exception e) {
              Server.logSevereMessage("Erro na notificao do infoChanged", e);
            }

          }
        }

        try {
          Thread.sleep(1000);
        }
        catch (Exception ignored) {
          //faz nada, pois a thread nao pode parar
        }
      }
    }
  };
}
