package csbase.server.services.filetransferservice;

import java.util.ArrayList;
import java.util.List;

import csbase.logic.User;
import csbase.logic.filetransferservice.FileTransferConnection;
import csbase.logic.filetransferservice.FileTransferRequest;

/**
 * Classe que modela as lista de requisies em andamento.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class FileTransferRequestQueue {

  /**
   * Lista das requisies
   */
  final private List<FileTransferRequest> waitList =
    new ArrayList<FileTransferRequest>();

  /**
   * Lista das requisies
   */
  final private List<FileTransferRequest> runList =
    new ArrayList<FileTransferRequest>();

  /**
   * Lista das requisies
   */
  final private List<FileTransferRequest> endList =
    new ArrayList<FileTransferRequest>();

  /**
   * Adiciona lista de entrada na lista de sada verificando usurio da conexo.
   * 
   * @param user usurio
   * @param inList lista de entrada
   * @param outList lista de sada
   */
  private void addList(final User user, final List<FileTransferRequest> inList,
    final List<FileTransferRequest> outList) {
    for (final FileTransferRequest request : inList) {
      if (user == null) {
        outList.add(request);
      }
      else {
        final FileTransferConnection connection = request.getConnection();
        final Object requestUserId = connection.getUserId();
        final Object userId = user.getId();
        if (userId.equals(requestUserId)) {
          outList.add(request);
        }
      }
    }
  }

  /**
   * Retorna a requisio com bas no id.
   * 
   * @param id id
   * @return a requisio.
   */
  synchronized FileTransferRequest getRequestFromId(final String id) {
    for (final FileTransferRequest rq : waitList) {
      if (rq.getId().equals(id)) {
        return rq;
      }
    }
    for (final FileTransferRequest rq : runList) {
      if (rq.getId().equals(id)) {
        return rq;
      }
    }
    for (final FileTransferRequest rq : endList) {
      if (rq.getId().equals(id)) {
        return rq;
      }
    }

    return null;
  }

  /**
   * @param user usurio
   * @return a lista
   */
  synchronized List<FileTransferRequest> getAllRequests(final User user) {
    final List<FileTransferRequest> list = new ArrayList<FileTransferRequest>();
    addList(user, waitList, list);
    addList(user, runList, list);
    addList(user, endList, list);
    return list;
  }

  /**
   * Colocao de request em fila.
   * 
   * @param request request.
   */
  synchronized void putToWaitList(final FileTransferRequest request) {
    waitList.add(request);
  }

  /**
   * Colocao de request em iniciao.
   * 
   * @param requestId id do request.
   */
  synchronized void moveFromWaitToRunList(final String requestId) {
    final FileTransferRequest request = delFromList(waitList, requestId);
    if (request != null) {
      waitList.remove(request);
      runList.add(request);
    }
  }

  /**
   * Colocao de request em ended.
   * 
   * @param requestId id do request.
   */
  synchronized void moveToEndedList(final String requestId) {
    FileTransferRequest request = null;

    request = delFromList(waitList, requestId);
    if (request != null) {
      waitList.remove(request);
      endList.add(request);
      return;
    }

    request = delFromList(runList, requestId);
    if (request != null) {
      runList.remove(request);
      endList.add(request);
      return;
    }
  }

  /**
   * Retira da lista (se estiver presente).
   * 
   * @param list a lista.
   * @param requestId id da requisio
   * @return a requisio.
   */
  private FileTransferRequest delFromList(final List<FileTransferRequest> list,
    final String requestId) {
    FileTransferRequest toRemove = null;
    for (final FileTransferRequest rq : list) {
      if (rq.getId().equals(requestId)) {
        toRemove = rq;
        break;
      }
    }
    if (toRemove != null && list.contains(toRemove)) {
      list.remove(toRemove);
      return toRemove;
    }
    return null;
  }

  /**
   * Retorna a quantidade de requisies em andamento.
   * 
   * @return a quantidade.
   */
  synchronized int getNumRunningRequests() {
    return runList.size();
  }

  /**
   * Consulta a prxima requisio pendente.
   * 
   * @return a requisio ou {@code null}
   */
  synchronized FileTransferRequest pickWaitingRequest() {
    if (waitList.size() > 0) {
      return waitList.get(0);
    }
    return null;
  }

  /**
   * Retirada de request em fila.
   * 
   * @param request request.
   */
  synchronized void removeFromAllLists(final FileTransferRequest request) {
    final String requestId = request.getId();
    delFromList(runList, requestId);
    delFromList(waitList, requestId);
    delFromList(endList, requestId);
  }
}
