/*
 * Detalhes da ltima alterao:
 * 
 * $Author: clinio $ $Date: 2010-09-06 09:41:01 -0300 (Mon, 06 Sep 2010) $
 * $Revision: 109841 $
 */
package csbase.server.services.loginservice;

import java.io.Serializable;
import java.rmi.Remote;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import csbase.logic.User;

/**
 * Representa a sesso de um usurio, armazenada no servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ServerSession {
  /**
   * O usurio.
   */
  private User user;
  /**
   * O <i>locale</i> do usurio.
   */
  private Locale locale;
  /**
   * O <i>timezone</i> do usurio.
   */
  private TimeZone timeZone;
  /**
   * O nome da aplicao cliente utilizada pelo usurio.
   */
  private String app;
  /**
   * Propriedades adicionais da sesso.
   */
  private Map<String, Serializable> properties;

  /**
   * O controlador remoto.
   */
  private Remote control;

  /**
   * Cria uma sesso de usurio que ser armazenada no servidor.
   * 
   * @param user O usurio.
   * @param locale O <i>locale</i> do usurio.
   */
  ServerSession(User user, Locale locale) {
    this.user = user;
    this.locale = locale;
    this.timeZone = TimeZone.getDefault();
    this.properties =
      Collections.synchronizedMap(new HashMap<String, Serializable>());
  }

  /**
   * Obtm o usurio.
   * 
   * @return O usurio.
   */
  public User getUser() {
    return this.user;
  }

  /**
   * Define o usurio.
   * 
   * @param user O usurio.
   */
  public void setUser(User user) {
    this.user = user;
  }

  /**
   * Obtm o <i>locale</i> do usurio.
   * 
   * @return O <i>locale</i> do usurio.
   */
  public Locale getLocale() {
    return this.locale;
  }

  /**
   * Obtm o <i>timezone</i> do usurio.
   * 
   * @return O <i>timezone</i> do usurio.
   */
  public TimeZone getTimeZone() {
    return this.timeZone;
  }

  /**
   * Define o <i>timezone</i> do usurio.
   * 
   * @param timeZone O <i>timezone</i> do usurio.
   */
  public void setTimeZone(TimeZone timeZone) {
    this.timeZone = timeZone;
  }

  /**
   * Obtm o nome da aplicao cliente utilizada pelo usurio.
   * 
   * @return O nome da aplicao cliente utilizada pelo usurio.
   */
  public String getApp() {
    return this.app;
  }

  /**
   * Define o nome da aplicao cliente utilizada pelo usurio.
   * 
   * @param app O nome da aplicaes cliente utilizada pelo usurio.
   */
  public void setApp(String app) {
    this.app = app;
  }

  /**
   * Obtm uma propriedade da sesso.
   * 
   * @param propertyName O nome da propriedade.
   * 
   * @return A propriedade da sesso.
   */
  public Object getProperty(String propertyName) {
    return this.properties.get(propertyName);
  }

  /**
   * Cria ou altera uma propriedade da sesso.
   * 
   * @param propertyName O nome da propriedade.
   * @param propertyValue O valor da propriedade
   */
  public void setProperty(String propertyName, Serializable propertyValue) {
    this.properties.put(propertyName, propertyValue);
  }

  /**
   * Remove uma propriedade da sesso.
   * 
   * @param propertyName O nome da propriedade.
   */
  public void removeProperty(String propertyName) {
    this.properties.remove(propertyName);
  }

  /**
   * Adiciona todos as propriedades recebidas na sesso.
   * 
   * @param newProperties As propriedades.
   */
  public void putAllProperties(Map<String, Serializable> newProperties) {
    this.properties.putAll(newProperties);
  }

  /**
   * Cosulta o mapa de propriedades.
   * 
   * @return o mapa.
   */
  public Map<String, Serializable> getPropertes() {
    return this.properties;
  }

  /**
   * Cosnulta do controle.
   * 
   * @return o controle
   */
  public Remote getControl() {
    return this.control;
  }

  /**
   * Ajuste do controle.
   * 
   * @param control o controle
   */
  public void setControl(Remote control) {
    this.control = control;
  }

}