package csbase.server.services.opendreamsservice.opendreams.v1_04;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;

import tecgraf.openbus.DRMAA.ExitTimeoutException;
import tecgraf.openbus.DRMAA.InvalidJobException;
import tecgraf.openbus.DRMAA.JobInfo;
import tecgraf.openbus.opendreams.FinalizationType;
import tecgraf.openbus.opendreams.OpenDreamsJobInfo;
import tecgraf.openbus.opendreams.OpenDreamsJobInfoImpl;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandNotification;
import csbase.logic.CommandSuccessNotification;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.server.Server;

/**
 * Fila de espera pelos jobs em execuo.
 * 
 * @author Tecgraf PUC-Rio
 */
public class JobInfoQueue {

  /**
   * O Job terminou mas no tem um codigo de sada
   */
  public final static int NO_EXIT_CODE = 256; // Acima de 255 no  um exit code vlido

  /**
   * Tempo em milisegundos para reavaliar se a fila j possui informao sobre o
   * trmino do job.
   */
  public static final long WAITING_TIMEOUT = 100;

  /**
   * Informao de trmino para cada job submetido. Diferentes threads podem
   * aguardar a espera por um mesmo comando, porm apenas uma delas recebe a
   * informao de trmino; as outras threads recebem uma exceo. Esse mapa
   * relaciona o identificador do comando submetido com a infornao de trmino
   * de job.
   */
  private Map<String, OpenDreamsJobInfo> jobsSubmited =
    new Hashtable<String, OpenDreamsJobInfo>();

  /**
   * Indica que um job foi submetido
   * 
   * @param jobId o identificador do job submetido
   */
  public synchronized void jobSubmited(String jobId) {
    OpenDreamsJobInfo jobInfo = jobsSubmited.get(jobId);
    if (jobInfo == null) {
      jobInfo = new OpenDreamsJobInfoImpl();
      jobInfo.jobId = jobId;
      jobsSubmited.put(jobId, jobInfo);
    }
  }

  /**
   * Indica que um job finalizou
   * 
   * @param cmdEvent evento com as informaes de trmino do job
   */
  public synchronized void jobExit(CommandNotification cmdEvent) {
    String jobId = cmdEvent.getCommandId().toString();
    OpenDreamsJobInfo jobInfo = jobsSubmited.get(jobId);
    if (jobInfo == null) {
      jobInfo = new OpenDreamsJobInfoImpl();
      jobInfo.jobId = jobId;
      jobsSubmited.put(jobId, jobInfo);
    }
    List<String[]> resourceUsage = new ArrayList<String[]>();
    if (cmdEvent.getFinalizationInfo().getFinalizationType() == CommandFinalizationType.SUCCESS) {
      CommandSuccessNotification endNotification =
        (CommandSuccessNotification) cmdEvent;
      if (endNotification.getCPUTime() != null) {
        resourceUsage.add(new String[] { "cpu_time",
            endNotification.getCPUTime().toString() });
      }
      if (endNotification.getElapsedTime() != null) {
        resourceUsage.add(new String[] { "elapsed_time",
            endNotification.getElapsedTime().toString() });
      }
      if (endNotification.getUserTime() != null) {
        resourceUsage.add(new String[] { "user_time",
            endNotification.getUserTime().toString() });
      }
    }
    if (resourceUsage.size() > 0) {
      jobInfo.resourceUsage =
        resourceUsage.toArray(new String[resourceUsage.size()][]);
    }
    if (cmdEvent.getFinalizationInfo().getInfoType() == FinalizationInfoType.EXTENDED) {
      ExtendedCommandFinalizationInfo finalizationInfo =
        (ExtendedCommandFinalizationInfo) cmdEvent.getFinalizationInfo();
      jobInfo.guiltyNodeId = String.valueOf(finalizationInfo.getGuiltyNodeId());
    }
    jobInfo.hasExited = true;
    CommandFinalizationInfo finalizationInfo = cmdEvent.getFinalizationInfo();
    if (finalizationInfo == null) {
      Server.logSevereMessage("A informao de finalizacao do comando " + jobId
        + " no existe (null)");
      jobInfo.exitStatus = NO_EXIT_CODE;
      jobInfo.finalizationType = FinalizationType.NO_FINALIZATION_INFO;
    }
    else {
      Integer exitCode = finalizationInfo.getExitCode();
      Server.logInfoMessage("A informao de finalizacao do comando " + jobId
        + " possui exitCode=" + exitCode);
      jobInfo.exitStatus = exitCode == null ? NO_EXIT_CODE : exitCode;
      jobInfo.finalizationType =
        getFinalizationType(finalizationInfo.getFinalizationType());
    }
    jobInfo.wasAborted =
      cmdEvent.getFinalizationInfo().getFinalizationType().equals(
        FinalizationType.KILLED);
    jobInfo.hasSignaled = false;
    jobInfo.terminatingSignal = "";
    jobInfo.hasCoreDump = false;
    notifyAll();

  }

  /**
   * Mapeia o tipo de finalizao do comando do CSBase para o tipo de
   * finalizao de comando definido pela IDL do OpenDreams.
   * 
   * @param type o tipo de finalizao do comando no CSBase
   * @return o tipo de finalizao do comando mapeado para o OpenDreams
   */
  private FinalizationType getFinalizationType(CommandFinalizationType type) {
    switch (type) {
      case NOT_FINISHED:
        return FinalizationType.NOT_FINISHED;
      case UNKNOWN:
        return FinalizationType.UNKNOWN;
      case END:
        return FinalizationType.END;
      case SUCCESS:
        return FinalizationType.SUCCESS;
      case EXECUTION_ERROR:
        return FinalizationType.EXECUTION_ERROR;
      case FAILED:
        return FinalizationType.FAILED;
      case KILLED:
        return FinalizationType.KILLED;
      case LOST:
        return FinalizationType.LOST;
      case NO_EXIT_CODE:
        return FinalizationType.NO_EXIT_CODE;
      default:
        return FinalizationType.UNDEFINED;
    }
  }

  /**
   * Aguarda que um determinado job submetido termine sua execuo.
   * 
   * Se mais de uma thread estiver aguardando pelo mesmo job, apenas uma delas
   * recebe a informao de trmino do job. As outras threads recebem e exceo
   * <code>InvalidJobException</code>. Essa mesma exceo  tambm lanada se o
   * identificador do job no for encontrado entre os jobs submetidos nessa
   * sesso.
   * 
   * No caso do timeout ter esgotado sem que o job tenha terminado, a exceo
   * <code>ExitTimeoutException</code>  lanada. Nesse caso, a informao sobre
   * o job permanece disponvel para uma outra requisio.
   * 
   * @param jobId o nome do job monitorado
   * @param timeout o tempo de espera limite em segundos
   * @return a informao sobre o job submetido
   * @throws InvalidJobException se o identificador do job no foi encontrado ou
   *         se outra requisio j recebeu a informao de trmino do job.
   * @throws ExitTimeoutException se o timeout esgotou e o job no terminou.
   */
  public JobInfo waitFor(String jobId, long timeout)
    throws InvalidJobException, ExitTimeoutException {
    synchronized (this) {
      /* Verifica se existe um job submetido com o identificador especificado */
      JobInfo info = jobsSubmited.get(jobId);
      if (info == null) {
        throw new InvalidJobException(
          "No foi encontrado um job com o nome especificado: " + jobId);
      }
      /* Se o tempo especificado foi TIMEOUT_NO_WAIT retorna o jobInfo */
      /* TODO mjulia Confirmar se o comportamento  esse mesmo */
      if (timeout == Session.TIMEOUT_NO_WAIT) {
        jobsSubmited.remove(jobId);
        return info;
      }
    }
    final long limit = System.currentTimeMillis() * 1000 + timeout;
    long timeToGo = limit - (System.currentTimeMillis() * 1000);
    boolean waitForever = (timeout == Session.TIMEOUT_WAIT_FOREVER);
    while (waitForever || timeToGo > 0) {
      try {
        synchronized (this) {
          wait(WAITING_TIMEOUT);
        }
        if (!waitForever) {
          timeToGo = limit - System.currentTimeMillis() * 1000;
        }
        synchronized (this) {
          JobInfo info = jobsSubmited.get(jobId);
          if (info == null) {
            throw new InvalidJobException(
              "Outra requisio j recebeu a informao de trmino desse"
                + "mesmo job: " + jobId);
          }
          if (info.hasExited) {
            jobsSubmited.remove(jobId);
            return info;
          }
        }
      }
      catch (InterruptedException e) {
      }
    }
    /* Quando o tempo esgotou, a informao sobre o job permanece guardada */
    throw new ExitTimeoutException("O timeout esgotou e o job no terminou: "
      + jobId);
  }

  /**
   * Aguarda que qualquer um dos jobs submetidos termine sua execuo. Quando um
   * deles terminar, retorna a informao sobre esse job.
   * 
   * Apenas uma thread recebe a informao de um determinado job. Se outras
   * threads estiverem aguardando e no houver mais jobs submetidos (no momento
   * em que a requisio de "wait" for feita), as outras recebem a exceo
   * <code>InvalidJobException</code>.
   * 
   * No caso do timeout ter esgotado sem que algum job tenha terminado, a
   * exceo <code>ExitTimeoutException</code>  lanada.
   * 
   * @param timeout o tempo de espera limite em segundos
   * @return a informao sobre o primeiro job que terminou
   * @throws InvalidJobException se no h mais jobs submetidos na sesso que
   *         ainda no tenham terminado sua execuo
   * @throws ExitTimeoutException se o timeout esgotou e nenhum job terminou.
   */
  public JobInfo waitForAnyJob(long timeout) throws ExitTimeoutException,
    InvalidJobException {
    final Set<String> jobsAvailable = jobsSubmited.keySet();
    synchronized (this) {
      if (jobsSubmited.size() == 0) {
        throw new InvalidJobException("No h mais jobs submetidos na sesso "
          + "que ainda no tenham terminado sua execuo");
      }
      /* Se o tempo especificado foi TIMEOUT_NO_WAIT retorna o jobInfo */
      if (timeout == Session.TIMEOUT_NO_WAIT) {
        for (String jobId : jobsAvailable) {
          JobInfo info = jobsSubmited.get(jobId);
          if (info.hasExited) {
            jobsSubmited.remove(jobId);
            return info;
          }
        }
        /* O tempo NO_WAIT esgotou e no h informao sobre trmino de jobs */
        /* TODO mjulia Confirmar se o comportamento  esse mesmo */
        throw new ExitTimeoutException("O timeout esgotou e o job no terminou");
      }
    }
    final long limit = System.currentTimeMillis() * 1000 + timeout;
    long timeToGo = limit - (System.currentTimeMillis() * 1000);
    boolean waitForever = (timeout == Session.TIMEOUT_WAIT_FOREVER);
    while (waitForever || timeToGo > 0) {
      try {
        synchronized (this) {
          wait(WAITING_TIMEOUT);
        }
        if (!waitForever) {
          timeToGo = limit - System.currentTimeMillis() * 1000;
        }
        synchronized (this) {
          JobInfo info = null;
          for (String jobId : jobsAvailable) {
            info = jobsSubmited.get(jobId);
            if (info.hasExited) {
              jobsSubmited.remove(jobId);
              return info;
            }
          }
          if (info == null) {
            throw new InvalidJobException("No h mais jobs submetidos na "
              + "sesso que ainda no tenham terminado sua execuo");
          }
        }
      }
      catch (InterruptedException e) {
      }
    }
    throw new ExitTimeoutException("O timeout esgotou e o job no terminou");
  }

  /**
   * Obtm as informaes sobre se o job submetido j finalizou sua execuo.
   * 
   * @param jobId o nome do job
   * @return o objeto {@link JobInfo} que possui as informaes sobre o job
   *         submetido
   * @throws InvalidJobException se o identificador do job no foi encontrado
   */
  public JobInfo getJobInfo(String jobId) throws InvalidJobException {
    synchronized (this) {
      JobInfo info = jobsSubmited.get(jobId);
      if (info == null) {
        throw new InvalidJobException(
          "No foi encontrado um job com o nome especificado: " + jobId);
      }
      return info;
    }
  }

}
