/*
 * $Id$
 */
package csbase.server.services.wioservice;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Calendar;

import org.omg.PortableServer.POA;

import csbase.exception.ServiceFailureException;
import csbase.logic.ClientProjectFile;
import csbase.logic.User;
import csbase.server.Server;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.wioservice.idl.BytesHolder;
import csbase.server.services.wioservice.idl.SeekType;
import csbase.server.services.wioservice.idl.WIODateTimeInfo;
import csbase.server.services.wioservice.idl.WIOFile;
import csbase.server.services.wioservice.idl.WIOFileHelper;
import csbase.server.services.wioservice.idl.WIOFileMode;
import csbase.server.services.wioservice.idl.WIOFilePOA;
import csbase.server.services.wioservice.idl.WIOProject;
import csbase.server.services.wioservice.idl.WIOProjectHelper;
import csbase.server.services.wioservice.idl.WIOServiceException;
import tecgraf.javautils.core.io.FileUtils;

/**
 * A classe <code>WIOServerFile</code> implementa o <i>servant</i> associado 
 * interface idl <code>WIOFile</code>.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class WIOServerFile extends WIOFilePOA {
  /**
   * Arquivo fsico no file system do servidor
   */
  private File javaFile = null;

  /**
   * Projeto relacionado ao arquivo/diretrio
   */
  private WIOServerProject wioServerProject = null;

  /**
   * Path relacionado ao arquivo/diretrio
   */
  private String serverPath = null;

  /**
   * Modo corrente do arquivo
   */
  private WIOFileMode fileMode = null;

  /**
   * Arquivo de acesso ao file system do servidor
   */
  private RandomAccessFile randomFile = null;

  /**
   * Mtodo de ativao do objeto CORBA WIOFile.
   * 
   * @return .
   * @throws Exception em caso de erro.
   */
  protected POA activateCorbaFile() throws Exception {
    final WIOService wioService = WIOService.getInstance();
    return wioService.activateCorbaObject(null, this);
  }

  /**
   * Adiciona um texto  descrio atual do arquivo.
   * 
   * @param description o texto a ser adicionado  descrio do arquivo.
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void appendDescription(final String description)
    throws WIOServiceException {
    if (wioServerProject == null) {
      throw new WIOServiceException("Projeto nulo em [" + this.toString() + "]");
    }
    try {
      wioServerProject.appendDescription(serverPath, description);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no appendDescription", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Constri a representao de uma data.
   * 
   * @param l a data a ser representada, como um nmero de milisegundos
   * @return a representao da data, como um objeto do tipo
   *         <code>WIODateTimeInfo</code>.
   * 
   */
  private WIODateTimeInfo buildDateTime(final long l) {
    final Calendar cal = Calendar.getInstance();
    cal.setTimeInMillis(l);
    final WIODateTimeInfo dateTime =
      new WIODateTimeInfo(cal.get(Calendar.DAY_OF_MONTH),
        cal.get(Calendar.MONTH) + 1, cal.get(Calendar.YEAR),
        cal.get(Calendar.HOUR_OF_DAY), cal.get(Calendar.MINUTE),
        cal.get(Calendar.SECOND));
    return dateTime;
  }

  /**
   * Verifica se o arquivo pode ser lido.
   * 
   * @return true se o arquivo pode ser lido, false caso contrrio
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public boolean canRead() throws WIOServiceException {
    if (javaFile == null) {
      throw new WIOServiceException("Arquivo fisico nulo!");
    }
    return javaFile.canRead();
  }

  /**
   * Verifica se o arquivo pode ser escrito.
   * 
   * @return true se o arquivo pode ser escrito, false caso contrrio
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public boolean canWrite() throws WIOServiceException {
    if (javaFile == null) {
      throw new WIOServiceException("Arquivo fisico nulo!");
    }
    return javaFile.canWrite();
  }

  /**
   * Fecha o arquivo fsico.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void close() throws WIOServiceException {
    if (fileMode == WIOFileMode.NOT_OPENED) {
      Server.logSevereMessage("Duplo-close detectado: " + serverPath);
      return;
    }
    try {
      if (fileMode == WIOFileMode.MODE_WRITE) {
        wioServerProject.setUnderConstruction(serverPath, false);
      }
      logInfo("Arquivo fechado: " + javaFile.getPath());
      logInfo("Agendando atualizao do arquivo: " + javaFile.getPath());
      final ProjectService projectService = ProjectService.getInstance();
      final Object loggedUserId = getMyUserId();
      if (loggedUserId == null) {
        throw new Exception("Sem usuario ajustado para o close");
      }
      ProjectService.setUserId(loggedUserId);
      final String uLogin = wioServerProject.getLoggedUserLogin();
      logInfo("Ativando atualizao (close) pelo usurio " + uLogin);
      final ClientProjectFile cpf = getClientProjectFile();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no close", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      fileMode = WIOFileMode.NOT_OPENED;
      closeRandomAccessFile();
      ProjectService.setUserId(null);
    }
  }

  /**
   * Fecha o arquivo de acesso ao sistema de arquivos.
   */
  private void closeRandomAccessFile() {
    if (randomFile != null) {
      try {
        randomFile.close();
      }
      catch (IOException e) {
        Server.logSevereMessage("Falha ao fechar o arquivo: ", e);
      } finally {
        randomFile = null;
      }
    }
  }

  /**
   * Obtm a data de criao do arquivo.
   * 
   * @return a data de criao do arquivo, representada por um objeto
   *         <code>WIODateTimeInfo</code>
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIODateTimeInfo creationTime() throws WIOServiceException {
    try {
      final ClientProjectFile cpf = getClientProjectFile();
      return buildDateTime(cpf.getCreationDate());
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no creationTime", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Mtodo de desativao do objeto CORBA WIOFile.
   * 
   * @throws Exception em caso de erro.
   */
  protected void deactivateCorbaFile() throws Exception {
    final WIOService wioService = WIOService.getInstance();
    wioService.deactivateCorbaObject(this);
  }

  /**
   * Desativa o objeto objeto remoto, liberando os recursos associados .
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void destroy() throws WIOServiceException {
    try {
      deactivateCorbaFile();
      wioServerProject.decNumFiles();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no destroy", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      fileMode = WIOFileMode.NOT_OPENED;
      closeRandomAccessFile();
    }
  }

  /**
   * Verifica se o arquivo est posicionado no fim.
   * 
   * @return true se arquivo posicionado no fim, false caso contrrio.
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public boolean eof() throws WIOServiceException {
    final String myName = "[" + this.toString() + "]";
    if (fileMode == WIOFileMode.NOT_OPENED) {
      final String err = "Falha de eof por arquivo fechado em " + myName;
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    if (randomFile == null) {
      final String err = "Falha de eof por random-file nulo em " + myName;
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    try {
      return randomFile.getFilePointer() >= randomFile.length();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no eof", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Retorna o ClientProjectFile que representa esse arquivo no projeto.
   * 
   * @return O ClientProjectFile que representa esse arquivo no projeto.
   */
  private ClientProjectFile getClientProjectFile() {
    final ProjectService projectService = ProjectService.getInstance();
    final Object loggedUserId = getMyUserId();
    ProjectService.setUserId(loggedUserId);

    final Object projectId = getMyProjectId();
    final String[] filePath = FileUtils.splitPath(serverPath);
    final String[] dirPath = new String[filePath.length - 1];
    for (int i = 0; i < filePath.length - 1; i++) {
      dirPath[i] = filePath[i];
    }
    final String fileName = filePath[filePath.length - 1];
    final ClientProjectFile child =
      projectService.getChild(projectId, dirPath, fileName);
    ProjectService.setUserId(null);
    return child;
  }

  /**
   * Obtm a descrio associada a um arquivo.
   * 
   * @return a descrio do arquivo
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public String getDescription() throws WIOServiceException {
    if (wioServerProject == null) {
      throw new WIOServiceException("Projeto nulo em [" + this.toString() + "]");
    }
    try {
      final String desc = wioServerProject.getDescription(serverPath);
      return desc.replaceAll("[^\\p{ASCII}]", "?");
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getDescription", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Obtm o i-simo objeto (arquivo ou diretrio) contido neste WIOFile.
   * 
   * @param i o ndice do objeto desejado. Esse ndice deve estar entre 0 e N-1
   *        (nmero mximo dado por <code>getNumFiles</code>).
   * @return o arquivo
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOFile getFile(final int i) throws WIOServiceException {
    if (serverPath == null) {
      throw new WIOServiceException("Path nulo detectado!");
    }
    if (wioServerProject == null) {
      throw new WIOServiceException("Projeto nulo detectado!");
    }
    ClientProjectFile[] children = null;
    try {
      final ProjectService projectService = ProjectService.getInstance();
      final Object loggedUserId = getMyUserId();
      ProjectService.setUserId(loggedUserId);
      final Object projectId = getMyProjectId();
      final String[] filePath = FileUtils.splitPath(serverPath);
      children = projectService.getChildren(projectId, filePath, false);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }

    if (children == null) {
      throw new WIOServiceException("Array de filhos nulo detectado!");
    }
    if ((i < 0) || (i >= children.length)) {
      throw new WIOServiceException("Indice invalido para array de filhos!");
    }
    final String childPath = children[i].getStringPath();
    WIOFile corbaFile = null;
    try {
      final WIOServerFile sonFile =
        new WIOServerFile(wioServerProject, childPath);
      final POA poa = sonFile.activateCorbaFile();
      corbaFile = WIOFileHelper.narrow(poa.servant_to_reference(sonFile));
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getFile(i)", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    if (corbaFile == null) {
      final String msg = "Falha na aquisicao de arquivo-filho";
      throw new WIOServiceException(msg);
    }
    return corbaFile;
  }

  /**
   * Obtm o tipo do arquivo.
   * 
   * @return uma String que identifica o tipo do arquivo.
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public String getFileType() throws WIOServiceException {
    if (serverPath == null) {
      throw new WIOServiceException("Path nulo detectado!");
    }
    if (wioServerProject == null) {
      throw new WIOServiceException("Projeto nulo detectado!");
    }
    try {
      final ClientProjectFile cpf = getClientProjectFile();
      return cpf.getType();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getFileType", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Busca do maior valor possvel para uma leitura (em bytes)
   * 
   * @return o tamanho mximo da leitura em bytes.
   */
  private int getMaxRead() {
    try {
      final WIOService wioService = WIOService.getInstance();
      return wioService.getMaxReadKb() * 1024;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getMaxRead", e);
      return 0;
    }
  }

  /**
   * Busca do maior valor possvel para uma escrita (em bytes)
   * 
   * @return o tamanho mximo da escrita em bytes.
   */
  private int getMaxWrite() {
    try {
      final WIOService wioService = WIOService.getInstance();
      return wioService.getMaxWriteKb() * 1024;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getMaxWrite", e);
      return 0;
    }
  }

  /**
   * Obtm o modo no qual o arquivo se encontra.
   * 
   * @return um objeto WIOFileMode que representa o modo atual do arquivo
   *         (NOT_OPENED, MODE_READ, MODE_WRITE)
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOFileMode getMode() throws WIOServiceException {
    if (javaFile == null) {
      throw new WIOServiceException("Arquivo fisico nulo!");
    }
    return fileMode;
  }

  /**
   * Obtem o identificador do projeto assocuiado ao arquivo.
   * 
   * @return o id
   */
  private Object getMyProjectId() {
    final User oldUser = ProjectService.getUser();
    final ProjectService projectService = ProjectService.getInstance();
    ProjectService.setUserId(User.getAdminId());
    final String prjName = wioServerProject.getProjectName();
    final Object userId = getMyUserId();
    final Object pid = projectService.getProjectId(userId, prjName);
    if (oldUser != null) {
      ProjectService.setUserId(oldUser.getId());
    }
    else {
      ProjectService.setUserId(null);
    }
    return pid;
  }

  /**
   * Obtem id do usurio loggado.
   * 
   * @return id
   */
  private Object getMyUserId() {
    final String uLogin = wioServerProject.getLoggedUserLogin();
    User loggedUser = null;
    try {
      loggedUser = User.getUserByLogin(uLogin);
    }
    catch (final Exception e) {
      final String fmt = "Falha na obtencao de usuario: %s";
      final String err = String.format(fmt, uLogin);
      Server.logSevereMessage(err);
      loggedUser = null;
    }

    if (loggedUser == null) {
      final String fmt = "Acesso a getProjectService com usuario invalido: %s";
      final String err = String.format(fmt, uLogin);
      Server.logSevereMessage(err);
      return null;
    }

    final Object loggedUserId = loggedUser.getId();
    return loggedUserId;
  }

  /**
   * Obtm o nmero de objetos (arquivos ou diretrios) contidos neste
   * <code>WIOFile</code>.
   * 
   * @return o nmero de arquivos/diretriosi contidos, ou 0 (zero) se este
   *         WIOFile representa um arquivo
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public int getNumFiles() throws WIOServiceException {
    if (serverPath == null) {
      throw new WIOServiceException("Path nulo detectado!");
    }
    if (wioServerProject == null) {
      throw new WIOServiceException("Projeto nulo detectado!");
    }
    ClientProjectFile[] children = null;
    try {
      final ProjectService projectService = ProjectService.getInstance();
      final Object loggedUserId = getMyUserId();
      ProjectService.setUserId(loggedUserId);
      final Object projectId = getMyProjectId();
      final String[] filePath = FileUtils.splitPath(serverPath);
      children = projectService.getChildren(projectId, filePath, false);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getNumFiles()", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }

    if (children == null) {
      throw new WIOServiceException("Array de filhos nulo detectado!");
    }
    return children.length;
  }

  /**
   * Obtm o diretrio que contm este arquivo.
   * 
   * @return o objeto <code>WIOFile</code> que representa o diretrio "pai"
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOFile getParent() throws WIOServiceException {
    final String[] splitPath = FileUtils.splitPath(getPath());
    String parentPath = "";
    final int limit = splitPath.length - 1;
    for (int i = 0; i < limit; i++) {
      parentPath = parentPath + splitPath[i];
      if (i != (limit - 1)) {
        parentPath = parentPath + File.separator;
      }
    }
    WIOFile corbaFile = null;
    try {
      final WIOServerFile parentFile =
        new WIOServerFile(wioServerProject, parentPath);
      final POA poa = parentFile.activateCorbaFile();
      corbaFile = WIOFileHelper.narrow(poa.servant_to_reference(parentFile));
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getParent", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }

    if (corbaFile == null) {
      final String err = "Falha de narrow acesso : " + this.toString();
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    return corbaFile;
  }

  /**
   * Obtm o caminho (path) deste arquivo.
   * 
   * @return o path como uma string.
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public String getPath() throws WIOServiceException {
    if (serverPath == null) {
      throw new WIOServiceException("Path nulo!");
    }
    return serverPath;
  }

  /**
   * Obtm o projeto ao qual pertence o arquivo.
   * 
   * @return um objeto <code>WIOProject</code> que representa o projeto que
   *         contm o arquivo
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOProject getProject() throws WIOServiceException {
    WIOProject corbaPrj = null;
    try {
      final POA poa = this.activateCorbaFile();
      final WIOServerProject wsp = wioServerProject;
      corbaPrj = WIOProjectHelper.narrow(poa.servant_to_reference(wsp));
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no getProject", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }

    if (corbaPrj == null) {
      final String err = "Falha de narrow de projeto : " + this.toString();
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    return corbaPrj;
  }

  /**
   * Verifica se  um diretrio.
   * 
   * @return true se  um diretrio, false caso contrrio
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public boolean isDirectory() throws WIOServiceException {
    if (javaFile == null) {
      throw new WIOServiceException("Arquivo fisico nulo!");
    }
    return javaFile.isDirectory();
  }

  /**
   * Verifica se este arquivo est publicado. O mecanismo de publicao foi
   * removido mas, para no alterarmos a IDL do WIO, este mtodo foi mantido.
   * Ele sempre retorna false, uma vez que no existe mais publicao.
   * 
   * @return false
   */
  @Override
  public boolean isPublished() {
    return false;
  }

  /**
   * Verifica se este arquivo est em construo.
   * 
   * @return true se o arquivo est em construo, false caso contrrio
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public boolean isUnderConstruction() throws WIOServiceException {
    try {
      final ClientProjectFile cpf = getClientProjectFile();
      return cpf.isUnderConstruction();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no isUnderConstrution", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Obtm a data de ltima modificao do arquivo.
   * 
   * @return a data de modificao do arquivo, representada por um objeto
   *         <code>WIODateTimeInfo</code>
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIODateTimeInfo lastModificationTime() throws WIOServiceException {
    try {
      final ClientProjectFile cpf = getClientProjectFile();
      return buildDateTime(cpf.getModificationDate());
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no lastModificationTime", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Mtodo de logging no servio de WIO.
   * 
   * @param msg a mensagem
   */
  private void logInfo(final String msg) {
    try {
      final WIOService wioService = WIOService.getInstance();
      if (wioService != null) {
        Server.logInfoMessage(msg);
      }
    }
    catch (final Exception e) {
      System.out.println(msg);
      System.out.println(e.getMessage());
      e.printStackTrace();
    }
  }

  /**
   * Abre o arquivo.
   * 
   * @param mode o modo no qual o arquivo deve ser aberto (leitura ou escrita)
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void open(final WIOFileMode mode) throws WIOServiceException {
    if (fileMode != WIOFileMode.NOT_OPENED) {
      throw new WIOServiceException("Arquivo ja esta aberto.");
    }
    if (mode == WIOFileMode.NOT_OPENED) {
      throw new WIOServiceException("Modo invalido para abertura de arquivo.");
    }
    try {
      if (mode == WIOFileMode.MODE_READ) {
        fileMode = mode;
        randomFile = new RandomAccessFile(javaFile, "r");
        logInfo("Arquivo aberto para leitura: " + javaFile.getPath());
      }
      else if (mode == WIOFileMode.MODE_WRITE) {
        fileMode = mode;
        randomFile = new RandomAccessFile(javaFile, "rws");
        wioServerProject.setUnderConstruction(serverPath, true);
        logInfo("Arquivo aberto para escrita: " + javaFile.getPath());
      }
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no open()", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Mtodo de leitura remota.
   * 
   * @param numBytes nmero de bytes a serem lidos
   * @param holder o buffer de leitura (usado na transferncia dos dados via
   *        CORBA)
   * @return o nmero de bytes efetivamente lidos
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public int read(final int numBytes, final BytesHolder holder)
    throws WIOServiceException {
    final String myName = "[" + this.toString() + "]";
    int size = numBytes;
    if (fileMode != WIOFileMode.MODE_READ) {
      final String err = "Leitura por arquivo fora do modo READ : " + myName;
      throw new WIOServiceException(err);
    }
    if (size <= 0) {
      final String err = "Tamanho de READ negativo! : " + myName;
      throw new WIOServiceException(err);
    }
    if (randomFile == null) {
      final String err = "Falha em READ por random-file nulo! : " + myName;
      throw new WIOServiceException(err);
    }
    if (size > getMaxRead()) {
      Server.logSevereMessage("Tamanho de READ fora do limite: " + myName
        + " - " + "truncando de " + size + " para " + getMaxRead());
      size = getMaxRead();
    }
    try {
      final long pos = randomFile.getFilePointer();
      final long len = randomFile.length();
      int nbytes = size;
      if ((pos + size) > len) {
        nbytes = (int) (len - pos);
        logInfo("Tamanho de READ maior que resto do arquivo: " + myName + " - "
          + "truncando de " + size + " para " + nbytes + " " + "(cursor=" + pos
          + " tamanho=" + len + ")");
      }
      final byte[] buffer = new byte[nbytes];
      randomFile.readFully(buffer);
      holder.value = buffer;
      return nbytes;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de read", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Redimensiona o arquivo.
   * 
   * @param size o novo tamanho do arquivo (em bytes)
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void resize(final long size) throws WIOServiceException {
    final String myName = "[" + this.toString() + "]";
    if (randomFile == null) {
      final String err = "Falha em READ por random-file nulo! : " + myName;
      throw new WIOServiceException(err);
    }
    try {
      randomFile.setLength(size);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de resize", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Estabelece a posio corrente do arquivo. Essa nova posio  determinada
   * pela soma de um deslocamento (offset) a uma posio "base" (inicio do
   * arquivo, posio corrente, fim do arquivo).
   * 
   * @param offset o deslocamento a ser somado  posio "base"
   * @param type a posio "base", representada por um objeto do tipo SeekType
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void seek(final long offset, final SeekType type)
    throws WIOServiceException {
    final String myName = "[" + this.toString() + "]";
    if (randomFile == null) {
      final String err = "Falha de seek por random-file nulo em " + myName;
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    try {
      long position = 0;
      if (type == SeekType.SEEK_TYPE_SET) {
        position = offset;
      }
      else if (type == SeekType.SEEK_TYPE_CUR) {
        final long pos = randomFile.getFilePointer();
        position = pos + offset;
      }
      else if (type == SeekType.SEEK_TYPE_END) {
        final long length = randomFile.length();
        position = length + offset;
      }
      randomFile.seek(position);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de seek", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Associa uma a descrio ao arquivo.
   * 
   * @param desc a descrio do arquivo
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void setDescription(final String desc) throws WIOServiceException {
    if (wioServerProject == null) {
      throw new WIOServiceException("Projeto nulo em [" + this.toString() + "]");
    }
    try {
      wioServerProject.setDescription(serverPath, desc);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de setDescription", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Obtm o tamanho do arquivo.
   * 
   * @return o tamanho do arquivo, em nmero de bytes
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public long size() throws WIOServiceException {
    if (javaFile == null) {
      final String myName = "[" + this.toString() + "]";
      String err = "Falha de size(). Arquivo fisico nao existe: ";
      err = err + myName;
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    if (isDirectory()) {
      final String myName = "[" + this.toString() + "]";
      String err = "Falha de size(). Pedido sobre um diretorio: ";
      err = err + myName;
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    try {
      return javaFile.length();
    }
    catch (final Exception e) {
      final String err = e.getMessage();
      Server.logSevereMessage(err);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Obtm a posio corrente do arquivo.
   * 
   * @return a posio corrente do arquivo, representada como um deslocamento
   *         (nmero de bytes) em relao ao incio do arquivo
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public long tell() throws WIOServiceException {
    if (randomFile == null) {
      final String myName = "[" + this.toString() + "]";
      final String err = "Falha de eof por random-file nulo em " + myName;
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    try {
      return randomFile.getFilePointer();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de tell", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    final Object uid = getMyUserId();
    final Object pid = getMyProjectId();

    String userText = "usurio nulo";
    if (uid != null) {
      userText = uid.toString();
    }

    String prjText = "projeto nulo";
    if (pid != null) {
      prjText = pid.toString();
    }

    return "[WIO File]: " + userText + "@" + prjText + ":" + serverPath;
  }

  /**
   * Obtm a identificao do usurio que criou o arquivo.
   * 
   * @return a identificao do usurio criador do arquivo
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public String whoCreated() throws WIOServiceException {
    try {
      final ClientProjectFile cpf = getClientProjectFile();
      final Object whoCreated = cpf.whoCreated();
      if (whoCreated == null) {
        return "???";
      }
      return whoCreated.toString();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha no mtodo whoCreated", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Mtodo de escrita remota.
   * 
   * @param content contedo (sequncia de bytes) a ser escrito a partir da
   *        posio corrente.
   * @return o nmero de bytes escritos efetivamente
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public int write(final byte[] content) throws WIOServiceException {
    final String myName = "[" + this.toString() + "]";
    if (fileMode != WIOFileMode.MODE_WRITE) {
      final String err = "Leitura por arquivo fora do modo WRITE : " + myName;
      throw new WIOServiceException(err);
    }
    if (randomFile == null) {
      final String err = "Falha em READ por random-file nulo! : " + myName;
      throw new WIOServiceException(err);
    }
    int size = content.length;
    if (size > getMaxWrite()) {
      final String err =
        "Tamanho de WRITE fora do limite: " + myName + " - " + "truncando de "
          + size + " para " + getMaxWrite();
      Server.logSevereMessage(err);
      size = getMaxWrite();
    }
    try {
      randomFile.write(content, 0, size);
      return size;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de write", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Constri o objeto que representa um arquivo remoto,
   * 
   * @param serverProject o projeto que contm o arquivo
   * @param serverPath o path do arquivo
   * @throws Exception em caso de falha.
   */
  protected WIOServerFile(final WIOServerProject serverProject,
    final String serverPath) throws Exception {
    if (serverProject == null) {
      throw new Exception("Projeto/usuario nao identificado! ");
    }
    if (serverPath == null) {
      throw new Exception("Arquivo nao identificado!");
    }
    this.wioServerProject = serverProject;
    this.serverPath = serverPath;
    this.fileMode = WIOFileMode.NOT_OPENED;

    final ProjectService projectService = ProjectService.getInstance();
    final Object loggedUserId = getMyUserId();
    ProjectService.setUserId(loggedUserId);

    final Object pid = getMyProjectId();
    final String[] filePath = FileUtils.splitPath(this.serverPath);
    try {
      this.javaFile = projectService.getFile(pid, filePath);
    }
    catch (final ServiceFailureException e) {
      final String fmt = "Falha de arquivo WIO: projeto '%s' (%s) getFile(%s)";
      final String prjName = serverProject.getProjectName();
      final String err = String.format(fmt, prjName, pid, this.serverPath);
      Server.logSevereMessage(err);
      throw new Exception(e);
    }
    finally {
      ProjectService.setUserId(null);
    }

    if (javaFile == null || !javaFile.exists()) {
      final String path = javaFile.getAbsolutePath();
      final String fmt = "Arquivo fisico nao existe no servidor: %s";
      final String err = String.format(fmt, path);
      Server.logSevereMessage(err);
      throw new Exception(err);
    }
    serverProject.incNumFiles();
  }
}
