/*
 * PriorityQueue.java
 *
 * $Author: fpina $ $Date: 2016-08-03 11:38:03 -0300 (Wed, 03 Aug 2016) $
 * $Revision: 175310 $
 */
package csbase.server.services.schedulerservice;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import csbase.logic.CommandInfo;
import csbase.logic.CommandInfoCache;
import csbase.logic.CommandStatus;
import csbase.logic.Priority;
import csbase.server.Server;

/**
 * Implementa uma fila com prioridades e a possibilidade de escolha de posio
 * fixa para um comando.
 *
 * @author Bruno Oliveira Silvestre
 *
 */
class PriorityQueue {
  /**
   * Filas de prioridade. Apesar da fila ter tamanho fixo, no podemos usar
   * array com generics.
   */
  private List<List<CommandInfo>> queues;
  /** ndice para localizao rpida do comando */
  private Map<Object, CommandInfo> commandIndex;
  /** ndice para localizao rpida da fila */
  private Map<Object, List<CommandInfo>> queueIndex;
  /**
   * Indica que a fila est bloqueada, ou seja, todos os comandos permanecem na
   * fila at que ela seja desbloqueada
   **/
  private boolean blocked;
  /** Caminho do arquivo para backup */
  private String backupFilePath;
  /** Referncia para o servio */
  private SchedulerService service;

  /**
   * Contri uma nova fila de prioridade.
   *
   * @param backupFilePath Caminho do arquivo de backup
   */
  protected PriorityQueue(String backupFilePath) {
    initQueue();
    this.backupFilePath = backupFilePath;
    this.service = SchedulerService.getInstance();
  }

  /**
   * Insere um comando na fila de prioridade.
   *
   * @param cmd Comando a ser inserido.
   */
  public synchronized void add(CommandInfo cmd) {
    List<CommandInfo> queue = getQueue(cmd.getPriority());
    queue.add(cmd);
    commandIndex.put(cmd.getId(), cmd);
    queueIndex.put(cmd.getId(), queue);
    saveToBackup();
    notify();
  }

  /**
   * Altera a prioridade.
   *
   * @param commandId Identificador do comando.
   * @param priority Nova prioridade do comando.
   *
   * @return <code>true</code> se a atualizao foi concluda com sucesso.
   */
  public synchronized boolean setPriority(Object commandId, Priority priority) {
    CommandInfo cmd = getCommand(commandId);
    if (cmd == null) {
      return false;
    }

    if (cmd.getPriority() != priority) {
      cmd.setPriority(priority);

      // Move o comando entre filas
      List<CommandInfo> fromQueue = getQueue(cmd);
      List<CommandInfo> toQueue = getQueue(priority);

      /*
       * Se um comando foi colocado manualmente em outra fila, evitar que ele
       * troque de posio dentro da mesma fila.
       */
      if (fromQueue != toQueue) {
        fromQueue.remove(cmd);
        toQueue.add(cmd);
        // Atualiza ndice de filas do comando
        queueIndex.put(commandId, toQueue);
      }
      saveToBackup();
    }
    return true;
  }

  /**
   * Altera a posio do comando.
   *
   * @param commandId Identificador do comando.
   * @param position Nova posio do commando na fila, no intervalo [0, size-1].
   *
   * @return <code>true</code> se a atualizao foi concluda com sucesso.
   */
  public synchronized boolean setPosition(Object commandId, int position) {
    CommandInfo cmd = getCommand(commandId);
    if (cmd == null) {
      return false;
    }

    int atual = indexOf(cmd);
    if (atual == position) {
      return true;
    }

    List<CommandInfo> fromQueue = getQueue(cmd);
    List<CommandInfo> toQueue = getQueue(position);
    int fromPos = normalizePosition(atual);
    int toPos = normalizePosition(position);

    // Corrige a posio devido ao futuro deslocamento.
    if (position > atual) {
      toPos++;
    }
    toQueue.add(toPos, cmd);

    // Objetos podem ter sido deslocados dentro da mesma fila, corrigir.
    if (fromQueue == toQueue && toPos < fromPos) {
      fromPos++;
    }
    fromQueue.remove(fromPos);

    // Altera o ndice de filas
    queueIndex.put(commandId, toQueue);

    saveToBackup();

    return true;
  }

  /**
   * Retorna o tamanho da fila.
   *
   * @return A quantidade de comandos na fila.
   */
  public synchronized int size() {
    return commandIndex.size();
  }

  /**
   * Retorna os comandos da fila, em ordem de posio. Este mtodo chama o
   * mtodo <code>wait</code> caso no haja comandos na fila. O
   * <code>notify</code>  chamado quando um novo processo  adicionado a fila
   * pelo mtodo <code>add</code>. O mtodo <code>getCommands</code> deve ser
   * chamado quando no  desejvel a espera pela chegada de comandos, ou seja,
   * no importa se a fila est vazia.
   *
   * @return Os comandos da fila.
   */
  public synchronized List<CommandInfo> getAndWaitForCommands() {
    if (blocked || size() <= 0) {
      try {
        do {
          wait();
        } while (blocked);
      }
      catch (InterruptedException e) {
      }
    }

    return getCommands();
  }


  /**
   * Retorna os comandos da fila, em ordem de posio.
   *
   * @return Os comandos da fila.
   */
  public synchronized List<CommandInfo> getCommands() {
    return queues.stream()
        .flatMap(q -> q.stream())
        .collect(Collectors.toList());
  }

  /**
   * Altera o estado da fila para bloqueada ou desbloqueada. Quando bloqueada,
   * todos os commandos permanecem na fila at que ela seja desbloqueada.
   *
   * @param blocked Indica o bloqueio (true) ou desbloqueio (false).
   */
  public synchronized void setBlocked(boolean blocked) {
    this.blocked = blocked;
    saveToBackup();
    /** Notifica a liberao da fila. */
    if (!blocked) {
      notify();
    }
  }

  /**
   * Remove o comando da fila.
   *
   * @param commandId Identificador do comando.
   * @return o comando removido ou null caso ele no estivesse presente na fila.
   */
  public synchronized CommandInfo remove(Object commandId) {
    CommandInfo cmd = getCommand(commandId);
    if (cmd == null) {
      return null;
    }

    List<CommandInfo> queue = getQueue(cmd);
    queue.remove(cmd);
    commandIndex.remove(cmd.getId());
    queueIndex.remove(cmd.getId());
    saveToBackup();

    return cmd;
  }

  /**
   * Dorme durante o tempo especificado ou at que seja acordada por outra
   * thread que invoque <code>notify</code>.
   *
   * @param queueProcessingInterval tempo mximo de interrupo da execuo.
   */
  protected synchronized void sleep(long queueProcessingInterval) {
    try {
      wait(queueProcessingInterval);
    }
    catch (InterruptedException e) {
    }
  }

  /**
   * Salva o contedo da fila de prioridade em arquivo.
   */
  private void saveToBackup() {
    try {
      ObjectOutputStream out =
        new ObjectOutputStream(new FileOutputStream(backupFilePath));

      PriorityQueueVO queueVO = new PriorityQueueVO();
      queueVO.commandIndex = commandIndex;
      queueVO.queueIndex = queueIndex;
      queueVO.queues = queues;
      queueVO.blocked = blocked;

      out.writeObject(queueVO);
      out.flush();
      out.close();
    }
    catch (IOException e) {
      Server.logSevereMessage("Erro ao salvar a fila de comandos.", e);
    }
  }

  /**
   * Carrega o contedo da fila de prioridade de arquivo.
   */
  protected synchronized void loadFromBackup() {
    // TODO #4601 WA para que as referncias de CommandInfo associadas um
    // comando nas filas do Scheduler e do SGA sejam as mesmas.
    CommandInfoCache.enable();

    try {
      ObjectInputStream input =
        new ObjectInputStream(new FileInputStream(backupFilePath));
      PriorityQueueVO queueVO = (PriorityQueueVO) input.readObject();
      input.close();

      // Se o nmero de filas persistidas for diferente do nmero
      // de filas inicializadas, despreza o backup.
      if (queueVO.queues.size() != queues.size()) {
        if (queueVO.commandIndex.size() < 1) {
          Server.logWarningMessage("As prioridades da fila foram alteradas. "
            + "No h comandos persistidos no backup. Ento, o "
            + "servio de escalonamente funcionar normalmente.");
        }
        else {
          Server.logSevereMessage("As prioridades da fila foram alteradas "
            + "e no  possvel recuperar o backup da fila. O servio "
            + "de escalonamente funcionar, mas sem os comandos persistidos.");
        }
        return;
      }
      commandIndex = queueVO.commandIndex;
      queueIndex = queueVO.queueIndex;
      queues = queueVO.queues;
      blocked = queueVO.blocked;
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro ao carregar o estado da fila de comandos.",
        e);
    }
  }

  /**
   * Inicializa o estado interno da fila de prioridade.
   *
   */
  private void initQueue() {
    final int len = Priority.values().length;
    queues = new ArrayList<List<CommandInfo>>(len);

    for (int i = 0; i < len; i++) {
      queues.add(i, new LinkedList<CommandInfo>());
    }

    commandIndex = new HashMap<Object, CommandInfo>();
    queueIndex = new HashMap<Object, List<CommandInfo>>();
    blocked = false;
  }

  /**
   * Normaliza a posio de um ndice global para um ndice local, de acordo com
   * a fila.
   *
   * @param position Posio global.
   *
   * @return Posio relativa  fila.
   */
  private int normalizePosition(int position) {
    for (List<CommandInfo> q : queues) {
      if (position < q.size()) {
        return position;
      }
      position -= q.size();
    }
    // no  para chegar aqui
    throw new AssertionError();
  }

  /**
   * Retorna o comando dado seu identificador.
   *
   * @param commandId Identificador do comando.
   *
   * @return O comando ou <code>null</code> se o comando no foi encontrado.
   */
  // TODO mjulia
  // Coloquei o mtodo publico para verificar se o comando ainda est na fila
  // depois que o scheduller obteve a lista de comandos a processar.
  public synchronized CommandInfo getCommand(Object commandId) {
    return commandIndex.get(commandId);
  }

  /**
   * Retorna a fila onde o comando se encontra, dado seu identificador.
   *
   * @param cmd Identificador do comando.
   *
   * @return A fila onde o comando est ou <code>null</code> se a fila no foi
   *         encontrada.
   */
  private List<CommandInfo> getQueue(CommandInfo cmd) {
    return queueIndex.get(cmd.getId());
  }

  /**
   * Retorna a fila responsvel por determinada prioridade.
   *
   * @param priority Prioridade do comando.
   *
   * @return A fila.
   */
  private List<CommandInfo> getQueue(Priority priority) {
    return queues.get(priority.ordinal());
  }

  /**
   * Retorna a fila responsvel por determinada posio.
   *
   * @param position Posio do objeto.
   *
   * @return A fila.
   */
  private List<CommandInfo> getQueue(int position) {
    for (List<CommandInfo> q : queues) {
      if (position < q.size()) {
        return q;
      }
      position -= q.size();
    }
    // no  para chegar aqui
    throw new AssertionError();
  }

  /**
   * Retorna a posio do comando na fila como um todo.
   *
   * @param cmd Comando.
   *
   * @return A posio do comando.
   */
  private int indexOf(CommandInfo cmd) {
    List<CommandInfo> queue = getQueue(cmd);

    // Conta a quantidade de elementos anteriores
    int extra = 0;
    for (List<CommandInfo> q : queues) {
      if (q == queue) {
        break;
      }
      extra += q.size();
    }

    return queue.indexOf(cmd) + extra;
  }

  /**
   * @return Se a fila de comandos submetidos est bloqueada para execuo ou
   *         no.
   */
  public boolean isBlocked() {
    return blocked;
  }
}
