package csbase.server.services.sgaservice;

import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.text.MessageFormat;
import java.util.Properties;

import org.omg.CORBA.IntHolder;

import sgaidl.CompletedCommandInfo;
import sgaidl.InvalidCommandException;
import sgaidl.InvalidParameterException;
import sgaidl.InvalidSGAException;
import sgaidl.NoPermissionException;
import sgaidl.RetrievedInfo;
import sgaidl.SGAAlreadyRegisteredException;
import sgaidl.SGACommand;
import sgaidl.SGADaemon;
import sgaidl.SGAManagerPOA;
import sgaidl.SGANotRegisteredException;
import sgaidl.SGAProperties;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.services.sgaservice.ssl.SSLContext;

/**
 * A classe <code>SGAHandler</code> implementa o <i>servant</i> associado 
 * interface idl <code>SGAManager</code>, atravs da qual os SGAs interagem com
 * o servio de gerncia de SGAs. As requisies recebidas dos SGAs so
 * repassadas pelo SGAHandler  implementao do servio de gerncia de SGAs
 * (SGAService).
 */
public class SGAHandler extends SGAManagerPOA {
  /** Referncia para a implementao do servio SGAService */
  private SGAService service = null;

  /** Servidor CORBA responsvel pela interao com os SGAs */
  private ORBHandler orbHandler = null;

  /**
   * SGA avisa ao Manager que est ativo.
   *
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   * @param sgaProperties informaes do SGA
   * @param updateInterval informa de quanto em quanto tempo o SGA deve dizer
   *        que est vivo.
   *
   * @throws NoPermissionException se o SGA no tem permisso para se registrar
   * @throws SGAAlreadyRegisteredException se o SGA j est registrado
   * @throws InvalidParameterException se algum parmetro em sgaProperties for
   *         invlido.
   */
  @Override
  public void registerSGA(SGADaemon sga, String sganame,
    SGAProperties sgaProperties, IntHolder updateInterval)
      throws NoPermissionException, InvalidParameterException,
      SGAAlreadyRegisteredException {
    /*
     * Timeout para as chamadas de mtodo do SGA. Caso o valor seja negativo, o
     * controle de timeout  desativado.
     */
    long timeout = service.getLongProperty("Timeout");
    final String msg =
      (timeout < 0) ? "Timeout na conexo com o SGA desativado"
        : "Timeout para as chamadas ao SGA (seg): " + timeout;
    Server.logInfoMessage(msg);
    // Configura o timeout para a conexo.
    if (timeout >= 0) {
      try {
        orbHandler.setTimeOut(sga, timeout);
      }
      catch (Exception e) {
        Server.logSevereMessage("Erro configurando o timeout do SGA: "
          + sganame, e);
      }
    }

    if (service.getBooleanProperty("SSL.enable")) {
      SSLContext inspector = SSLContext.getSSLContext(_orb());
      String authenticatedSGA = inspector.getContextInformation();
      if (!authenticatedSGA.equals(sganame)) {
        String message =
          "Negada a conexo do SGA {0}: nome informado no  igual ao do certificado utilizado.";
        Server.logWarningMessage(MessageFormat.format(message, sganame));
        throw new NoPermissionException();
      }
      else {
        if (!service.isSGAAuthorized(sganame)) {
          String message =
            "Negada a conexo do SGA {0}: no est na lista de SGAs autorizados.";
          Server.logWarningMessage(MessageFormat.format(message, sganame));
          throw new NoPermissionException();
        }
        ;
      }
    }

    service.registerSGA(sga, sganame, sgaProperties, updateInterval);
  }

  /**
   * SGA avisa ao Manager que sair do ar.
   *
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   *
   * @throws SGANotRegisteredException se o SGA no est registrado
   * @throws NoPermissionException se o SGA no tem permisso de registro
   */
  @Override
  public void unregisterSGA(SGADaemon sga, String sganame)
    throws NoPermissionException, SGANotRegisteredException {
    service.unregisterSGA(sga, sganame);
  }

  /**
   * SGA avisa ao Manager que um comando terminou
   *
   * @param sgaName nome do SGA
   * @param cmd referncia para o gerenciador do comando
   * @param cmdId identificador para o comando
   * @param info informaes sobre o trmino de execuo do comando
   *
   * @return <code>true</code> se a notificao foi feita, <code>false</code> se
   *         o SSI desconhece o SGA
   *
   * @throws InvalidCommandException se o comando for invlido
   * @throws InvalidSGAException se o SGA for invlidos
   * @throws NoPermissionException se ocorrer um erro de permisso
   */
  @Override
  public boolean commandCompleted(String sgaName, SGACommand cmd, String cmdId,
    CompletedCommandInfo info) throws InvalidSGAException,
    InvalidCommandException, NoPermissionException {

    return service.commandCompleted(sgaName, cmd, cmdId, info);
  }

  /**
   * SGA avisa ao Manager que um comando no foi recuperado
   *
   * @param sgaName nome do SGA
   * @param cmdId identificador do comando
   *
   * @return <code>true</code> se a notificao foi feita, <code>false</code> se
   *         o SSI desconhece o SGA
   *
   * @throws InvalidSGAException se o SGA for invlido
   * @throws InvalidCommandException se o comando for invlido
   */
  @Override
  public boolean commandLost(String sgaName, String cmdId)
    throws InvalidSGAException, InvalidCommandException {
    return service.commandLost(sgaName, cmdId);
  }

  /**
   * SGA avisa ao Manager que comandos foram recuperados.
   *
   * @param sgaName nome do SGA
   * @param cmds comandos recuperados
   *
   * @return <code>true</code> se os comandos foram registrados com sucesso,
   *         <code>false</code> se o SSI desconhece o SGA
   *
   * @throws InvalidSGAException se o SGA for invlido
   * @throws InvalidCommandException se o comando for invlido
   * @throws NoPermissionException se ocorrer um erro de permisso
   */
  @Override
  public boolean commandRetrieved(String sgaName, RetrievedInfo[] cmds)
    throws InvalidSGAException, NoPermissionException, InvalidCommandException {
    /*
     * o mtodo do servio se encarrega de definir o usurio na thread para cada
     * comando
     */
    return service.commandRetrieved(sgaName, cmds);
  }

  /**
   * SGA verifica se o Manager est ativo
   *
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   *
   * @return true se o SGA j est registrado
   *
   * @throws NoPermissionException se o SGA no tem permisso de registro
   * @throws InvalidSGAException se o SGA for invlido
   */
  @Override
  public boolean isRegistered(SGADaemon sga, String sganame)
    throws InvalidSGAException, NoPermissionException {
    return service.isRegistered(sga, sganame);
  }

  /**
   * Atualiza os dados de um SGA no Manager.
   *
   * @param sga referncia para o SGA
   * @param sganame nome do SGA
   * @param sgaProperties propriedades do SGA
   *
   * @return <code>true</code> se os dados foram atualizados com sucesso
   *
   * @throws SGANotRegisteredException se o SGA no est registrsdo
   * @throws NoPermissionException se o SGA no tem permisso
   * @throws InvalidParameterException caso alguem parmetro seja invlido
   */
  @Override
  public boolean updateSGAInfo(SGADaemon sga, String sganame,
    SGAProperties sgaProperties) throws InvalidParameterException,
    NoPermissionException, SGANotRegisteredException {
    return service.updateSGAInfo(sga, sganame, sgaProperties);
  }

  /**
   * Constri um servant para a interface SGAManager
   *
   * @param srv referncia para a implementao do servio SGAService
   * @throws ServerException
   */
  public SGAHandler(SGAService srv) throws ServerException {
    service = srv;
    initORB();
  }

  /**
   * Inicializao do ORB do servio de gerncia de SGAs. Cria o servidor CORBA
   * responsvel pela interao com os SGAs e ativa o objeto responsvel pela
   * implementao da interface SGAManager.
   *
   * @throws ServerException se ocorrer um erro na criao do servidor CORBA ou
   *         na ativao do servant.
   */
  final private void initORB() throws ServerException {
    try {
      Server.logInfoMessage("Acessando ambiente ORB...");
      final Properties props = service.getExternalPropertyFile("ORB.file");
      /*
       * As propriedades que definem o ip e porta so definidas nas propriedades
       * do servio.
       */
      if (!service.isPropertyNull("ORB.port")) {
        int port = service.getIntProperty("ORB.port");
        String portStr = Integer.toString(port);
        props.setProperty("OAPort", portStr);
      }
      if (!service.isPropertyNull("ORB.hostAddr")) {
        String addr = service.getStringProperty("ORB.hostAddr");
        props.setProperty("OAIAddr", addr);
      }

      if (service.getBooleanProperty("SSL.enable")) {
        final Properties SSLProps =
          service.getExternalPropertyFile("ORB.SSL.file");
        props.putAll(SSLProps);

        String ks = service.getStringProperty("SSL.keystore");
        String ksPass = service.getStringProperty("SSL.keystore_password");

        props.setProperty("jacorb.security.keystore", ks);
        props.setProperty("jacorb.security.keystore_password", ksPass);
        orbHandler = new ORBHandler(props, true);
      }
      else {
        orbHandler = new ORBHandler(props, false);
      }
      Server.logInfoMessage("Acesso ao ambiente ORB inicializado");
    }
    catch (ORBException e) {
      throw new ServerException("Impossvel criar servidor CORBA: "
        + e.getMessage(), e);
    }
    try {
      String ior = orbHandler.activateObject("SGAManager", this);
      if (service.getBooleanProperty("SSL.enable")) {
        String iorFileName = service.getStringProperty("SSL.exported.ior.file");

        Files.write(Paths.get(iorFileName), ior.getBytes(),
          StandardOpenOption.WRITE, StandardOpenOption.CREATE);
      }
    }
    catch (Exception e) {
      throw new ServerException("Impossvel criar handler de SGA's", e);
    }

    Server.logInfoMessage("Acesso ao ambiente ORB inicializado");
  }

  /**
   *
   * @throws ServerException
   */
  public void shutdown() throws ServerException {
    try {
      if (orbHandler != null) {
        orbHandler.shutdown();
      }
      orbHandler = null;
    }
    catch (ORBException e) {
      throw new ServerException("Impossvel desativar servidor CORBA", e);
    }
  }
}
