package csbase.tools;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;

/**
 * Funcionalidade comum aos geradores de classes auxiliares para servio do
 * csbase
 * 
 * @author Jorge Alexandre & Ana Moura
 */
public abstract class Generator {
  private final StringBuffer buffer = new StringBuffer();
  protected String interfaceName;
  protected String serviceClassName;
  protected String genClassName;

  /**
   * Obtm o nome do pacote de uma classe.
   * 
   * @param className nome completo da class (incluindo pacote)
   * 
   * @return o nome do pacote
   */
  protected static String getPackageName(String className) {
    int i = className.lastIndexOf(".");
    if (i != -1) {
      return className.substring(0, i);
    }
    return null;
  }

  /**
   * Obtm o nome de uma classe, sem o pacote.
   * 
   * @param className nome completo da class (incluindo pacote)
   * 
   * @return o nome da classe sem o pacote
   */
  protected static String getNoPackagesClassName(String className) {
    int i = className.lastIndexOf(".");
    if (i != -1) {
      return className.substring(i + 1);
    }
    return null;
  }

  /**
   * Exibe mensagem com instrues para execuo da linha de comando.
   */
  protected static void usage() {
    System.out.println("uso:");
    System.out.println("args[0]: nome da interface remota do servio");
    System.out.println(" ex.: websintesi.remote.OpenSpiritServiceInterface");
    System.out.println("\nargs[1]: nome da classe que implementa o servio");
    System.out.println(" ex: websintesi.ssi.ospservice.OpenSpiritService");
  }

  /**
   * Contri uma instncia do gerador.
   * 
   * @param interfaceName nome completo da interface remota do servio
   * @param serviceClassName nome completo da classe que implementa o servio
   * @param suffix sufixo a acrescentar no nome da classe de servio para formar
   *        o nome da classe gerada (Proxy, Unavailable, etc)
   */
  protected Generator(String interfaceName, String serviceClassName,
    String suffix) {
    this.interfaceName = interfaceName;
    this.serviceClassName = serviceClassName;
    this.genClassName = serviceClassName + suffix;
  }

  /**
   * Obtm o buffer que armazena o cdigo.
   * 
   * @return buffer com o cdigo armazenado
   */
  protected StringBuffer getBuffer() {
    return buffer;
  }

  /**
   * Gera o cdigo fonte em um <code>StringBuffer</code>
   * 
   * @throws ClassNotFoundException se a classe da interface remota no for
   *         encontrada
   */
  protected final void generate() throws ClassNotFoundException {
    String noPackagesClassName = getNoPackagesClassName(genClassName);
    final Class interfaceClass = Class.forName(interfaceName);

    /* Gerando tag para svn svn:keywords (evitando colocar a string literal) */
    buffer.append("/*\n");
    buffer.append(" * $");
    buffer.append("Id:");
    buffer.append("$\n");
    buffer.append(" */\n\n");

    /* O pacote da classe gerada  o mesmo da implementao do servio */
    buffer.append("package ");
    buffer.append(getPackageName(serviceClassName));
    buffer.append(";\n\n");
    buffer.append("import ");
    buffer.append(interfaceName);
    buffer.append(";\n\n");

    /* Pacotes importados so definidos pelas especializaes do gerador */
    String[] imports = getImports();
    for (int i = 0; (imports != null) && (i < imports.length); i++) {
      buffer.append("import ");
      buffer.append(imports[i]);
      buffer.append(";\n");
    }

    /* Declarao de abertura da classe gerada */
    buffer.append("\n\n");
    buffer.append("/**\n");
    buffer.append(" * Classe de proxy gerada automaticamente para:\n");
    buffer.append(" * interface do servio: " + interfaceName + "\n");
    buffer.append(" * @author Tecgraf/PUC-Rio\n");
    buffer.append(" */\n");
    buffer.append("public class ");
    buffer.append(noPackagesClassName);
    buffer.append(" extends ");
    buffer.append(getSuperClassName());
    buffer.append(" implements ");
    buffer.append(getNoPackagesClassName(interfaceName));
    buffer.append(" {\n\n");

    /* Construtores */
    if (invokesSuperClassConstructor()) {
      redefineSuperClassConstructors(noPackagesClassName);
    }
    else {
      String[] excpts = getConstructorExceptions();
      buffer.append("/**\n");
      buffer.append(" * Construtor\n");
      if (excpts != null && excpts.length > 0) {
        for (int e = 0; e < excpts.length; e++) {
          buffer.append(" * @throws ");
          buffer.append(excpts[e].toString());
        }
        buffer.append(" no caso de erro.\n");
      }

      buffer.append(" */\n");
      buffer.append("public ");
      buffer.append(noPackagesClassName);
      buffer.append("() ");
      String[] constructorExceptions = getConstructorExceptions();
      if (constructorExceptions != null && constructorExceptions.length > 0) {
        buffer.append("throws ");
        for (int j = 0; j < constructorExceptions.length; j++) {
          if (j > 0) {
            buffer.append(", ");
          }
          buffer.append(constructorExceptions[j]);
        }
      }
      buffer.append(" {}\n");
    }

    /* Mtodos */
    Method[] methods = interfaceClass.getMethods();
    generateMethods(methods);
    buffer.append("}");
  }

  /**
   * Gera um bloco de cdigo referente a um conjunto de mtodos.
   * 
   * @param methods mtodos que tero seus cdigos gerados.
   */
  private void generateMethods(Method[] methods) {
    for (int i = 0; i < methods.length; i++) {
      buffer.append("\n/** ");
      buffer.append("\n * {@inheritDoc} ");
      buffer.append("\n */ ");
      buffer.append("\n@Override");
      buffer.append("\npublic ");
      Type grt = methods[i].getGenericReturnType();
      if (grt instanceof Class) {
        buffer.append(typeToString((Class) grt));
      }
      else {
        buffer.append(grt.toString());
      }
      buffer.append(' ');
      buffer.append(methods[i].getName());
      buffer.append("(");
      Type[] parameters = methods[i].getGenericParameterTypes();
      String[] paremeterNames = new String[parameters.length];
      if (parameters != null && parameters.length > 0) {
        for (int j = 0; j < parameters.length; j++) {
          if (parameters[j] instanceof Class) {
            buffer.append(typeToString((Class) parameters[j]));
          }
          else {
            buffer.append(parameters[j].toString());
          }
          paremeterNames[j] = "param_" + j;
          buffer.append(" " + paremeterNames[j]);
          if (j >= parameters.length - 1) {
            break;
          }
          buffer.append(", ");
        }
      }
      buffer.append(") ");
      if (throwsInterfaceExceptions()) {
        Type[] throwses = methods[i].getGenericExceptionTypes();
        if (throwses != null && throwses.length > 0) {
          buffer.append("throws ");
          for (int j = 0; j < throwses.length; j++) {
            if (throwses[j] instanceof Class) {
              buffer.append(((Class) throwses[j]).getName());
            }
            else {
              buffer.append(throwses[j].toString());
            }
            if (j >= throwses.length - 1) {
              break;
            }
            buffer.append(", ");
          }
        }
      }
      buffer.append(" {\n");
      getMethodLines(buffer, methods[i], paremeterNames);
      buffer.append("}\n");
    }
  }

  /**
   * Redefine os construtores da superclasse, acrescentando cdigo especfico
   * aps a invocao do construtor da superclasse.
   * 
   * @param noPackagesClassName nome da classe gerada (sem o pacote)
   * 
   * @throws ClassNotFoundException em caso de falha na classe
   */
  private void redefineSuperClassConstructors(String noPackagesClassName)
    throws ClassNotFoundException {
    Constructor[] constructors = getSuperClass().getDeclaredConstructors();
    if (constructors != null && constructors.length > 0) {
      for (int i = 0; i < constructors.length; i++) {
        Type[] parameters = constructors[i].getGenericParameterTypes();
        Type[] exceptions = constructors[i].getGenericExceptionTypes();

        buffer.append("/**\n");
        buffer.append(" * Construtor\n");
        if (parameters != null && parameters.length > 0) {
          for (int p = 0; p < parameters.length; p++) {
            buffer.append(" * @param param_" + p + " gerao automtica.\n");
          }
          if (exceptions != null && exceptions.length > 0) {
            for (int e = 0; e < exceptions.length; e++) {
              buffer.append(" * @throws ");
              if (exceptions[e] instanceof Class) {
                buffer.append(((Class) exceptions[e]).getName());
              }
              else {
                buffer.append(exceptions[e].toString());
              }
            }
            buffer.append(" no caso de erro.\n");
          }
        }
        buffer.append(" */\n");
        buffer.append("public ");
        buffer.append(noPackagesClassName);
        buffer.append("(");

        String[] paremeterNames = new String[parameters.length];
        if (parameters != null && parameters.length > 0) {
          for (int j = 0; j < parameters.length; j++) {
            if (parameters[j] instanceof Class) {
              buffer.append(typeToString((Class) parameters[j]));
            }
            else {
              buffer.append(parameters[j].toString());
            }
            paremeterNames[j] = "param_" + j;
            buffer.append(" " + paremeterNames[j]);
            if (j >= parameters.length - 1) {
              break;
            }
            buffer.append(", ");
          }
        }
        buffer.append(") ");

        if (exceptions != null && exceptions.length > 0) {
          buffer.append("throws ");
          for (int j = 0; j < exceptions.length; j++) {
            if (exceptions[j] instanceof Class) {
              buffer.append(((Class) exceptions[j]).getName());
            }
            else {
              buffer.append(exceptions[j].toString());
            }
            if (j >= exceptions.length - 1) {
              break;
            }
            buffer.append(", ");
          }
        }
        buffer.append(" {\n");
        buffer.append("  super(");
        if (parameters != null && parameters.length > 0) {
          for (int j = 0; j < parameters.length; j++) {
            buffer.append(paremeterNames[j]);
            if (j >= parameters.length - 1) {
              break;
            }
            buffer.append(", ");
          }
        }
        buffer.append(");\n");
        getConstructorLines(buffer, paremeterNames);
        buffer.append("}\n");
      }
    }
  }

  /**
   * Descarrega o buffer para um arquivo. O arquivo contm a classe gerada, e
   * ser gravado na localizao correspondente  mesma (isto , obedecendo seu
   * pacote).
   * 
   * @param code .
   */
  protected void flushToFile(StringBuffer code) {
    String sep = File.separator;
    String[] pathParts = genClassName.split("\\.");
    StringBuffer sb = new StringBuffer(pathParts[0]);
    for (int i = 1; i < pathParts.length; i++) {
      sb.append(sep);
      sb.append(pathParts[i]);
    }
    sb.append(".java");
    String path = sb.toString();
    try {
      Charset cs = Charset.forName("ISO-8859-1");
      CharsetEncoder encoder = cs.newEncoder();
      PrintWriter out =
        new PrintWriter(new BufferedWriter(new OutputStreamWriter(
          new FileOutputStream(path), encoder)));
      out.print(code.toString());
      if (out.checkError()) {
        System.err.println("Erro gravando arquivo " + path);
      }
      out.close();
    }
    catch (IOException ie) {
      System.err.println("Erro criando arquivo: " + path + " - "
        + ie.getMessage());
      ie.printStackTrace();
      System.exit(1);
    }
  }

  /**
   * Converte classe rm texto.
   * 
   * @param type tipo
   * @return texto
   */
  private final String typeToString(Class type) {
    if (type == null) {
      throw new IllegalArgumentException("type == null");
    }
    String s = "";
    if (!type.isArray()) {
      s = type.getName();
      return s.replace('$', '.');
    }
    while (type != null && type.isArray()) {
      type = type.getComponentType();
      s += "[]";
    }
    s = type.getName() + s;
    return s.replace('$', '.');
  }

  /**
   * Indica se o construtor da classe gerada invoca o construtor da super
   * classe.
   * 
   * @return true se o construtor da super classe  invocado false caso
   *         contrrio
   */
  protected abstract boolean invokesSuperClassConstructor();

  /**
   * Indica se lana as excees relacionadas na interface.
   * 
   * @return true se as excees so lanadas false caso contrrio
   */
  protected abstract boolean throwsInterfaceExceptions();

  /**
   * Acrescenta excees especficas a um construtor. Deve ser redefinida, se o
   * construtor da classe tem excees especficas.
   * 
   * @return array com os nomes das excees lanadas
   */
  protected String[] getConstructorExceptions() {
    return null;
  }

  /**
   * Acrescenta as linhas de cdigo a um construtor.
   * 
   * @param code buffer que armazena o cdigo
   * @param paremeterNames nomes dos parmetros passados pelo construtor.
   */
  protected abstract void getConstructorLines(StringBuffer code,
    String[] paremeterNames);

  /**
   * Acrescenta ao cdigo a implementao de um mtodo.
   * 
   * @param code buffer que armazena o cdigo
   * @param method descrio do mtodo a implementar
   * @param parameterNames nomes dos parmetros do mtodo
   */
  protected abstract void getMethodLines(StringBuffer code, Method method,
    String[] parameterNames);

  /**
   * Obtm a super classe da classe gerada.
   * 
   * @return a super classe que o classe gerada ir estender.
   * 
   * @throws ClassNotFoundException em caso de falha na super-classe
   */
  protected abstract Class getSuperClass() throws ClassNotFoundException;

  /**
   * Obtm o nome da super classe da classe gerada.
   * 
   * @return o nome da super classe que o classe gerada ir estender.
   */
  protected abstract String getSuperClassName();

  /**
   * Obtm os pacotes a serem importados pela classe gerada.
   * 
   * @return um array com strings que representam os pacotes a serem importados.
   */
  protected abstract String[] getImports();
}
