/**
 * $Id$
 */
package validations;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import validations.util.ValidatorUtils;
import csbase.logic.CommonProjectInfo;
import csbase.util.FileSystemUtils;
import tecgraf.javautils.core.lng.LNG;

/**
 * Converte os antigos arquivos de controle (.csbase) para o novo formato. Essa
 * converso necessita usar verses alternativas da classe CommonProjectInfo e,
 * por esse motivo, precisa carregar essas verses explicitamente, usando um
 * ClassLoader prprio.
 * 
 * @author Tecgraf
 */
public class PrjInfoValidation extends AbstractValidation {
  /**
   * Sufixo dos arquivos de controle dos projetos (arquivos que contm objetos
   * {@link CommonProjectInfo} serializados).
   */
  private static final String CONTROL_FILE_SUFFIX = ".csbase_project_info";

  /**
   * Lista de todos os arquivos de controle do sistema, a serem convertidos.
   */
  private List<File> allFiles = new ArrayList<File>();

  /**
   * ClassLoader que carrega a primeira verso de CommonProjectInfo.
   */
  private Loader loaderV0 = new Loader();

  /**
   * ClassLoader que carrega a segunda verso de CommonProjectInfo.
   */
  private Loader loaderV1 = new Loader();

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean init() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStartMessage() {
    return "Convertendo arquivos de controle dos projetos.";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean runsOnlyOnce() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void getSpecificFailureMessage(Status inputStatus,
    List<String> errors) {
    switch (inputStatus) {
      case PATCH_FAILED:
        errors.add(LNG.get("validations.ControlFileConversionError"));
        break;
      default:
        errors.add(LNG.get("validations.InvalidStateLower") +
    			inputStatus.toString());	
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getSuccessMessage(Status inputStatus) {
    switch (inputStatus) {
      case VALIDATION_OK:
        return LNG.get("validations.NoConversions");
      case PATCH_OK:
        return LNG.get("validations.InfoConverted");
      default:
        return LNG.get("validations.InvalidStateLower") +
        		inputStatus.toString();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validate() throws ValidationException {
    findFiles();
    if (allFiles.size() == 0) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Preenche a estrutura allFiles com todos os arquivos de controle de projetos
   * do sistema.
   */
  private void findFiles() {
    File baseProjectDir = getProjectDir();
    if (baseProjectDir == null) {
      return;
    }
    File[] usersProjects = FileSystemUtils.getSubDirs(baseProjectDir);
    if (usersProjects == null || usersProjects.length == 0) {
      final String path = baseProjectDir.getAbsolutePath();
      logger.fine(LNG.get("validations.NothingToConvert") + path);
      return;
    }
    for (File userPrj : usersProjects) {
      File[] cfgFiles = getProjectControlFilesForUser(userPrj);
      if (cfgFiles.length == 0) {
        String path = userPrj.getAbsolutePath();
        logger.fine(LNG.get("validations.NoProjectToConvert") + path);
        continue;
      }
      for (File cfgFile : cfgFiles) {
        allFiles.add(cfgFile);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean backupData() {
    try {
      for (File file : allFiles) {
        File projectFile = file.getParentFile();
        File userPrjsFile = projectFile.getParentFile();
        File targetDir = new File(this.getBackupDirPath());
        targetDir = new File(targetDir, projectFile.getName());
        targetDir = new File(targetDir, userPrjsFile.getName());
        if (!targetDir.exists()) {
          targetDir.mkdirs();
        }
        ValidatorUtils.copyFile(file, targetDir, logger, false);
      }
    }
    catch (Exception e) {
      logger.log(Level.SEVERE, LNG.get("validations.BackupError"), e);
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean restoreBackup() {
    try {
      for (File file : allFiles) {
        File projectFile = file.getParentFile();
        File userPrjsFile = projectFile.getParentFile();
        File sourceFile = new File(this.getBackupDirPath());
        sourceFile = new File(sourceFile, projectFile.getName());
        sourceFile = new File(sourceFile, userPrjsFile.getName());
        sourceFile = new File(sourceFile, file.getName());
        ValidatorUtils.copyFile(sourceFile, file, logger, false);
      }
    }
    catch (Exception e) {
      logger.log(Level.SEVERE, LNG.get("validations.RestoreError"), e);
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean applyPatch() {
    int nProjs = 0;
    File baseProjectDir = getProjectDir();
    File[] usersProjects = FileSystemUtils.getSubDirs(baseProjectDir);
    if (usersProjects == null || usersProjects.length == 0) {
      final String path = baseProjectDir.getAbsolutePath();
      logger.fine(LNG.get("validations.NothingToConvert") + path);
      return true;
    }
    for (File userPrj : usersProjects) {
      logger.fine(LNG.get("validations.ConvertingProjects") +
    		  userPrj.getName());
      nProjs = patchUserDir(userPrj, nProjs);
    }
    logger.toConsole(MessageFormat.format(LNG.get(
    		"validations.TotalAnalyzedProjects"),
    		new Object[] { nProjs }));
    return true;
  }

  /**
   * Converte os projetos de um usurio.
   * 
   * @param userDir Diretrio de projetos do usurio.
   * @param index quantidade de projetos analisados.
   * @return uma nova quantidade de projetos analisados.
   */
  private int patchUserDir(File userDir, int index) {
    int i = index;
    File[] cfgFiles = getProjectControlFilesForUser(userDir);
    if (cfgFiles.length == 0) {
      String path = userDir.getAbsolutePath();
      logger.fine(LNG.get("validations.NoProjectToConvert") + path);
      return i;
    }
    for (File cfgFile : cfgFiles) {
      patchFile(cfgFile);
      i = i + 1;
      if (i % 100 == 0) {
        logger.toConsole(LNG.get("validations.NumAnalyzedProjects"));
      }
    }
    return i;
  }

  /**
   * Obtm os arquivo de controle para todos os projetos de um determinado
   * usurio.
   * 
   * @param userDir Diretrio de projetos do usurio.
   * @return Array com os arquivos de controle de todos os projetos do usurio.
   */
  private File[] getProjectControlFilesForUser(File userDir) {
    File[] cfgFiles = userDir.listFiles(new FileFilter() {
      @Override
      public boolean accept(File pathname) {
        final boolean isFile = pathname.isFile();
        final String name = pathname.getName();
        return isFile && name.endsWith(CONTROL_FILE_SUFFIX);
      }
    });
    return cfgFiles;
  }

  /**
   * @param cfgFile
   */
  private void patchFile(File cfgFile) {
    // Tentando fazer a converso V0
    String cfgPath = cfgFile.getAbsolutePath();
    try {
      Class<?> c =
        loaderV0.getCPIClass(RESOURCE_DIR_NAME + "/CommonProjectInfo.v0.class");
      Method m = c.getDeclaredMethod("convert", File.class);
      m.invoke(null, cfgFile);
    }
    catch (Throwable e) {
      // Verificando se j est convertido
      ObjectInputStream ois = null;
      try {
        FileInputStream fis = new FileInputStream(cfgFile);
        ois = new ObjectInputStream(fis);
        Object object = ois.readObject();
        ois.close();
        if (object instanceof CommonProjectInfo) {
          logger.log(Level.FINE, LNG.get("validations.ConversionNotNeeded") +
        		  cfgPath);
          return;
        }
        logger.log(Level.SEVERE,MessageFormat.format(
        		LNG.get("validations.ConversionV0NotAppliedAt"),
        		new Object[] { object.getClass(), cfgPath } ));
        return;
      }
      catch (Throwable t) {
        logger.log(Level.SEVERE,LNG.get("validations.ConversionV0NotApplied") +
    		cfgPath, t);
        return;
      }
      finally {
        if (ois != null) {
           try {
             ois.close();
           }
           catch (IOException ioe) {
              logger.log(Level.SEVERE, LNG.get(
        		  "validations.StreamCloseError") + cfgPath);
           }
        }
      }
    }

    // Fazendo converso V1
    try {
      Class<?> c =
        loaderV1.getCPIClass(RESOURCE_DIR_NAME + "/CommonProjectInfo.v1.class");
      Method m = c.getDeclaredMethod("convert", File.class);
      m.invoke(null, cfgFile);
    }
    catch (Throwable e) {
      logger.log(Level.SEVERE, LNG.get("validations.V1Fail") + cfgPath, e);
    }
  }

  /**
   * ClassLoader para carregar as verses modificadas de CommonProjectInfo que
   * fazem a converso dos arquivos persistidos.
   * 
   * @author Tecgraf
   */
  private class Loader extends ClassLoader {
    /**
     * Cache das classes j construdas.
     */
    private Map<String, Class<?>> cache = new HashMap<String, Class<?>>();

    /**
     * Carrega uma verso de csbase.logic.CommonProjectInfo a partir do arquivo
     * indicado.
     * 
     * @param filename O arquivo que contm a definio da classe (bytecode).
     * @return A classe.
     * @throws Exception Em caso de erro.
     */
    Class<?> getCPIClass(String filename) throws Exception {
      Class<?> c = cache.get(filename);
      if (c != null) {
        return c;
      }
      InputStream is =
        PrjInfoValidation.this.getClass().getResourceAsStream(filename);
      if (is == null) {
        throw new RuntimeException(String.format(LNG.get(
        		"validations.ResourceNotfound"), filename ));
      }
      ByteArrayOutputStream os = new ByteArrayOutputStream();
      while (true) {
        int b = is.read();
        if (b == -1) {
          break;
        }
        os.write(b);
      }
      byte[] b = os.toByteArray();
      c = defineClass("csbase.logic.CommonProjectInfo", b, 0, b.length);
      cache.put(filename, c);
      return c;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finish() {
  }
}
