/**
 * $Id: GUIUtils.java 150399 2014-02-26 19:08:39Z oikawa $
 */
package tecgraf.javautils.gui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoManager;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNGKeys;

/**
 * Classe de mtodos utilitrios para a construo de interfaces grficas.
 * 
 * @see GUIUtilsSample
 * 
 * @author Maria Julia e Ana Moura
 */
public class GUIUtils {

  /**
   * Posies possveis da janela em relao ao X (LEFT, CENTER e RIGHT) e ao Y
   * (TOP, MIDDLE e BOTTOM)
   * 
   * Alteraes aqui devem ser refletidas em
   * {@link tecgraf.javautils.gui.GUIUtils#placeWindowAt(Window, Position)}
   * 
   */
  public enum Position {

    /**  esquerda e em cima */
    LEFT_TOP,

    /**  esquerda e no centro */
    LEFT_MIDDLE,

    /**  esquerda e em baixo */
    LEFT_BOTTOM,

    /** No meio e em cima */
    CENTER_TOP,

    /** No meio e no centro */
    CENTER_MIDDLE,

    /** No meio e em baixo */
    CENTER_BOTTOM,

    /**  direita e em cima */
    RIGHT_TOP,

    /**  direita e no centro */
    RIGHT_MIDDLE,

    /**  direita e em baixo */
    RIGHT_BOTTOM

  }

  /**
   * Cor de fundo default para dilogos (cor de fundo default dos painis).
   */
  public final static Color DEFAULT_PANEL_BACKGROUND_COLOR = UIManager
    .getColor("Panel.background");

  /**
   * Cor default para labels desabilitados.
   */
  public final static Color DEFAULT_DISABLED_TEXT_COLOR = UIManager
    .getColor("Label.disabledForeground");

  /**
   * Fonte "plain" default (obtida a partir da fonte default para labels, sem o
   * atributo "bold").
   */
  public final static Font DEFAULT_LABEL_PLAIN_FONT = UIManager.getFont(
    "Label.font").deriveFont(Font.PLAIN);

  /**
   * Ajusta o tamanho de um boto contendo um cone para que ele fique
   * praticamente do mesmo tamanho do seu cone (fora as margens). O mtodo usa
   * a constante <code>Config.IMAGE_BUTTON_MARGIN</code> para definir a margem a
   * ser usada entre a borda do boto e a borda do seu cone interno.
   * 
   * @param button boto a ser ajustado.
   * 
   * @throws IllegalArgumentException se houver erro de parametrizao.
   */
  public static final void trimImageButton(AbstractButton button) {
    if (button == null) {
      throw new IllegalArgumentException("button == null");
    }
    Icon icon = button.getIcon();
    if (icon == null) {
      throw new IllegalArgumentException("icon == null");
    }
    Dimension size =
      new Dimension(icon.getIconWidth() + Config.IMAGE_BUTTON_MARGIN, icon
        .getIconHeight()
        + Config.IMAGE_BUTTON_MARGIN);
    button.setPreferredSize(size);
    button.setMaximumSize(size);
    button.setMinimumSize(size);
  }

  /**
   * Cria um boto com o cone especificado. O tamanho do boto  ajustado para
   * exibir adequadamente a imagem (ver
   * {@link GUIUtils#trimImageButton(AbstractButton)}).
   * 
   * @param icon cone para o boto.
   * 
   * @return boto com o cone criado.
   */
  public static JButton createImageButton(Icon icon) {
    JButton button = new JButton();
    button.setIcon(icon);
    trimImageButton(button);
    return button;
  }

  /**
   * Cria um cone "vazio" (transparente) com dimenses especficas.
   * 
   * @param width - largura
   * @param height - altura
   * @return cone transparente com as dimenses especificadas
   */
  public static Icon createEmptyIcon(int width, int height) {
    return new EmptyIcon(width, height);
  }

  /**
   * Mtodo de convenincia para adicionar um <code>Component</code> a um
   * <code>Container</code> usando um <code>GridBagLayout</code>. O mtodo no
   * verifica a corretude dos parmetros de restrio, ficando tal
   * responsabilidade a cargo do desenvolvedor.
   * 
   * @param container <code>Container</code> que receber o
   *        <code>Component</code>.
   * @param component componente a ser adicionado de acordo com as
   *        especificaes passadas.
   * @param constraints objeto que representa as restries do
   *        <code>GridBagLayout</code>.
   * @param horizontalIndex posio horizontal do elemento na grade.
   * @param verticalIndex posio vertical do elemento na grade.
   * @param colSpan nmero de colunas que o componente ir ocupar.
   * @param rowSpan nmero de linhas que o componente ir ocupar.
   * @param horizontalWeight peso do componente em relao aos demais para
   *        redimensionamentos horizontais.
   * @param verticalWeight peso do componente em relao aos demais para
   *        redimensionamentos verticais.
   * 
   * @throws IllegalArgumentException se o container, o component ou as
   *         constraints forem nulos.
   * 
   * @see GridBagLayout
   */
  public static final void addWithGridBagConstraints(Container container,
    Component component, GridBagConstraints constraints, int horizontalIndex,
    int verticalIndex, int colSpan, int rowSpan, int horizontalWeight,
    int verticalWeight) {
    if (container == null) {
      throw new IllegalArgumentException("container == null");
    }
    if (component == null) {
      throw new IllegalArgumentException("component == null");
    }
    if (constraints == null) {
      throw new IllegalArgumentException("constraints == null");
    }
    constraints.gridx = horizontalIndex;
    constraints.gridy = verticalIndex;
    constraints.gridwidth = colSpan;
    constraints.gridheight = rowSpan;
    constraints.weightx = horizontalWeight;
    constraints.weighty = verticalWeight;
    container.add(component, constraints);
  }

  /**
   * Iguala as dimenses desejadas de um conjunto de componentes usando como
   * base a maior medida em cada dimenso.
   * 
   * @param components conjunto de componentes de tamanhos potencialmente
   *        diferentes.
   * 
   * @throws IllegalArgumentException se houver erro de parametrizao.
   */
  public static final void matchPreferredSizes(JComponent... components)
    throws IllegalArgumentException {
    if (components == null) {
      throw new IllegalArgumentException("components == null");
    }
    Dimension maxSize = new Dimension(0, 0);
    for (int i = 0; i < components.length; i++) {
      if (components[i] == null) {
        throw new IllegalArgumentException("component[" + i + "] == null");
      }
      double maxHeigth =
        Math.max(maxSize.getHeight(), components[i].getPreferredSize()
          .getHeight());
      double maxWidth =
        Math.max(maxSize.getWidth(), components[i].getPreferredSize()
          .getWidth());
      maxSize.setSize(maxWidth, maxHeigth);
    }
    for (int i = 0; i < components.length; i++) {
      components[i].setPreferredSize(maxSize);
    }
  }

  /**
   * Aviso de erro de preenchimento em um boto. O boto apresenta um cone de
   * erro, o sistema toca um "bip" de aviso e a mensagem de erro especificada 
   * exibida. Aps o usurio fechar o dilogo da mensagem, o boto retoma seu
   * aspecto normal.
   * 
   * @param window janela no qual ocorreu o erro. Pode ser <code>null</code>.
   * @param msg mensagem de erro a ser exibida.
   * @param button boto que ir apresentar o cone de erro.
   * 
   * @throws IllegalArgumentException se houver erro de parametrizao.
   */
  public static void showErrorByButton(Window window, String msg, JButton button)
    throws IllegalArgumentException {
    if (msg == null || msg.equals("")) {
      throw new IllegalArgumentException("msg == " + msg);
    }
    if (button == null) {
      throw new IllegalArgumentException("button == null");
    }
    button.setIcon(GUIResources.LABEL_ERROR_ICON);
    Toolkit.getDefaultToolkit().beep();
    JOptionPane.showMessageDialog(window, msg, LNG
      .get(LNGKeys.INPUT_ERROR_MESSAGE), JOptionPane.ERROR_MESSAGE);
    button.requestFocus();
    button.setIcon(null);
  }

  /**
   * Aviso de erro de preenchimento em um <code>JLabel</code> que esteja
   * qualificando um <code>JTextField</code>. O <code>JLabel</code> apresenta um
   * cone de erro, o sistema toca um "bip" de aviso e a mensagem de erro
   * especificada  exibida. Aps o usurio fechar o dilogo da mensagem, o
   * <code>JLabel</code> retoma seu aspecto normal e o foco passa para o
   * <code>JTextField</code>.
   * 
   * @param window janela no qual ocorreu o erro. Pode ser <code>null</code>.
   * @param msg mensagem de erro a ser exibida.
   * @param label <code>JLabel</code> que ir apresentar o cone de erro.
   * @param field <code>JTextField</code> com o texto a ser corrigido.
   * 
   * @throws IllegalArgumentException se houver erro de parametrizao.
   */
  public static void showErrorByTextField(Window window, String msg,
    JLabel label, JTextField field) {
    if (msg == null || msg.equals("")) {
      throw new IllegalArgumentException("msg == " + msg);
    }
    if (label == null) {
      throw new IllegalArgumentException("label == null");
    }
    if (field == null) {
      throw new IllegalArgumentException("field == null");
    }
    label.setIcon(GUIResources.LABEL_ERROR_ICON);
    Toolkit.getDefaultToolkit().beep();
    JOptionPane.showMessageDialog(window, msg, LNG
      .get(LNGKeys.INPUT_ERROR_MESSAGE), JOptionPane.ERROR_MESSAGE);
    field.requestFocus();
    field.setSelectionStart(0);
    field.setSelectionEnd(field.getText().length());
    label.setIcon(null);
  }

  /**
   * Cria um painel "bsico", em forma de grade na forma do mtodo
   * <code>mountBasicGridPanel</code>
   * 
   * @param rows matriz contendo os componentes a serem inseridos no painel.
   *        Cada linha da matriz contm os componentes da linha correspondente
   *        no painel.
   * 
   * @return o <code>JPanel</code> criado.
   * @see #mountBasicGridPanel(JPanel, JComponent[][])
   */
  public static final JPanel createBasicGridPanel(JComponent[][] rows) {
    final JPanel panel = new JPanel();
    mountBasicGridPanel(panel, rows);
    return panel;
  }

  /**
   * Monta um painel "bsico", em forma de grade, diagramado por um
   * <code>GridBagLayout</code>. Este tipo de painel  utilizado, tipicamente,
   * em dilogos que solicitam um conjunto de parmetros alfanumricos. 
   * composto por um nmero varivel de linhas contendo de 1 a 3 (por default)
   * componentes com as seguintes caractersticas:
   * 
   * <ul>
   * <li>O primeiro compontente de cada linha  normalmente um
   * <code>JLabel</code>;</li>
   * <li>O segundo componente  um <code>JTextField</code> ou uma
   * <code>JComboBox</code>;</li>
   * <li>O terceiro componente  um <code>JLabel</code> ou um
   * <code>JButton</code>.</li>
   * <li>Outros componentes (quarto, quinto etc) seguem o padro do terceiro.</li>
   * </ul>
   * 
   * OBSERVAES:
   * 
   * <ul>
   * <li>Linhas nulas ou vazias sero ignoradas.</li>
   * <li>Componentes nulos deixaro simplesmente um espao vazio.</li>
   * <li>O componente central (secundrio) ir ocupar todo o espao disponvel
   * na horizontal (com exceo do espao reservado para o <code>JLabel</code> 
   * sua esquerda).</li>
   * <li>Todos os componentes esto ancorados a Oeste.</li>
   * <li>Todos os componentes tm o mesmo peso em caso de redimensionamento
   * vertical.</li>
   * <li>O mtodo <b>no mais</b> ignorar componentes alm da terceira posio.
   * </li>
   * </ul>
   * 
   * @param panel o painel a ser montado.
   * @param rows matriz contendo os componentes a serem inseridos no painel.
   *        Cada linha da matriz contm os componentes da linha correspondente
   *        no painel.
   * 
   */
  public static final void mountBasicGridPanel(final JPanel panel,
    final JComponent[][] rows) {
    panel.setLayout(new GridBagLayout());
    if (rows == null) {
      return;
    }

    final int T = 12;
    final int TI = 6;
    final int L = 11;
    final int B = 12;
    final int R = 11;

    final int numRows = rows.length;
    for (int rowNumber = 0; rowNumber < numRows; rowNumber++) {
      final JComponent[] row = rows[rowNumber];
      if (row == null) {
        continue;
      }
      final int numCols = row.length;
      for (int colNumber = 0; colNumber < numCols; colNumber++) {
        final JComponent cell = row[colNumber];
        if (cell == null) {
          continue;
        }

        GBC gbc = new GBC(colNumber, rowNumber);
        gbc = gbc.gridheight(1).gridwidth(1);
        gbc = gbc.west().weights(1.0, 0.0);

        if (numRows == 1) {
          gbc = gbc.insets(T, L, B, R);
        }
        else {
          if (rowNumber == 0) {
            gbc = gbc.insets(T, L, 0, R);
          }
          else if (rowNumber == numRows - 1) {
            gbc = gbc.insets(TI, L, B, R);
          }
          else {
            gbc = gbc.insets(TI, L, 0, R);
          }
        }

        if (colNumber == 1) {
          gbc = gbc.horizontal().weightx(100.0);
        }
        else {
          gbc = gbc.none().west();
        }

        panel.add(cell, gbc);
      }
    }
  }

  /**
   * <p>
   * Centraliza uma janela na tela.
   * </p>
   * 
   * <p>
   * OBS: Caso esteja sendo usado para exibir a janela em uma mquina com mais
   * de um monitor, centraliza a janela na tela do monitor principal (padro).
   * </p>
   * 
   * Apenas chama placeWindowAt(window, Position.CENTER_MIDDLE). Foi mantido
   * para garantir compatibilidade com aplicaes que j chamavam esse mtodo,
   * ao invs do placeWindowAt
   * 
   * 
   * @param window A janela a ser centralizada.
   * @see #placeWindowAt(Window, Position)
   * 
   * @throws IllegalArgumentException Caso a janela recebida seja nula.
   */
  public static final void centerOnScreen(Window window) {
    placeWindowAt(window, Position.CENTER_MIDDLE);
  }

  /**
   * Centraliza uma janela em relao a outra janela de referncia. Se a janela
   * de referncia no estiver sendo exibida, a referncia  substituda pela
   * janela que a contm (seu owner) at que seja encontrada uma janela que
   * esteja sendo exibida, ou que se chegue ao fim da sequncia. Se a janela de
   * referncia for nula, ou no houver janela de referncia sendo exibida,
   * centraliza em relao  tela.
   * 
   * @param window janela a ser centralizada.
   * @param refWindow janela de referncia.
   */
  public static void centerWindow(Window window, Window refWindow) {
    Window parentWindow = refWindow;
    while (parentWindow != null && !parentWindow.isVisible()) {
      parentWindow = parentWindow.getOwner();
    }
    Dimension size = null;
    Point local = null;
    if (parentWindow != null) {
      size = parentWindow.getSize();
      local = parentWindow.getLocationOnScreen();
      int x = (size.width / 2) + local.x;
      int y = (size.height / 2) + local.y;
      Dimension wsize = window.getSize();
      x -= wsize.width / 2;
      y -= wsize.height / 2;
      window.setLocation(x, y);
    }
    else {
      centerOnScreen(window);
    }
  }

  /**
   * Coloca uma janela em uma determinada posio na tela
   * 
   * @param window A janela a ser posicionada
   * @param pos A posio em que deve fica (BOTTOM_UP, CENTER_RIGHT etc.). De
   *        acordo com Enumeration
   *        {@link tecgraf.javautils.gui.GUIUtils.Position}
   * @throws IllegalArgumentException Caso a janela recebida seja nula ou caso a
   *         posio seja invlida
   * 
   * 
   * 
   */
  public static final void placeWindowAt(Window window, Position pos) {
    if (window == null) {
      throw new IllegalArgumentException(
        "No  possvel posicionar uma janela nula.");
    }

    Dimension screenSize = GUIUtils.getScreenDimension();
    Dimension windowSize = window.getSize();
    int newX, newY;

    switch (pos) {
      case RIGHT_BOTTOM:
        newX = Math.max((screenSize.width - windowSize.width), 0);
        newY = Math.max((screenSize.height - windowSize.height), 0);
        break;
      case RIGHT_MIDDLE:
        newX = Math.max((screenSize.width - windowSize.width), 0);
        newY = Math.max(((screenSize.height - windowSize.height) / 2), 0);
        break;
      case RIGHT_TOP:
        newX = Math.max((screenSize.width - windowSize.width), 0);
        newY = 0;
        break;
      case CENTER_BOTTOM:
        newX = Math.max(((screenSize.width - windowSize.width) / 2), 0);
        newY = Math.max((screenSize.height - windowSize.height), 0);
        break;
      case CENTER_MIDDLE:
        newX = Math.max(((screenSize.width - windowSize.width) / 2), 0);
        newY = Math.max(((screenSize.height - windowSize.height) / 2), 0);
        break;
      case CENTER_TOP:
        newX = Math.max(((screenSize.width - windowSize.width) / 2), 0);
        newY = 0;
        break;
      case LEFT_BOTTOM:
        newX = 0;
        newY = Math.max((screenSize.height - windowSize.height), 0);
        break;
      case LEFT_MIDDLE:
        newX = 0;
        newY = Math.max(((screenSize.height - windowSize.height) / 2), 0);
        break;
      case LEFT_TOP:
        newX = 0;
        newY = 0;
        break;
      default:
        throw new RuntimeException("Posio invlida: " + pos.name());
    }

    window.setLocation(newX, newY);
  }

  /**
   * <p>
   * Obtm as dimenses da tela do monitor principal.
   * </p>
   * 
   * @return As dimenses da tela do monitor principal.
   */
  public final static Dimension getScreenDimension() {
    GraphicsEnvironment graphicsEnvironment =
      GraphicsEnvironment.getLocalGraphicsEnvironment();
    GraphicsDevice graphicsDevice =
      graphicsEnvironment.getDefaultScreenDevice();
    GraphicsConfiguration graphicsConfiguration =
      graphicsDevice.getDefaultConfiguration();
    Dimension screenSize = graphicsConfiguration.getBounds().getSize();
    return screenSize;
  }

  /**
   * Traduz os componentes do Swing.
   * 
   * @param locale O locale que define o idioma para o qual se deseja traduzir.
   * @param baseNameList Uma lista com os caminhos dos arquivos que contm as
   *        mensagens traduzidas. O nome de cada arquivo  o nome base para o
   *        arquivo conforme indicao na classe {@link ResourceBundle}.
   */
  public static void translateSwingComponents(Locale locale,
    List<?> baseNameList) {
    Iterator<?> baseNameIterator = baseNameList.iterator();
    while (baseNameIterator.hasNext()) {
      ResourceBundle resourceBundle =
        ResourceBundle.getBundle((String) baseNameIterator.next(), locale);
      Enumeration<?> e = resourceBundle.getKeys();
      while (e.hasMoreElements()) {
        String key = (String) e.nextElement();
        UIManager.put(key, resourceBundle.getString(key));
      }
    }
  }

  /**
   * Aplica e gerencia aes de 'desfazer' e 'refazer' no componente de texto
   * indicado.
   * <ul>
   * <li>CTRL + Z = desfazer
   * <li>CTRL + Y = refazer
   * </ul>
   * 
   * @param textcomp - o componente de texto que ser incrementado.
   */
  public static void applyUndoRedoActions(JTextComponent textcomp) {
    final UndoManager undoManager = new UndoManager();
    Document doc = textcomp.getDocument();

    // ouvinte de eventos de 'undo' e 'redo'
    doc.addUndoableEditListener(new UndoableEditListener() {
      @Override
      public void undoableEditHappened(UndoableEditEvent evt) {
        undoManager.addEdit(evt.getEdit());
      }
    });

    // ///////////
    // DESFAZER:
    textcomp.getActionMap().put("Undo", new AbstractAction("Undo") {
      @Override
      public void actionPerformed(ActionEvent evt) {
        try {
          if (undoManager.canUndo()) {
            undoManager.undo();
          }
        }
        catch (CannotUndoException e) {
        }
      }
    });
    textcomp.getInputMap().put(KeyStroke.getKeyStroke("control Z"), "Undo");

    // ///////////
    // REFAZER:
    textcomp.getActionMap().put("Redo", new AbstractAction("Redo") {
      @Override
      public void actionPerformed(ActionEvent evt) {
        try {
          if (undoManager.canRedo()) {
            undoManager.redo();
          }
        }
        catch (CannotRedoException e) {
        }
      }
    });
    textcomp.getInputMap().put(KeyStroke.getKeyStroke("control Y"), "Redo");
  }

  /**
   * Cria um cone a partir de uma imagem no diretrio de resources.
   * 
   * @param iconDir nome do subdiretrio de resources/images que possui o cone
   * @param iconFileName nome do arquivo de imagem que possui o cone
   * @return cone
   */
  static ImageIcon createIcon(String iconDir, String iconFileName) {
    return new ImageIcon(GUIResources.class.getResource(FileUtils.joinPath(
      true, '/', Config.IMAGES_DIRECTORY, iconDir, iconFileName)));
  }
}
