package tecgraf.javautils.gui.calendar;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIResources;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.field.DateDocument;
import tecgraf.javautils.gui.field.DateField;
import tecgraf.javautils.gui.field.DateListener;

/**
 * Painel com campo para data podendo ser um {@link JTextField} ou
 * {@link DateField} por default. Contm um boto para ativar um calendrio
 * visual com o objetivo de atualizar a data do campo para data.
 * 
 */
public class CalendarDateField extends JPanel {

  private ArrayList<CalendarDateFieldListener> listeners;
  private CalendarPanel calendar;
  private JTextField field;
  private JLabel calendarButton;
  private boolean buttonsInCalendarVisible;

  /**
   * Construtor. O campo para data  um {@link DateField} com o
   * <code>locale</code> default do <code>LNG</code>
   * 
   */
  public CalendarDateField() {
    this(null, null);
  }

  /**
   * Construtor
   * 
   * @param locale Locale para a visualizao das datas.
   */
  public CalendarDateField(Locale locale) {
    this(null, locale);
  }

  /**
   * Construtor
   * 
   * @param field Campo para data
   */
  public CalendarDateField(JTextField field) {
    this(field, null);
  }

  /**
   * Construtor
   * 
   * @param field Campo para data
   * @param locale Locale para a visualizao das datas.
   */
  public CalendarDateField(JTextField field, Locale locale) {
    calendar = new CalendarPanel(locale);
    if (field == null) {
      field = new DateField(locale);
      field.setColumns(10);
    }
    else if (field instanceof DateField) {
      field.setLocale(locale);
    }
    this.field = field;
    this.listeners = new ArrayList<CalendarDateFieldListener>();
    buttonsInCalendarVisible = false;

    setLayout(new GridBagLayout());
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.gridx = 0;
    gbc.gridy = 0;
    gbc.fill = GridBagConstraints.HORIZONTAL;
    gbc.weightx = 1;
    gbc.insets = new Insets(0, 0, 0, 0);
    add(field, gbc);
    gbc.gridx = 1;
    gbc.weightx = 0;
    gbc.fill = GridBagConstraints.NONE;
    gbc.anchor = GridBagConstraints.WEST;
    gbc.insets = new Insets(0, 3, 0, 0);
    add(getCalendarButton(), gbc);
  }

  /**
   * Obtm o campo para data
   * 
   * @return {@link JTextField}
   */
  public JTextField getField() {
    return field;
  }

  /**
   * Obtm o calendrio
   * 
   * @return {@link CalendarPanel}
   */
  public CalendarPanel getCalendar() {
    return calendar;
  }

  /**
   * Obtm o boto de calendrio
   * 
   * @return {@link JLabel}
   */
  public JLabel getCalendarButton() {
    if (calendarButton == null) {
      calendarButton = new JLabel();
      calendarButton.setIcon(GUIResources.BUTTON_CALENDAR_ICON);
      calendarButton.addMouseListener(new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent e) {
          if (CalendarDateField.this.calendarButton.isEnabled()) {
            showCalendar(buttonsInCalendarVisible);
          }
        }
      });
    }
    return calendarButton;
  }

  /**
   * Verifica se o campo  um {@link DateField}
   * 
   * @return boolean
   */
  private boolean isDateField() {
    return (field instanceof DateField);
  }

  /**
   * Obtm o valor do campo
   * 
   * @return o valor do campo
   */
  public Date getDate() {
    if (isDateField()) {
      Long date = ((DateField) field).getDate();
      if (date != null) {
        return new Date(date);
      }
      return null;
    }
    return DateDocument.getDate(field.getText(), getLocale());
  }

  /**
   * Obtm o valor do campo em milisegundos.
   * 
   * @return o valor do campo em milisegundos.
   */
  public Long getDateInMillis() {
    Date date = getDate();
    return (date != null) ? new Long(date.getTime()) : null;
  }

  /**
   * Muda o valor do campo. Os listeners da classe
   * {@link CalendarDateFieldListener} so chamados para tratar o evento de
   * mudana na data.
   * 
   * @param date o novo valor do campo
   */
  public void setDate(Date date) {
    Date oldDate = getDate();
    if (date != null) {
      if (isDateField()) {
        ((DateField) field).setDate(date.getTime());
      }
      else {
        field.setText(DateDocument.toString(getLocale(), date.getTime()));
      }
    }
    else {
      field.setText("");
    }
    if (oldDate != null && date != null)
      fireDateChanged(oldDate.getTime(), date.getTime());
  }

  /**
   * Notifica os listeners que houve uma alterao no campo de data.
   * 
   * @param oldDate a data anterior
   * @param newDate a nova data
   */
  private void fireDateChanged(long oldDate, long newDate) {
    int actualMonth;
    int actualYear;
    GregorianCalendar gc = new GregorianCalendar();
    gc.setTimeInMillis(oldDate);
    actualMonth = gc.get(GregorianCalendar.MONTH);
    actualYear = gc.get(GregorianCalendar.YEAR);
    gc.setTimeInMillis(newDate);
    boolean dateInSameMonth = (gc.get(GregorianCalendar.MONTH) == actualMonth);
    boolean dateInSameYear = (gc.get(GregorianCalendar.YEAR) == actualYear);
    CalendarEvent event =
      new CalendarEvent(newDate, dateInSameMonth && dateInSameYear);
    for (CalendarDateFieldListener listener : listeners) {
      listener.dateChanged(event);
    }
  }

  /**
   * Retorna o texto do campo de data.
   * 
   * @return o texto do campo de data
   */
  public String getText() {
    return field.getText();
  }

  /**
   * Verifica se a data est correta
   * 
   * @return boolean
   */
  public boolean isValidDate() {
    if (isDateField()) {
      return ((DateField) field).isValidValue();
    }
    else {
      if (DateDocument.getDate(getText(), getLocale()) == null) {
        return false;
      }
      return true;
    }
  }

  /**
   * Indica se este componente deve ou no ser editvel. Quando este estado
   * muda,  acionado um PropertyChange event do tipo "editable".
   * 
   * @param editable o boolean indicando se deve ou no ser editvel.
   */
  public void setEditable(boolean editable) {
    field.setEditable(editable);
    getCalendarButton().setEnabled(editable);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setEnabled(boolean enabled) {
    super.setEnabled(enabled);
    field.setEnabled(enabled);
    getCalendarButton().setEnabled(enabled);
  }

  /**
   * Carrega o calendrio visual.
   * 
   * @param showButtons Indica se deve exibir botes para fechar e cancelar. Se
   *        os botes no estiverem visveis o <code>dialog</code> retornar a
   *        primeira data clicada.
   * 
   */
  private void showCalendar(boolean showButtons) {
    JFrame frame =
      (JFrame) SwingUtilities.getAncestorOfClass(JFrame.class, this);
    Long value = getDateInMillis();
    if (value != null) {
      calendar.setSelectedDate(value);
    }
    else {
      calendar.setSelectedDate(System.currentTimeMillis());
    }
    String title = LNG.get("javautils.gui.calendar.title");
    if (showButtons) {
      String btOK = LNG.get("javautils.confirm");
      String btCancel = LNG.get("javautils.cancel");
      value = calendar.showAsDialog(frame, title, btOK, btCancel, this);
    }
    else {
      value = calendar.showAsDialog(frame, title, this);
    }
    if (value != null) {
      setDate(new Date(value));
    }
  }

  /**
   * Indica se os botes Confirmar e Cancelar estaro visveis no calendrio.
   * 
   * @return boolean
   */
  public boolean isButtonsInCalendarVisible() {
    return buttonsInCalendarVisible;
  }

  /**
   * Seta se os botes Confirmar e Cancelar estaro visveis no calendrio.
   * 
   * @param visible boolean
   */
  public void setButtonsInCalendarVisible(boolean visible) {
    this.buttonsInCalendarVisible = visible;
  }

  /**
   * Adiciona um listener
   * 
   * @param listener {@link CalendarListener}
   * 
   * @deprecated utilizar mtodo <code>addDateListener</code>
   */
  @Deprecated
  public void addCalendarDateFieldListener(CalendarDateFieldListener listener) {
    listeners.add(listener);
  }

  /**
   * Remove um listener
   * 
   * @param listener {@link CalendarListener}
   * 
   * @deprecated utilizar mtodo <code>removeDateListener</code>
   */
  @Deprecated
  public void removeCalendarDateFieldListener(CalendarDateFieldListener listener) {
    listeners.remove(listener);
  }

  /**
   * Exibe um {@link JDialog} com o campo e boto para exibir o calendrio.
   * 
   * @param owner {@link JFrame}
   * @param title Ttulo para o dialog
   * @param comp {@link Component} usado para localizar o {@link JDialog} na
   *        tela ou <CODE>NULL</CODE> para ser centralizado na tela.
   * @return {@link Long} com a data em milisegundos ou <CODE>NULL</CODE> caso o
   *         {@link JDialog} seja fechado sem escolher nenhum valor.
   */
  public Long showAsDialog(JFrame owner, String title, Component comp) {
    String btOK = LNG.get("javautils.confirm");
    String btCancel = LNG.get("javautils.cancel");
    final JDialog dialog = new JDialog(owner, title, true);
    dialog.setResizable(false);
    dialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        setDate(null);
      }
    });
    JPanel buttonsPanel = new JPanel();
    JButton confirmButton = new JButton();
    confirmButton.setText(btOK);
    confirmButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        dialog.dispose();
      }
    });
    buttonsPanel.add(confirmButton);
    JButton cancelButton = new JButton();
    cancelButton.setText(btCancel);
    cancelButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        setDate(null);
        dialog.dispose();
      }
    });
    buttonsPanel.add(cancelButton);
    JButton[] buttons = { confirmButton, cancelButton };
    GUIUtils.matchPreferredSizes(buttons);
    JPanel bodyPanel = new JPanel();
    bodyPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 0, 10));
    bodyPanel.setLayout(new BorderLayout(10, 0));
    bodyPanel.add(buttonsPanel, BorderLayout.SOUTH);
    bodyPanel.add(new JLabel(GUIResources.LABEL_QUESTION_LARGE_ICON),
      BorderLayout.WEST);
    bodyPanel.add(this, BorderLayout.CENTER);
    dialog.add(bodyPanel);
    dialog.pack();
    dialog.setLocationRelativeTo(comp);
    dialog.setVisible(true);
    return getDateInMillis();
  }

  /**
   * Adiciona um listener de data
   * 
   * @param listener {@link DateListener}
   */
  public void addDateListener(DateListener listener) {
    if (isDateField()) {
      DateField dateField = (DateField) field;
      dateField.addDateListener(listener);
    }
  }

  /**
   * Remove um listener de data
   * 
   * @param listener {@link DateListener}
   */
  public void removeDateListener(DateListener listener) {
    if (isDateField()) {
      DateField dateField = (DateField) field;
      dateField.removeDateListener(listener);
    }
  }

}
