package tecgraf.javautils.gui.field;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

/**
 * Documento usado para manipular mscaras via expresso regular e aceitar
 * complementos de texto.
 */
public abstract class RegexDocument extends PlainDocument {

  /**
   * expresso regular para verificar os caracteres vlidos e a sequncia
   * permitida
   */
  private String regex;

  /** objeto de complemento para o texto */
  private CompleteText completeText;

  /**
   * Construtor
   * <P>
   * A expresso regular define os caracteres vlidos e a sequncia permitida
   * para o campo. Essa expresso permitir ou no que o texto digitado seja
   * aceito. A cada entrada, o texto atual acrescentado da nova entrada ser
   * verificado na expresso regular.
   * <P>
   * Para um campo do tipo NN/N, definido por uma expresso "[0-9]{1,2}/[0-9]",
   * esse mtodo dever retornar uma expresso com grupos no obrigatrios:
   * "([0-9]{1,2})?(/)?([0-9])?", pois dessa maneira permitir que cada caracter
   * da sequncia seja digitado separadamente ou omitindo alguns podendo ser
   * corrigido futuramente. Uma segunda opo seria: "[0-9]{1,2}(/([0-9])?)?" o
   * que tambm permite que cada caracter da sequncia seja digitado
   * separadamente porm obrigatoriamente na ordem NN -> NN/ -> NN/N
   * 
   * @param regex expresso regular
   */
  public RegexDocument(String regex) {
    this(regex, null);
  }

  /**
   * Construtor
   * <P>
   * A expresso regular define os caracteres vlidos e a sequncia permitida
   * para o campo. Essa expresso permitir ou no que o texto digitado seja
   * aceito. A cada entrada, o texto atual acrescentado da nova entrada ser
   * verificado na expresso regular.
   * <P>
   * Para um campo do tipo NN/N, definido por uma expresso "[0-9]{1,2}/[0-9]",
   * esse mtodo dever retornar uma expresso com grupos no obrigatrios:
   * "([0-9]{1,2})?(/)?([0-9])?", pois dessa maneira permitir que cada caracter
   * da sequncia seja digitado separadamente ou omitindo alguns podendo ser
   * corrigido futuramente. Uma segunda opo seria: "[0-9]{1,2}(/([0-9])?)?" o
   * que tambm permite que cada caracter da sequncia seja digitado
   * separadamente porm obrigatoriamente na ordem NN -> NN/ -> NN/N
   * 
   * @param regex expresso regular
   * @param completeText {@link CompleteText} utilizado para completar
   *        automaticamente um texto
   */
  public RegexDocument(String regex, CompleteText completeText) {
    this.regex = regex;
    this.completeText = completeText;
  }

  /**
   * Dispara os listeners
   */
  protected void fireAllListeners(Object oldValue, Object newValue,
    boolean valueIsAdjusting) {
  }

  /**
   * Obtm o valor atual
   * 
   * @return Object
   */
  protected abstract Object getValue();

  @Override
  public void remove(int offs, int len) throws BadLocationException {
    boolean valueIsAdjusting = true;
    if (getCurrentWriter() == null) {
      valueIsAdjusting = false;
    }
    Object oldValue = getValue();
    super.remove(offs, len);
    Object newValue = getValue();
    fireAllListeners(oldValue, newValue, valueIsAdjusting);
  }

  @Override
  public void insertString(int offs, String str, AttributeSet a)
    throws BadLocationException {
    String newText = null;
    if (getLength() > 0) {
      String text = getText(0, getLength());
      if (getCompleteText() != null && offs == getLength()) {
        str = getCompleteText().buildCompletedText(text, str);
      }
      newText =
        text.substring(0, offs) + str + text.substring(offs, getLength());
    }
    else {
      if (getCompleteText() != null && offs == getLength()) {
        str = getCompleteText().buildCompletedText("", str);
      }
      newText = str;
    }
    if (newText.matches(regex)) {
      Object oldValue = getValue();
      super.insertString(offs, str, a);
      Object newValue = getValue();
      fireAllListeners(oldValue, newValue, false);
    }
  }

  /**
   * Retorna o objeto de complemento de texto.
   * 
   * @return CompleteText
   */
  public CompleteText getCompleteText() {
    return completeText;
  }

  /**
   * Seta o objeto de complemento de texto.
   * 
   * @param completeText
   */
  public void setCompleteText(CompleteText completeText) {
    this.completeText = completeText;
  }
}
