/*
 * $Id: PrintableFooter.java 150399 2014-02-26 19:08:39Z oikawa $
 */
package tecgraf.javautils.gui.print;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.print.PageFormat;
import java.text.MessageFormat;

import tecgraf.javautils.core.lng.LNG;

/**
 * Imprime o footer da pgina com a data e hora alinhadas  esquerda e o nmero
 * da pagina  direita. Se a privacidade do relatrio for definida, ser escrito
 * um texto em vermelho  direita um pouco acima do nmero da pgina com o texto
 * (privado, reservado, confidencial).
 */
public class PrintableFooter extends PrintableText {
  /** Texto a ser centralizado no rodap */
  private String centerText;
  /** Texto a ser exibido  direita em vermelho */
  private String privateText;

  /**
   * Construtor.
   * 
   * @param leftText texto a ser escrito antes da data.
   * @param font fonte do footer.
   */
  public PrintableFooter(String leftText, Font font) {
    this(leftText, font, true);
  }

  /**
   * Construtor.
   * 
   * @param leftText texto a ser escrito antes da data.
   * @param font fonte do footer.
   * @param changeFont indica se a fonte pode ser alterada pela configurao.
   */
  public PrintableFooter(String leftText, Font font, boolean changeFont) {
    this(leftText, null, font, changeFont);
  }

  /**
   * Construtor.
   * 
   * @param leftText texto a ser escrito antes da data.
   * @param centerText texto a ser escrito centralizado.
   * @param font fonte do footer.
   */
  public PrintableFooter(String leftText, String centerText, Font font) {
    this(leftText, centerText, font, true);
  }

  /**
   * Construtor.
   * 
   * @param leftText texto a ser escrito antes da data.
   * @param centerText texto a ser escrito centralizado.
   * @param font fonte do footer.
   * @param changeFont indica se a fonte pode ser alterada pela configurao.
   */
  public PrintableFooter(String leftText, String centerText, Font font,
    boolean changeFont) {
    this(leftText, centerText, font, changeFont, null);
  }

  /**
   * Construtor.
   * 
   * @param leftText texto a ser escrito antes da data.
   * @param centerText texto a ser escrito centralizado.
   * @param font fonte do footer.
   * @param changeFont indica se a fonte pode ser alterada pela configurao.
   * @param privateText mensagem que indica a privacidade do relatrio.
   */
  public PrintableFooter(String leftText, String centerText, Font font,
    boolean changeFont, String privateText) {
    super("\n"
      + MessageFormat.format(LNG
        .get("tecgraf.javautils.gui.print.PrintableFooter.left.text"),
        leftText, System.currentTimeMillis()), font, changeFont);
    this.centerText = centerText;
    this.privateText = privateText;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean print(Graphics2D g2, PageFormat format, int pageIndex) {
    boolean fullPrinted = super.print(g2, format, pageIndex);
    if (!fullPrinted) {
      return false;
    }
    fullPrinted = printCenterText(g2, format, pageIndex);
    if (!fullPrinted) {
      return false;
    }
    return printPageNumber(g2, format, pageIndex);
  }

  /**
   * Imprime o texto centralizado.
   * 
   * @param g2 contexto grfico.
   * @param format formato da pgina.
   * @param pageIndex ndice da pgina.
   * 
   * @return verdadeiro se conseguiu imprimir o texto centralizado.
   */
  private boolean printCenterText(Graphics2D g2, PageFormat format,
    int pageIndex) {
    if (centerText == null) {
      return true;
    }
    pt.setLocation(format.getImageableX(), format.getImageableY());
    textTool.getBBox(g2, getText(), pt, TextTool.NORTH_WEST, rect);
    pt.setLocation(rect.getMaxX(), format.getImageableY());
    String text = "\n      " + centerText;
    textTool.getBBox(g2, text, pt, TextTool.NORTH_WEST, rect);
    if (rect.getHeight() > format.getImageableHeight()) {
      return false;
    }
    textTool.draw(g2, text, pt, TextTool.NORTH_WEST);
    return true;
  }

  /**
   * Imprime o nmero da pgina  direita.
   * 
   * @param g2 contexto grfico.
   * @param format formato da pgina.
   * @param pageIndex ndice da pgina.
   * 
   * @return verdadeiro se conseguiu imprimir o nmero da pgina.
   */
  private boolean printPageNumber(Graphics2D g2, PageFormat format,
    int pageIndex) {
    pt.setLocation(format.getImageableX() + format.getImageableWidth(), format
      .getImageableY());

    /*
     * Para que o texto privado no tenha posies alteradas, dependendo do
     * nmero da pgina, utilizo sempre o mesmo nmero para calcular a altura e
     * largura.
     */
    textTool.getBBox(g2, "\n1", pt, TextTool.NORTH_EAST, rect);
    if (rect.getHeight() > format.getImageableHeight()) {
      return false;
    }
    String pageNumber = "\n" + Integer.toString(pageIndex + 1);
    textTool.draw(g2, pageNumber, pt, TextTool.NORTH_EAST);

    if (privateText != null) {
      Font originalFont = g2.getFont();
      Color originalColor = g2.getColor();
      pt.setLocation(pt.getX() - (10 * rect.getWidth()), pt.getY()
        + (rect.getHeight() * 4 / 5));
      g2.setFont(PrintableUI.PRIVATE_PRINT_FONT);
      g2.setColor(PrintableUI.PRIVATE_COLOR_FONT);
      textTool.draw(g2, privateText, pt, TextTool.SOUTH_EAST);
      g2.setFont(originalFont);
      g2.setColor(originalColor);
    }

    return true;
  }
}
