/*
 * $Id: PrintableHeader.java 138821 2013-03-11 17:32:49Z allan $
 */
package tecgraf.javautils.gui.print;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.ImageObserver;
import java.awt.print.PageFormat;

import javax.swing.ImageIcon;

/**
 * Imprime uma header com duas imagens, uma alinhada  esquerda e outra alinhada
 *  direita.
 * 
 * @see PrintableHeaderSample
 */
public class PrintableHeader implements PrintableReportItem, ImageObserver {
  /** Imagem a ser alinhada  esquerda. */
  private ImageIcon leftImage;
  /** Imagem a ser alinhada  direita. */
  private ImageIcon rightImage;
  /** Altura da imagem */
  private float height;
  /** Indica se a imagem foi desenhada */
  private boolean done;
  /** Largura do cabealho (ocupa toda a largura da pgina) */
  private float width;

  /**
   * Construtor.
   * 
   * @param leftImage imagem a ser alinhada  esquerda.
   * @param rightImage imagem a ser alinhada  direita.
   */
  public PrintableHeader(ImageIcon leftImage, ImageIcon rightImage) {
    this.leftImage = leftImage;
    this.rightImage = rightImage;
    if (leftImage == null && rightImage == null) {
      throw new IllegalArgumentException("all image null");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean simulatePrint(Graphics2D g2, PageFormat format, int pageIndex) {
    int leftHeight = leftImage == null ? 0 : leftImage.getIconHeight();
    int rightHeight = rightImage == null ? 0 : rightImage.getIconHeight();
    height = Math.max(leftHeight, rightHeight);
    height += g2.getFontMetrics(PrintableUI.PRINT_FONT).getHeight();
    return (height <= format.getImageableHeight());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean print(Graphics2D g2, PageFormat format, int pageIndex) {
    int leftHeight = leftImage == null ? 0 : leftImage.getIconHeight();
    int rightHeight = rightImage == null ? 0 : rightImage.getIconHeight();
    height = Math.max(leftHeight, rightHeight);
    if (height + g2.getFontMetrics(PrintableUI.PRINT_FONT).getHeight() > format
      .getImageableHeight()) {
      return false;
    }
    width = (float) format.getImageableWidth();

    int x = (int) format.getImageableX();
    int baseY = (int) format.getImageableY();

    if (leftImage != null) {
      int y = baseY + (int) ((height - leftHeight) / 2);
      drawImage(leftImage, g2, x, y);
    }

    if (rightImage != null) {
      int y = baseY + (int) ((height - rightHeight) / 2);
      x += ((int) format.getImageableWidth() - rightImage.getIconWidth());
      drawImage(rightImage, g2, x, y);
    }

    height += g2.getFontMetrics(PrintableUI.PRINT_FONT).getHeight();
    return true;
  }

  /**
   * Desenha uma imagem nas coordenadas passadas.
   * 
   * @param image imagem a ser desenhada.
   * @param g2 contexto grfico.
   * @param x coordenada horizontal.
   * @param y coordenada vertical.
   */
  private void drawImage(ImageIcon image, Graphics2D g2, int x, int y) {
    int height = image.getIconHeight();
    int width = image.getIconWidth();
    Image img = image.getImage();
    done = false;
    boolean drawn = g2.drawImage(img, x, y, width, height, this);
    if (!drawn) {
      while (true) {
        synchronized (this) {
          if (done) {
            break;
          }
        }
        try {
          Thread.sleep(100);
        }
        catch (InterruptedException e) {
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getHeight() {
    return height;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean imageUpdate(Image img, int infoflags, int x,
    int y, int width, int height) {
    done = ((infoflags & (ERROR | FRAMEBITS | ALLBITS)) != 0);
    return !done;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initPrinting(PrintConfiguration configuration) {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getWidth() {
    return width;
  }
}
