/*
 * $Id: TextTool.java 124697 2011-12-07 16:17:34Z letnog $
 */
package tecgraf.javautils.gui.print;

import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * <p>
 * Ferramenta de desenho de textos. Objetos desta classe podem ser usados para
 * desenhar textos em coordenadas de mundo. Os textos podem ter mltiplas linhas
 * e h duas formas de posicionamento: atravs de um ponto de referncia e uma
 * indicao sobre a relao entre este ponto e a caixa envolvente do texto, ou
 * atravs de uma caixa dentro da qual o texto deve ser desenhado.
 * </p>
 * <p>
 * Os mtodos desta classe no so sincronizados. O cdigo cliente deve garantir
 * que todas as chamadas a esta classe sejam feitas dentro da thread do Swing.
 * </p>
 * 
 * @see TextToolSample
 */
public class TextTool {
  /** Texto para indicar a posio norte */
  public static final String NORTH = "n ";
  /** Texto para indicar a posio sul */
  public static final String SOUTH = "s ";
  /** Texto para indicar a posio leste */
  public static final String EAST = "e";
  /** Texto para indicar a posio oeste */
  public static final String WEST = "w";
  /** Texto para indicar a posio nordeste */
  public static final String NORTH_EAST = "ne";
  /** Texto para indicar a posio noroeste */
  public static final String NORTH_WEST = "nw";
  /** Texto para indicar a posio sudeste */
  public static final String SOUTH_EAST = "se";
  /** Texto para indicar a posio sudoeste */
  public static final String SOUTH_WEST = "sw";
  /** Texto para indicar a posio centralizada */
  public static final String CENTER = "c";

  /** Texto para indicar o alinhamento  esquerda */
  public static final String ALIGN_LEFT = "left";
  /** Texto para indicar o alinhamento centralizado */
  public static final String ALIGN_CENTER = "center";
  /** Texto para indicar o alinhamento  direita */
  public static final String ALIGN_RIGHT = "right";

  /** Texto para indicar que no deve ser feita quebra de linha */
  public static final String WRAP_NONE = "none";
  /** Texto para indicar que deve ser feita quebra de linha aps uma palavra */
  public static final String WRAP_WORD = "word";
  /** Texto para indicar que deve ser feita quebra de linha aps um caracter */
  public static final String WRAP_CHAR = "char";

  /** Caracteres especiais */
  private static Set<Character> WHITESPACE = new HashSet<Character>();
  static {
    WHITESPACE.add(' ');
    WHITESPACE.add('\t');
    WHITESPACE.add('\n');
    WHITESPACE.add('\f');
    WHITESPACE.add('\r');
  }

  /** Tamanho do TAB. O default  45 pixels */
  private int TAB_SIZE = 45;
  /** Lista das linhas */
  private List<String> lines = new ArrayList<String>();
  /** Lista das larguras */
  private List<Integer> widths = new ArrayList<Integer>();

  /** Mximo nmero de linhas */
  private int maxNumLines;
  /** Largura mxima */
  private int maxWidth;
  /** Distncia da base da letra at o topo da maioria dos caracteres */
  private int ascent;
  /** Distncia da base da letra at a parte de baixo dos caracteres */
  private int descent;
  /** Altura do fonte */
  private int height;
  /** Mtrica dos fontes */
  private FontMetrics metrics;
  /** Matriz de transformao original */
  private AffineTransform oT;
  /** Matriz de transformao atual */
  private AffineTransform cT;
  /**
   * Matriz original de transformaes. Antes da aplicao de qualquer
   * transformao nesta classe, a matriz de transformaes original deve ser
   * armazenada neste atributo, para ao trmino das transformaes ser
   * restaurada (dentro do mesmo mtodo que salvou o contexto).
   */
  private AffineTransform backupTransform;
  /** Usado para quebrar texto. */
  private StringBuilder wrapText = new StringBuilder();

  /**
   * Inicializa as matrizes de transformao afim que sero utilizadas para
   * posicionar os componentes grficos.
   * 
   * @param oT matriz de transformao original (pode ser nula).
   * @param cT matriz de transformao atual (pode ser nula).
   */
  public void setTransforms(AffineTransform oT, AffineTransform cT) {
    this.oT = oT;
    this.cT = cT;
  }

  /**
   * Armazena a matriz original de transformaes, para permitir que esta seja
   * restaurada aps quaisquer aplicaes de transformaes. Este mtodo
   * funciona em conjunto com seu complemento
   * {@link #restoreTransform(Graphics2D)}, e portanto todo mtodo que salvar a
   * matriz antes de aplicar transformaes  responsvel por restaur-la ao
   * concluir as transformaes, garantindo assim a integridade do contexto.
   * 
   * @param g2d referncia para a superfcie de desenho.
   */
  private void saveTransform(Graphics2D g2d) {
    backupTransform = g2d.getTransform();
  }

  /**
   * Restaura a matriz original de transformaes. Este mtodo funciona em
   * conjunto com seu complemento {@link #saveTransform(Graphics2D)}, e portanto
   * todo mtodo que salvar a matriz antes de aplicar transformaes 
   * responsvel por restaur-la ao concluir as transformaes, garantindo assim
   * a integridade do contexto.
   * 
   * @param g2d referncia para a superfcie de desenho.
   */
  private void restoreTransform(Graphics2D g2d) {
    g2d.setTransform(backupTransform);
  }

  /**
   * Inicializa as variveis necessrias durante o desenho do texto. Em
   * particular, este mtodo quebra o texto nas quebras de linha e preenche as
   * estrutura <code>lines</code>, que guarda as linhas, e <code>widths</code>,
   * que guarda a largura que cada linha ocuparia na tela se desenhada.
   * 
   * Como este mtodo atende tanto ao desenho por ponto de referncia, que
   * sempre desenha todo o texto, quanto ao desenho em caixa, que s desenha a
   * parte de texto que cabe, ele inicializa variveis que podem servir para
   * ambos ou apenas uma dessas formas de desenho.
   * 
   * Se a matriz de transformao denominada "original" no for nula, ela ir
   * substituir a matriz atual da ferramenta de desenho. Deve-se garantir que,
   * antes da invocao deste mtodo, a matriz original tenha sido salva e que a
   * mesma ser restaurada quando as operaes de transformao tiverem sido
   * concludas.
   * 
   * @param g2d ferramenta de desenho.
   * @param text o texto a ser desenhado.
   * @param maxHeight a altura mxima que o texto pode atingir se houver muitas
   *        quebras de linha, ou -1 se no houver limite.
   */
  private void setup(Graphics2D g2d, String text, float maxHeight) {
    initMetrics(g2d);
    // Quebra as linhas, sem descartar eventuais linhas em branco no final
    String[] tempLines = text.split("\n", -1);
    lines.clear();
    widths.clear();
    maxWidth = 0;
    int numLines = tempLines.length;
    if (maxHeight >= 0) {
      maxNumLines = (int) Math.floor(maxHeight / height);
      numLines = Math.min(numLines, maxNumLines);
    }
    for (int i = 0; i < numLines; i++) {
      lines.add(tempLines[i]);
      int width = calculateLineWidth(tempLines[i]);
      widths.add(width);
      maxWidth = Math.max(maxWidth, width);
    }
  }

  /**
   * Inicializa as variveis relacionadas  ferramenta de desenho necessrias
   * durante o desenho do texto.
   * 
   * Se a matriz de transformao denominada "original" no for nula, ela ir
   * substituir a matriz atual da ferramenta de desenho. Deve-se garantir que,
   * antes da invocao deste mtodo, a matriz original tenha sido salva e que a
   * mesma ser restaurada quando as operaes de transformao tiverem sido
   * concludas.
   * 
   * @param g2d ferramenta de desenho.
   */
  private void initMetrics(Graphics2D g2d) {
    if (oT != null) {
      g2d.setTransform(oT);
    }
    metrics = g2d.getFontMetrics();
    ascent = metrics.getAscent();
    descent = metrics.getDescent();
    height = metrics.getHeight();
  }

  /**
   * <p>
   * Calcula a largura de uma linha. Este mtodo leva em conta o deslocamento
   * dos TABs contidos na linha. A marcao de um tab  definida em
   * {@link #TAB_SIZE}.
   * </p>
   * Este mtodo depende do mtodo {@link #setup(Graphics2D, String, float)} ter
   * sido chamado anteriormente.
   * 
   * @param line a linha a ser calculada a largura
   * @return a largura da linha, considerando o deslocamento dos tabs
   */
  private int calculateLineWidth(String line) {
    String[] tokens = line.split("\t");
    if (tokens.length == 1) {
      /* No possui tabs */
      return metrics.stringWidth(tokens[0]);
    }
    int lineWidth = 0;
    for (String str : tokens) {
      int tempWidth = metrics.stringWidth(str);
      if (tempWidth < TAB_SIZE) {
        lineWidth += TAB_SIZE;
      }
      else {
        lineWidth += (((tempWidth / TAB_SIZE) + 1) * TAB_SIZE);
      }
    }
    return lineWidth;
  }

  /**
   * <p>
   * Ajusta o ponto de referncia para que ele aponte para o local de incio do
   * desenho. O ponto fornecido para o Graphics2D desenhar um texto  a baseline
   * deste texto. Como o ponto de referncia  fornecido pelo usurio do mtodo
   * como uma indicao da caixa envolvente, este ponto precisa ser movido para
   * o ponto no qual o texto precisa comear a ser desenhado.
   * </p>
   * <p>
   * O retngulo, se fornecido,  preenchido com a caixa envolvente que o texto
   * ocupar quando desenhado.
   * </p>
   * Este mtodo depende do mtodo {@link #setup(Graphics2D, String, float)} ter
   * sido chamado anteriormente.
   * 
   * 
   * @param base o ponto para o incio do desenho, deve vir preenchido com as
   *        coordenadas do ponto de referncia fornecido pelo usurio.
   * @param ref o ponto de referncia, em relao  caixa envolvente.
   * @param bbox se fornecido,  preenchido com a caixa envolvente.
   */
  private void adjustBasePoint(Point2D base, String ref, Rectangle2D bbox) {
    float dx = 0;
    float dy = 0;
    switch (ref.charAt(0)) {
      case 'n':
        dy = ascent;
        switch (ref.charAt(1)) {
          case 'e':
            dx = -maxWidth;
            break;
          case ' ':
            dx = -maxWidth / 2F;
            break;
        }
        break;
      case 's':
        dy = -descent - height * (lines.size() - 1);
        switch (ref.charAt(1)) {
          case 'e':
            dx = -maxWidth;
            break;
          case ' ':
            dx = -maxWidth / 2F;
            break;
        }
        break;
      case 'c':
        dx = -maxWidth / 2F;
        dy = ascent / 2F - height * (lines.size() - 1) / 2F;
        break;
      case 'e':
        dx = -maxWidth;
        dy = ascent / 2F - height * (lines.size() - 1) / 2F;
        break;
      case 'w':
        dy = ascent / 2F - height * (lines.size() - 1) / 2F;
        break;
    }
    base.setLocation(base.getX() + dx, base.getY() + dy);
    if (bbox != null) {
      bbox.setRect(base.getX(), base.getY() - ascent, maxWidth, lines.size()
        * height);
    }
  }

  /**
   * <p>
   * Determina o ponto fornecido para que ele aponte para a posio na qual a
   * indicada deve comear a ser desenhada.
   * </p>
   * Este mtodo depende do mtodo {@link #setup(Graphics2D, String, float)} ter
   * sido chamado anteriormente.
   * 
   * @param pt o ponto a ser ajustado para a posio de desenho da linha.
   * @param base o ponto de incio do desenho do texto.
   * @param lineIndex o ndice da linha, na lista lines.
   * @param alignment o alinhamento desenhado para o texto.
   */
  private void adjustLinePoint(Point2D pt, Point2D base, int lineIndex,
    String alignment) {
    float dx = 0;
    float dy = height * lineIndex;
    if (alignment == ALIGN_LEFT) {
      dx = 0;
    }
    else if (alignment == ALIGN_CENTER) {
      dx = (maxWidth - widths.get(lineIndex)) / 2F;
    }
    else if (alignment == ALIGN_RIGHT) {
      dx = maxWidth - widths.get(lineIndex);
    }
    pt.setLocation(base.getX() + dx, base.getY() + dy);
  }

  /**
   * Obtm a caixa envolvente, em coordenadas de mundo, com referncia ao ponto
   * fornecido. O ponto pode representar o centro ou um dos pontos cardeais da
   * caixa envolvente. O texto pode conter quebras de linhas.
   * 
   * @param g2d ferramenta de desenho.
   * @param text o texto a ser desenhado.
   * @param p o ponto de referncia.
   * @param ref a relao entre o ponto e caixa envolvente do texto.
   * @param bbox se fornecido,  preenchido com a caixa envolvente do texto.
   */
  public void getBBox(Graphics2D g2d, String text, Point2D p, String ref,
    Rectangle2D bbox) {
    //#TODO ao invs de uma string, esse ponto de referncia no podia ser um
    // Enum?
    if (g2d == null) {
      throw new IllegalArgumentException("g2d == null");
    }
    if (p == null) {
      throw new IllegalArgumentException("p == null");
    }
    if (ref == null || ref.isEmpty()) {
      throw new IllegalArgumentException("ref est vazia");
    }
    if (bbox == null) {
      throw new IllegalArgumentException("bbox == null");
    }
    saveTransform(g2d);
    setup(g2d, text, -1);
    Point2D base = getPoint(p);
    adjustBasePoint(base, ref, bbox);
    adjustBoundingBox(bbox, oT);
    restoreTransform(g2d);
  }

  /**
   * Obtm um ponto a partir de outro, usando as matrizes de transformao.
   * 
   * @param p o ponto.
   * 
   * @return ponto transformado.
   */
  private Point2D getPoint(Point2D p) {
    Point2D pt = new Point2D.Float();
    pt.setLocation(p);
    if (cT != null) {
      cT.transform(pt, pt);
    }
    return pt;
  }

  /**
   * Ajusta o retngulo, usando as matrizes de transformao.
   * 
   * @param bbox o retngulo a ser ajustado.
   * @param aT matriz para transformar o retngulo.
   */
  private void adjustBoundingBox(Rectangle2D bbox, AffineTransform aT) {
    if (aT == null) {
      return;
    }
    double x = bbox.getX();
    double y = bbox.getY();
    Point2D upperLeft = new Point2D.Float();
    Point2D lowerRight = new Point2D.Float();
    upperLeft.setLocation(x, y);
    lowerRight.setLocation(bbox.getWidth() - x, bbox.getHeight() - y);
    aT.transform(upperLeft, upperLeft);
    aT.transform(lowerRight, lowerRight);
    bbox.setFrameFromDiagonal(upperLeft, lowerRight);
  }

  /**
   * Desenha um texto, em coordenadas de mundo, com referncia ao ponto
   * fornecido. O ponto pode representar o centro ou um dos pontos cardeais da
   * caixa envolvente do texto. O texto deve ter apenas uma linha.
   * 
   * @param g2d ferramenta de desenho.
   * @param text o texto a ser desenhado.
   * @param p o ponto de referncia.
   * @param ref a relao entre o ponto e caixa envolvente do texto.
   */
  public void draw(Graphics2D g2d, String text, Point2D p, String ref) {
    draw(g2d, text, p, ref, ALIGN_LEFT);
  }

  /**
   * Desenha um texto, em coordenadas de mundo, com referncia ao ponto
   * fornecido. O ponto pode representar o centro ou um dos pontos cardeais da
   * caixa envolvente do texto. O texto pode conter quebras de linhas.
   * 
   * @param g2d ferramenta de desenho.
   * @param text o texto a ser desenhado.
   * @param p o ponto de referncia.
   * @param ref a relao entre o ponto e caixa envolvente do texto.
   * @param alignment o alinhamento do texto, caso tenha mais de uma linha.
   */
  public void draw(Graphics2D g2d, String text, Point2D p, String ref,
    String alignment) {
    //#TODO ao invs de uma string, esse ponto de referncia no podia ser um
    // Enum?
    if (g2d == null) {
      throw new IllegalArgumentException("g2d == null");
    }
    if (p == null) {
      throw new IllegalArgumentException("p == null");
    }
    if (ref == null || ref.isEmpty()) {
      throw new IllegalArgumentException("ref est vazia");
    }
    saveTransform(g2d);
    setup(g2d, text, -1);
    Point2D base = getPoint(p);
    adjustBasePoint(base, ref, null);
    Point2D pt = getPoint(p);
    for (int lineIndex = 0; lineIndex < lines.size(); lineIndex++) {
      adjustLinePoint(pt, base, lineIndex, alignment);
      drawTabbedString(g2d, lines.get(lineIndex), pt);
    }
    restoreTransform(g2d);
  }

  /**
   * <p>
   * Renderiza o texto especificado em <code>text</code> levando em conta o
   * espaamento do caracter <code>'\t'</code>, definido em {@link #TAB_SIZE}.
   * </p>
   * Este mtodo depende do mtodo {@link #setup(Graphics2D, String, float)} ter
   * sido chamado anteriormente.
   * 
   * @param g2d ferramenta de desenho.
   * @param text o texto a ser renderizado
   * @param startPoint ponto inicial da renderizao
   */
  private void drawTabbedString(Graphics2D g2d, String text, Point2D startPoint) {
    String[] tokens = text.split("\t");
    float dx = (float) startPoint.getX();
    for (String str : tokens) {
      g2d.drawString(str, dx, (float) startPoint.getY());
      int tempWidth = metrics.stringWidth(str);
      if (tempWidth < TAB_SIZE) {
        dx += TAB_SIZE;
      }
      else {
        dx += (((tempWidth / TAB_SIZE) + 1) * TAB_SIZE);
      }
    }
  }

  /**
   * <p>
   * Determina a posio na qual a linha deve ser quebrada de modo a respeitar o
   * tipo de quebra solicitado e caber na largura fornecida. Caso o modo de
   * quebra seja por palavra e nenhum espao seja encontrado de forma a atender
   * a quebra, o tipo  automaticamente trocado para quebra por caracter.
   * </p>
   * <p>
   * Note que apenas a primeira parte da linha precisa caber na largura. Este
   * mtodo no altera as listas lines e widths.
   * </p>
   * Este mtodo depende do mtodo {@link #setup(Graphics2D, String, float)} ter
   * sido chamado anteriormente.
   * 
   * @param lineIndex o ndice da linha, na lista lines.
   * @param boxWidth largura mxima que a linha pode ter.
   * @param wrap tipo de quebra de linha solicitada.
   * 
   * @return a posio na qual a linha deve ser quebrada.
   */
  private int findSplitPos(int lineIndex, int boxWidth, String wrap) {
    String line = lines.get(lineIndex);
    int pos;
    for (pos = line.length() - 1; pos >= 0; pos--) {
      if (wrap == WRAP_WORD) {
        char c = line.charAt(pos);
        if (!WHITESPACE.contains(c)) {
          continue;
        }
      }
      String firstPart = line.substring(0, pos);
      int width = calculateLineWidth(firstPart);
      if (width > boxWidth) {
        continue;
      }
      return pos;
    }
    if (wrap == WRAP_WORD) {
      return findSplitPos(lineIndex, boxWidth, WRAP_CHAR);
    }
    throw new RuntimeException("Erro em TextTool.findCharSplitPos.");
  }

  /**
   * Quebra um texto em linhas de acordo com a largura permitida. O caracter de
   * quebra de linha  <CODE>\n</CODE>. Caso o menor texto, um nico caracter ou
   * uma palavra, no couber sozinho na largura, o mesmo  considerado uma nova
   * linha.
   * 
   * @param g2d ferramenta de desenho.
   * @param text texto a ser quebrado
   * @param boxWidth largura mxima permitida para o texto
   * @param wrapStyle tipo de quebra de linha (<code>WRAP_CHAR</code> ou
   *        <code>WRAP_WORD</code>)
   * @return texto com quebras de linha ou o texto original, caso o tipo de
   *         quebra de linha no seja reconhecido.
   */
  public String wrapText(Graphics2D g2d, String text, int boxWidth,
    String wrapStyle) {
    //#TODO ao invs de uma string, esse wrapStyle no podia ser Enum?
    if (g2d == null) {
      throw new IllegalArgumentException("g2d == null");
    }
    if (boxWidth < 0) {
      throw new IllegalArgumentException("boxWidth < 0");
    }
    if (wrapStyle != TextTool.WRAP_CHAR && wrapStyle != TextTool.WRAP_WORD) {
      return text;
    }
    saveTransform(g2d);
    initMetrics(g2d);
    wrapText.setLength(0);
    //ltima posio de quebra de linha
    int breakPos = 0;
    int startPos = 0;
    for (int pos = 1; pos <= text.length(); pos++) {
      //obtm a nova String      
      String newStr = text.substring(startPos, pos);
      char c = text.charAt(pos - 1);
      //se wrap for por palavra continua at posicionar na quebra de palavras
      if ((c != '\n' && !WHITESPACE.contains(c))
        && wrapStyle == TextTool.WRAP_WORD && pos < text.length()) {
        continue;
      }
      String accumulatedStr = wrapText.substring(breakPos);
      int newStrSize = calculateLineWidth(newStr);
      int totalSize = newStrSize + calculateLineWidth(accumulatedStr);
      if (totalSize > boxWidth) {
        if (wrapText.length() > 0) {
          wrapText.append("\n");
        }
        breakPos = wrapText.length();
      }
      startPos = pos;
      wrapText.append(newStr);
      if (c == '\n') {
        breakPos = wrapText.length();
      }
    }
    restoreTransform(g2d);
    return wrapText.toString();
  }

  /**
   * <p>
   * Quebra a linha indicada em duas, respeitando o tipo de quebra solicitado,
   * de modo que a primeira parte dela caiba na largura fornecida. A segunda
   * parte pode no caber--nesse caso, este mtodo precisa ser chamado de novo
   * para quebr-la.
   * </p>
   * <p>
   * Este mtodo altera as linhas, na lista <code>lines</code>, e suas larguras,
   * na lista <code>widths</code>. Em particular, este mtodo garante que o
   * nmero de linhas no passe do mximo que cabe na caixa envolvente.
   * </p>
   * Este mtodo depende do mtodo {@link #setup(Graphics2D, String, float)} ter
   * sido chamado anteriormente.
   * 
   * @param lineIndex o ndice da linha, na lista lines.
   * @param boxWidth largura mxima que a linha pode ter.
   * @param wrap tipo de quebra de linha solicitada.
   */
  private void splitLine(int lineIndex, int boxWidth, String wrap) {
    int splitPos = findSplitPos(lineIndex, boxWidth, wrap);
    String line = lines.get(lineIndex);
    String firstLine = line.substring(0, splitPos);
    String secondLine = line.substring(splitPos);
    lines.set(lineIndex, firstLine);
    lines.add(lineIndex + 1, secondLine);
    widths.set(lineIndex, calculateLineWidth(firstLine));
    widths.add(lineIndex + 1, calculateLineWidth(secondLine));
    if (lines.size() > maxNumLines) {
      lines.remove(lines.size() - 1);
      widths.remove(widths.size() - 1);
    }
  }

  /**
   * <p>
   * Ajusta as linhas da lista lines de modo a que elas caibam na caixa
   * envolvente fornecida e respeitem o tipo de quebra de linha solicitado pelo
   * usurio. Este mtodo retorna o nmero de caracteres que sero efetivamente
   * desenhados, isto , o nmero de caracteres que cabem na caixa fornecida, j
   * considerando as quebras efetuadas.
   * </p>
   * Este mtodo depende do mtodo {@link #setup(Graphics2D, String, float)} ter
   * sido chamado anteriormente.
   * 
   * @param bbox caixa envolvente dentro da qual o texto deve ser desenhado.
   * @param wrap tipo de quebra de linha solicitada.
   * 
   * @return nmero de caracteres que sero desenhados.
   */
  private int adjustLines(Rectangle2D bbox, String wrap) {
    int boxWidth = (int) Math.floor(bbox.getWidth());
    int lastChar = 0;
    for (int lineIndex = 0; lineIndex < lines.size(); lineIndex++) {
      if (widths.get(lineIndex) > boxWidth && wrap != WRAP_NONE) {
        splitLine(lineIndex, boxWidth, wrap);
        lastChar += lines.get(lineIndex).length(); // quebra de linha forada
      }
      else {
        lastChar += lines.get(lineIndex).length() + 1; // quebra de linha original
      }
    }
    return lastChar;
  }

  /**
   * Desenha um texto, em coordenadas de mundo, dentro da caixa envolvente
   * fornecida. O texto pode ser quebrado, ou no, horizontalmente caso no
   * caiba na caixa, conforme especificado pelo parmetro wrap. No caso de
   * quebrar, a preferncia pode ser dada a quebrar em espaos, ou quebrar
   * palavras no meio, de modo a ocupar mais cada linha. O texto pode conter
   * quebras de linha.
   * 
   * O texto  desenhado a partir do caracter indicado e este mtodo retorna o
   * ndice posterior ao ltimo caracter desenhado. Ou seja, no caso de retorno
   * igual ao nmero de caracteres, o texto foi todo desenhado. Caso contrrio,
   * o texto no coube na caixa e o que falta desenhar comea no ndice
   * retornado.
   * 
   * A altura da caixa fornecida  atualizada para indicar a altura desenhada.
   * 
   * @param g2d ferramenta de desenho.
   * @param text texto a ser desenhado.
   * @param pos posio inicial do texto a desenhar.
   * @param bbox caixa envolvente da rea na qual o texto deve ser desenhado.
   * @param wrap forma de quebrar linhas horizontais que no caibam na caixa.
   * @param alignment alinhamento das linhas.
   * 
   * @return o ndice seguinte ao ltimo caracter desenhado.
   */
  public int draw(Graphics2D g2d, String text, int pos, Rectangle2D bbox,
    String wrap, String alignment) {
    //#TODO ao invs de uma string, esse wrap no podia ser um Enum?
    if (g2d == null) {
      throw new IllegalArgumentException("g2d == null");
    }
    if (pos < 0) {
      throw new IllegalArgumentException("pos < 0");
    }
    if (pos > text.length() - 1) {
      throw new IllegalArgumentException("pos > text.length() - 1");
    }
    if (bbox == null) {
      throw new IllegalArgumentException("bbox == null");
    }
    if (wrap != TextTool.WRAP_CHAR && wrap != TextTool.WRAP_WORD) {
      throw new IllegalArgumentException("wrap desconhecido");
    }
    saveTransform(g2d);
    adjustBoundingBox(bbox, cT);
    text = text.substring(pos);
    setup(g2d, text, (float) bbox.getHeight());
    int lastChar = adjustLines(bbox, wrap);
    Point2D pt = new Point2D.Float();
    for (int lineIndex = 0; lineIndex < lines.size(); lineIndex++) {
      String line = lines.get(lineIndex).trim();
      int width = widths.get(lineIndex);
      setLinePoint(bbox, lineIndex, width, alignment, pt);
      drawTabbedString(g2d, line, pt);
    }
    float drawnHeight = lines.size() * height;
    bbox.setRect(bbox.getX(), bbox.getY(), bbox.getWidth(), drawnHeight);
    adjustBoundingBox(bbox, oT);
    restoreTransform(g2d);
    return lastChar + pos;
  }

  /**
   * Obtm a caixa envolvente, em coordenadas de mundo, como uma caixa contida
   * na caixa fornecida (se o texto no ocupar a caixa toda) ou a prpria caixa,
   * se o texto preencher a caixa ou no couber totalmente na caixa.
   * 
   * @param g2d ferramenta de desenho.
   * @param text o texto a ser desenhado.
   * @param pos posio inicial do texto a desenhar.
   * @param bbox caixa envolvente da rea na qual o texto deve ser desenhado,
   *        sua altura pode ser reduzida para se ajustar ao texto desenhado.
   * @param wrap forma de quebrar linhas horizontais que no caibam na caixa.
   * @param alignment alinhamento das linhas.
   * 
   * @return o ndice seguinte ao ltimo caracter desenhado.
   */
  public int getBBox(Graphics2D g2d, String text, int pos, Rectangle2D bbox,
    String wrap, String alignment) {
    //#TODO ao invs de uma string, esses wrap e alignment no podiam ser Enums?
    if (g2d == null) {
      throw new IllegalArgumentException("g2d == null");
    }
    if (pos < 0) {
      throw new IllegalArgumentException("pos < 0");
    }
    if (bbox == null) {
      throw new IllegalArgumentException("bbox == null");
    }
    if (wrap == null || wrap.isEmpty()) {
      throw new IllegalArgumentException("wrap est vazia");
    }
    if (alignment == null || alignment.isEmpty()) {
      throw new IllegalArgumentException("alignment est vazia");
    }
    saveTransform(g2d);
    adjustBoundingBox(bbox, cT);
    text = text.substring(pos);
    setup(g2d, text, (float) bbox.getHeight());
    int lastChar = adjustLines(bbox, wrap);
    float drawnHeight = lines.size() * height;
    bbox.setRect(bbox.getX(), bbox.getY(), bbox.getWidth(), drawnHeight);
    adjustBoundingBox(bbox, oT);
    restoreTransform(g2d);
    return lastChar + pos;
  }

  /**
   * Obtm as coordenadas do ponto onde a linha deve ser desenhada.
   * 
   * @param bbox retngulo onde a linha deve ser desenhada.
   * @param lineIndex ndice da linha a ser desenhada.
   * @param lineWidth largura da linha a ser desenhada.
   * @param alignment alinhamento da linha.
   * @param pt ponto a ter as coordenadas alteradas.
   */
  private void setLinePoint(Rectangle2D bbox, int lineIndex, int lineWidth,
    String alignment, Point2D pt) {
    float baseX = (float) bbox.getX();
    float baseY = (float) bbox.getY() + ascent;
    float dx = 0;
    float dy = height * lineIndex;
    if (alignment == ALIGN_LEFT) {
      dx = 0;
    }
    else if (alignment == ALIGN_CENTER) {
      dx = ((float) bbox.getWidth() / 2F) - (lineWidth / 2);
    }
    else if (alignment == ALIGN_RIGHT) {
      dx = (float) bbox.getWidth() - lineWidth;
    }
    pt.setLocation(baseX + dx, baseY + dy);
  }

  /**
   * Configura o tamanho do TAB a ser usado na impresso. O valor default  45
   * pixels.
   * 
   * @param size o novo tamanho para o TAB
   */
  public void setTabSize(int size) {
    if (size < 0) {
      throw new IllegalArgumentException("size < 0");
    }
    this.TAB_SIZE = size;
  }

  int getMaxWidth() {
    return maxWidth;
  }
}
