/*
 * Detalhes da ltima alterao:
 * 
 * $Author: pietroguedes $ $Date: 2013-05-16 12:31:56 -0300 (Thu, 16 May 2013) $
 * $Revision: 141061 $
 */
package tecgraf.javautils.gui.wizard;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * <p>
 * Representa um histrico de dados coletados pelo Wizard.
 * </p>
 * 
 * <p>
 * Cada dado coletado pertence a um passo (o passo onde foi coletado). Os passos
 * tm acesso aos seus dados e aos dados dos passos anteriores.
 * </p>
 * 
 * <p>
 * Existe a possiblidade de se inserir dados globais, ou seja, dados
 * pertencentes ao Wizard e que podem ser acessados por qualquer passo.
 * </p>
 * 
 * @author Tecgraf
 */
public final class History implements WizardListener, StepListener {
  /** Um mapa que liga um passo a um mapa de dados. */
  private Map mapSteps;

  /** Um mapa com os dados globais. */
  private Map global;

  /** O resultado da execuo do wizard. */
  private Object result;

  /** Indica se houve retorno no wizard */
  private boolean wasGonePrevious;

  /** A instncia do wizard desse histrico */
  private Wizard wizard;

  /**
   * Indica se o histrico ser limpo em caso de retrocesso do wizard e mudana
   * dos dados
   */
  private boolean resetHistoryOnChange;

  /**
   * Adiciona um dado ao histrico.
   * 
   * @param step O passo dono do dado.
   * @param name O nome do dado.
   * @param information O dado, propriamente dito.
   * 
   * @throws IllegalArgumentException Caso o passo seja nulo.
   */
  public void add(Step step, String name, Object information) {
    if (step == null) {
      throw new IllegalArgumentException("O passo no pode ser nulo.");
    }
    get(step).put(name, information);
  }

  /**
   * Verifica se o histrico contm um dado acessvel pelo passo especificado.
   * 
   * @param step O passo.
   * @param name O nome do dado.
   * 
   * @return true, caso exista o dado com o referido nome, ou false, caso
   *         contrrio.
   */
  public boolean contains(Step step, String name) {
    for (Step currentStep = step; currentStep != null; currentStep =
      currentStep.getPrevious()) {
      if (get(currentStep).containsKey(name)) {
        return true;
      }
    }
    return this.global.containsKey(name);
  }

  /**
   * Obtm um dado acessvel de um passo.
   * 
   * @param step O passo.
   * @param name O nome do dado.
   * 
   * @return O dado, ou null, caso no exista dado.
   * 
   * @throws IllegalArgumentException Caso o nome do dado esteja nulo.
   */
  public Object get(Step step, String name) {
    if (name == null) {
      throw new IllegalArgumentException("O nome do dado no pode ser nulo.");
    }
    for (Step currentStep = step; currentStep != null; currentStep =
      currentStep.getPrevious()) {
      if (get(currentStep).containsKey(name)) {
        return get(currentStep).get(name);
      }
    }
    return this.global.get(name);
  }

  /**
   * Obtm o resultado do wizard.
   * 
   * @return O resultado.
   */
  public Object getResult() {
    return this.result;
  }

  /**
   * Remove um dado acessvel por um passo.
   * 
   * @param step O passo.
   * @param name O nome do dado a ser removido.
   */
  public void remove(Step step, String name) {
    for (Step currentStep = step; currentStep != null; currentStep =
      currentStep.getPrevious()) {
      if (get(currentStep).containsKey(name)) {
        get(currentStep).remove(name);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasCancelled(Step step) {
    clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasClosed(Step step) {
    clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasConfirmed(Step step, Object object) {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasGoneNext(Step step, History history) {
    this.wasGonePrevious = false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasGonePrevious(Step step, History history) {
    this.wasGonePrevious = true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasHappenedException(Step step, WizardException exception) {
    clear();
  }

  /**
   * Apaga os dados de um determinado passo.
   * 
   * @param step O passo.
   */
  void clear(Step step) {
    Map map = get(step);
    map.clear();
  }

  /**
   * Adiciona um dado global.
   * 
   * @param key A chave que identificar o dado.
   * @param datum O dado.
   * 
   * @throws IllegalArgumentException Caso a chave ou o dado estejam nulos.
   */
  public void addGlobal(String key, Object datum) {
    if (key == null) {
      throw new IllegalArgumentException("A chave no pode ser nula.");
    }
    if (datum == null) {
      throw new IllegalArgumentException("O dado no pode ser nulo.");
    }
    this.global.put(key, datum);
  }

  /**
   * Adiciona os dados recebidos nos dados globais do histrico.
   * 
   * @param globalData Os dados a serem inseridos como globais.
   * 
   * @throws IllegalArgumentException Caso o mapa de dados globais esteja nulo.
   */
  public void addGlobal(Map<?, ?> globalData) {
    if (globalData == null) {
      throw new IllegalArgumentException(
        "O mapa de dados globais no pode ser nulo.");
    }
    Iterator<?> keyIterator = globalData.keySet().iterator();
    while (keyIterator.hasNext()) {
      String key = (String) keyIterator.next();
      this.addGlobal(key, globalData.get(key));
    }
  }

  /**
   * Define o resultado do wizard.
   * 
   * @param result O resultado.
   * 
   * @throws IllegalArgumentException Caso o resultado esteja nulo.
   */
  void setResult(Object result) {
    if (result == null) {
      throw new IllegalArgumentException("O resultado no pode ser nulo.");
    }
    this.result = result;
  }

  /**
   * Apaga os dados contidos no histrico.
   */
  private void clear() {
    this.mapSteps.clear();
    this.global.clear();
  }

  /**
   * Limpa os dados globais.
   */
  public void clearGlobalData() {
    this.global.clear();
  }

  /**
   * Obtm um mapa com os dados pertencentes a um determinado passo.
   * 
   * @param step O passo.
   * 
   * @return Um mapa com os dados do passo.
   */
  private Map get(Step step) {
    Map<?, ?> map;
    if (this.mapSteps.containsKey(step)) {
      map = (Map<?, ?>) this.mapSteps.get(step);
    }
    else {
      map = new HashMap();
      this.mapSteps.put(step, map);
    }
    return map;
  }

  /**
   * Limpa o histrico a partir do passo inicial informado
   * 
   * @param stepClass A classe do passo inicial
   */
  private void reset(Class<?> stepClass) {
    Step step = this.wizard.getStep(stepClass);
    while (step != null) {
      this.clear(step);
      step = this.wizard.getStep(step.getNext(this));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void wasChanged(Step step) {
    if (this.wasGonePrevious && this.resetHistoryOnChange) {
      this.reset(step.getClass());
    }
  }

  /**
   * Cria um histrico.
   * 
   * @param wizard A instncia do wizard desse histrico
   * @param resetHistoryOnChange true habilita reset do histrico em caso de
   *        mudana aps um retrocesso no fluxo do wizard
   */
  public History(Wizard wizard, boolean resetHistoryOnChange) {
    this.resetHistoryOnChange = resetHistoryOnChange;
    this.wizard = wizard;
    this.mapSteps = new HashMap();
    this.global = new HashMap();
    this.result = null;
  }
}
