package tecgraf.javautils.gui.table;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

/**
 * Exemplo de uso da classe {@link SortableTable}.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SortableTableSample {

  /**
   * Teste para a classe. Cria uma tabela ordenvel clicando-se nos cabealhos
   * das colunas.
   * 
   * @param args
   */
  public static void main(String[] args) {
    // Tabela
    final TestModel model = new TestModel();
    final SortableTable table = new SortableTable(model);
    table.sort(new RowSorter.SortKey(1, SortOrder.ASCENDING),
      new RowSorter.SortKey(0, SortOrder.ASCENDING));
    table.setNoSortStateEnabled(true);
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    table.setDefaultRenderer(Date.class, new MyDateTableCellRenderer());
    table.adjustColumnWidth();
    JComponent tablePane = new JScrollPane(table);

    // Painel de botes
    final JLabel labelSelectedRow = new JLabel();
    JButton buttonSelectedRow = new JButton("Linha selecionada");
    buttonSelectedRow.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        int viewRow = table.getSelectedRow();
        if (viewRow == -1) {
          return;
        }
        int modelRow = table.convertRowIndexToModel(viewRow);
        labelSelectedRow.setText("View: " + viewRow + " / Model: " + modelRow);
      }
    });
    JButton buttonNewRow = new JButton("Nova linha");
    buttonNewRow.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        model.addRow("F", "Pablo\nPicasso", 23, new Date(5341242343L));
      }
    });
    JButton buttonRemoveRow = new JButton("Remove linha");
    buttonRemoveRow.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        int viewRow = table.getSelectedRow();
        if (viewRow == -1) {
          return;
        }
        int modelRow = table.convertRowIndexToModel(viewRow);
        model.removeRow(modelRow);
      }
    });
    JButton buttonAdjustColumns = new JButton("Ajusta colunas");
    buttonAdjustColumns.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        table.adjustColumnWidth();
      }
    });
    final String ACTIVATE_LINE_WRAP = "Ativa quebra de linha";
    final String DEACTIVATE_LINE_WRAP = "Desativa quebra de linha";
    final JButton buttonLineWrap = new JButton(ACTIVATE_LINE_WRAP);
    buttonLineWrap.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        TableColumn column = table.getColumnModel().getColumn(1);//coluna nomes
        if (!(column.getCellRenderer() instanceof DefaultLineWrapTableCellRenderer)) {
          TableCellRenderer renderer = new DefaultLineWrapTableCellRenderer();
          column.setCellRenderer(renderer);
          buttonLineWrap.setText(DEACTIVATE_LINE_WRAP);
        }
        else {
          column.setCellRenderer(null);
          buttonLineWrap.setText(ACTIVATE_LINE_WRAP);
        }
        table.adjustRowHeight();
      }
    });
    final String DEACTIVATE_TOOLTIP = "Desativa tooltip";
    final String ACTIVATE_TOOLTIP = "Ativa tooltip";
    final JButton buttonEnableToolTip = new JButton(ACTIVATE_TOOLTIP);
    buttonEnableToolTip.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (table.getToolTipEnabled()) {
          table.setToolTipEnabled(false);
          buttonEnableToolTip.setText(ACTIVATE_TOOLTIP);
        }
        else {
          table.setToolTipEnabled(true);
          buttonEnableToolTip.setText(DEACTIVATE_TOOLTIP);
        }
        table.adjustRowHeight();
      }
    });
    final JButton buttonSortColumn = new JButton("Coluna 3 Decrescente");
    buttonSortColumn.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        table.sort(2, SortOrder.DESCENDING);
      }
    });
    final String DEACTIVATE_COMPARATOR = "Comparador original";
    final String ACTIVATE_COMPARATOR = "Comparador personalizado";
    final JButton buttonSetComparator = new JButton(ACTIVATE_COMPARATOR);
    buttonSetComparator.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (table.getComparator(1) instanceof TestComparator) {
          table.setComparator(1, null);
          buttonSetComparator.setText(ACTIVATE_COMPARATOR);
          table.sort(1, SortOrder.ASCENDING);
        }
        else {
          table.setComparator(1, new TestComparator());
          buttonSetComparator.setText(DEACTIVATE_COMPARATOR);
          table.sort(1, SortOrder.DESCENDING);
        }
      }
    });
    JButton buttonBaseColumn = new JButton("Coluna base");
    buttonBaseColumn.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        JOptionPane.showMessageDialog(null, table.getSortedColIndexView());
      }
    });

    JPanel buttonPanel = new JPanel();
    buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.Y_AXIS));
    buttonPanel.add(buttonSelectedRow);
    buttonPanel.add(buttonNewRow);
    buttonPanel.add(buttonRemoveRow);
    buttonPanel.add(buttonAdjustColumns);
    buttonPanel.add(buttonLineWrap);
    buttonPanel.add(buttonEnableToolTip);
    buttonPanel.add(buttonSortColumn);
    buttonPanel.add(buttonSetComparator);
    buttonPanel.add(buttonBaseColumn);

    // Status Bar
    final JLabel labelSortOrder = new JLabel();
    labelSortOrder.setForeground(Color.RED);
    table.addSortableTableListener(new SortableTableListener() {
      @Override
      public void tableSorted(Object source) {
        new Thread() {
          @Override
          public void run() {
            labelSortOrder.setText("Tabela ordenada!");
            try {
              Thread.sleep(1000);
            }
            catch (InterruptedException e) {
              e.printStackTrace();
            }
            labelSortOrder.setText("");
          }
        }.start();
      }
    });
    JPanel searchPanel =
      table.createSearchPanel("Busca", "Ignorar acentos", "Anterior",
        "Prximo", 0);
    JPanel statusBar = new JPanel(new FlowLayout(FlowLayout.LEFT));
    statusBar.add(searchPanel);
    statusBar.add(labelSortOrder);
    statusBar.add(labelSelectedRow);

    // Janela
    JFrame frame = new JFrame("1.6");
    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    frame.add(tablePane, BorderLayout.CENTER);
    frame.add(buttonPanel, BorderLayout.WEST);
    frame.add(statusBar, BorderLayout.SOUTH);
    int width = 650;
    int height = 400;
    frame.setSize(width, height);
    final int FRAME_INTERNAL_MARGIN = 13; // Margem interna a ser descontada
    table.setPreferredScrollableViewportSize(new Dimension(width
      - FRAME_INTERNAL_MARGIN,
      table.getPreferredScrollableViewportSize().height));
    frame.setVisible(true);
  }
  

  /**
   * Classe de comparador para teste.
   */
  private static class TestComparator implements Comparator<String> {
    /**
     * {@inheritDoc}
     */
    @Override
    public int compare(String o1, String o2) {
      if (o1 == null || o1.length() == 0) {
        return -1;
      }
      if (o2 == null || o2.length() == 0) {
        return 1;
      }
      Character c1 = o1.charAt(o1.length() - 1);
      Character c2 = o2.charAt(o2.length() - 1);
      return c1.compareTo(c2);
    }
  }

  /**
   * Classe interna que fornece um modelo qualquer para teste. Note que a nica
   * condio  que todos os elementos desse modelo devem implementar
   * <code>Comparable</code>, o que  o caso da <code>String</code>.
   */
  private static class TestModel extends AbstractTableModel {
    /** Matriz que representa os elementos da tabela */
    List<List<?>> values = new ArrayList<List<?>>();

    /** Array de cabealhos das colunas */
    String[] headers = new String[] { "Col A", "Col B", "Col C", "Col D" };

    /**
     * Constri o modelo.
     */
    public TestModel() {
      values.add(Arrays.asList(new Object[] { "A",
          "Joo\nAna\nPaulo Mrio Achtung Filho", new Double(2.15),
          new Date(23412342343L) }));
      values.add(Arrays.asList(new Object[] { "D", "J\nMac\nMari\nLuca",
          new Double(1.335), new Date(83412342343L) }));
      values.add(Arrays.asList(new Object[] { "g", "Romo\nSelena",
          new Double(2.23), new Date(53412342343L) }));
      values.add(Arrays.asList(new Object[] { "v", "Luis\nLera\nRonald",
          new Double(11), new Date(13412342343L) }));
      values.add(Arrays.asList(new Object[] { "E", "", null,
          new Date(13412302343L) }));
      values.add(Arrays.asList(new Object[] { "S", null, new Double(665.5524),
          new Date(63412342343L) }));
      values.add(Arrays.asList(new Object[] { "Z", null, null, null }));
    }

    /**
     * Adiciona uma linha ao modelo.
     * 
     * @param letter letra.
     * @param names nomes separados por quebras de linha.
     * @param number nmero inteiro ou fracionrio.
     * @param date data.
     */
    public void addRow(String letter, String names, double number, Date date) {
      int newIndex = values.size();
      values.add(Arrays.asList(new Object[] { letter, names, number, date }));
      fireTableRowsInserted(newIndex, newIndex);
    }

    /**
     * Remove uma linha do modelo.
     * 
     * @param modelRow ndice da linha a ser removida.
     */
    public void removeRow(int modelRow) {
      values.remove(modelRow);
      fireTableRowsDeleted(modelRow, modelRow);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCellEditable(int rowIndex, int colIndex) {
      return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      return values.size();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getColumnCount() {
      return values.get(0).size();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
      return values.get(rowIndex).get(columnIndex);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName(int columnIndex) {
      return headers[columnIndex];
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {
      return (values.get(0).get(columnIndex)).getClass();
    }
  }

  /**
   * Classe interna para teste da formatao de datas.
   */
  private static class MyDateTableCellRenderer extends DefaultTableCellRenderer {
    /**
     * {@inheritDoc}
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value,
      boolean isSelected, boolean hasFocus, int row, int column) {
      Component comp =
        super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
          row, column);
      if (value instanceof Date) {
        DateFormat formatter =
          DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT);
        ((DefaultTableCellRenderer) comp).setText(formatter.format(value));
      }
      return comp;
    }
  }  
}
