package tecgraf.javautils.pdfviewer.core;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.ResourceBundle;

import javax.swing.JPanel;
import javax.swing.JScrollPane;

import tecgraf.javautils.pdfviewer.core.listeners.PDFDocumentOpenCloseListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFPageChangedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomChangedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomFitPolicyChangedListener;

import com.sun.pdfview.PDFFile;

/**
 * Painel que exibe um PDF dado a partir de um {@link InputStream}.
 * 
 * Contm mtodos para:
 * <ul>
 * <li>trocar a pgina
 * <li>atribuir zoom e
 * <li>mudar o modo de zoom ({@link ZoomFitPolicy#WIDTH},
 * {@link ZoomFitPolicy#HEIGHT} e {@link ZoomFitPolicy#WHOLE_PAGE} )
 * </ul>
 * .
 * 
 * @author Tecgraf
 */
public class PDFCorePanel extends JPanel {

  /**
   * Poltica dvaluee Zoom a ser aplicada
   * 
   * @author Tecgraf
   */
  public enum ZoomFitPolicy {
    /** Faz a pgina ocupar toda largura */
    WIDTH,
    /** Faz a pgina ocupar toda a altura */
    HEIGHT,
    /** Zoom que exibe toda a pgina */
    WHOLE_PAGE,
    /** Zoom livre, controlado pelo usurio */
    FREE
  }

  /**
   * ScrollPane que exibe o PDF
   */
  private final JScrollPane scrollPane = new JScrollPane();

  /**
   * Painel onde  renderizado o PDF dentro de um {@link JScrollPane}
   */
  final private PDFRendererPagePanel rendererPanel = new PDFRendererPagePanel();

  /**
   * Lista de {@link PDFPageChangedListener}
   */
  final private List<PDFPageChangedListener> pageChangeListeners =
    new ArrayList<PDFPageChangedListener>();

  /**
   * Lista de {@link PDFDocumentOpenCloseListener}
   */
  final private List<PDFDocumentOpenCloseListener> pdfDocumentOpenCloseListeners =
    new ArrayList<PDFDocumentOpenCloseListener>();

  /**
   * Lista de {@link PDFZoomChangedListener}
   */
  final private List<PDFZoomChangedListener> pdfZoomChangedListeners =
    new ArrayList<PDFZoomChangedListener>();

  /**
   * Lista de {@link PDFZoomFitPolicyChangedListener}
   */
  final private List<PDFZoomFitPolicyChangedListener> pdfZoomFitChangedListeners =
    new ArrayList<PDFZoomFitPolicyChangedListener>();

  /**
   * Listener que aplica a poltica de zoom ao rendererPanel
   */
  final private ComponentListener applyZoomFitPolicyListerner =
    new ComponentAdapter() {
      @Override
      public void componentResized(ComponentEvent e) {
        applyZoomFitPolicy();
      }

      @Override
      public void componentShown(ComponentEvent e) {
        applyZoomFitPolicy();
      }
    };

  /**
   * Pgina a ser exibida
   */
  private int pageNumber = -1;

  /**
   * Arquivo PDF a ser exibido
   */
  private PDFFile pdfFile;

  /**
   * Poltica de zoom corrente
   */
  private ZoomFitPolicy zoomFitPolicy = ZoomFitPolicy.WHOLE_PAGE;

  /**
   * Bundle a ser utilizado.
   */
  final private ResourceBundle resourceBunble;

  /**
   * Construtor default
   * 
   * @param resourceBundle
   */
  public PDFCorePanel(ResourceBundle resourceBundle) {
    this.resourceBunble = resourceBundle;
    setPreferredSize(new Dimension(600, 400));
    scrollPane.setViewportView(rendererPanel);
    layout(scrollPane);

  }

  /**
   * Faz o layout dos componentes neste JPanel
   * 
   * @param pageView visualizao da pgina
   */
  protected void layout(Component pageView) {
    setLayout(new BorderLayout());
    add(pageView, BorderLayout.CENTER);
  }

  /**
   * Atribui o valor de zoom percentual.
   * 
   * @param percent
   * 
   */
  public void setZoom(double percent) {
    setZoomFitPolicy(ZoomFitPolicy.FREE);
    rendererPanel.setZoom(percent);
    notifyZoomChanged(percent);
  }

  /**
   * Consulta o nvel de zoom.
   * 
   * @return o zoom percentual.
   */
  public double getZoom() {
    return rendererPanel.getZoom();
  }

  /**
   * Carrega um documento PDF no visualizador
   * 
   * @param document documento PDF
   */
  public void loadDocument(PDFDocument document) {

    if (hasOpenFile()) {
      closePDF();
    }

    pdfFile = document.getPDFFile();

    if (getTotalPageNumber() > 0) {
      openPDF();
    }

  }

  /**
   * Adiciona um {@link PDFPageChangedListener}  lista de listeners
   * 
   * @param pageChangeListener {@link PDFPageChangedListener} a ser adicionado
   */
  public void addPageChangeListener(PDFPageChangedListener pageChangeListener) {
    pageChangeListeners.add(pageChangeListener);
  }

  /**
   * Adiciona um {@link PDFDocumentOpenCloseListener}  lista de listeners
   * 
   * @param listener novo listener
   */
  public void addPDFDocumentOpenCloseListener(
    PDFDocumentOpenCloseListener listener) {
    pdfDocumentOpenCloseListeners.add(listener);
  }

  /**
   * Adiciona um {@link PDFZoomChangedListener}  lista de listeners
   * 
   * @param listener novo listener
   */
  public void addPDFZoomChangedListener(PDFZoomChangedListener listener) {
    pdfZoomChangedListeners.add(listener);
  }

  /**
   * Adiciona um {@link PDFZoomFitPolicyChangedListener}  lista de listeners
   * 
   * @param listener novo listener
   */
  public void addPDFZoomFitPolicyChangedListener(
    PDFZoomFitPolicyChangedListener listener) {
    pdfZoomFitChangedListeners.add(listener);
  }

  /**
   * Exibe a pgina corrente e notifica os listeners que a pgina mudou
   */
  protected void showPageFromNumberOneBased() {
    rendererPanel.showPage(pdfFile.getPage(pageNumber + 1));
    notifyPageChanged(pageNumber, pdfFile.getNumPages());
  }

  /**
   * Notifica os {@link PDFPageChangedListener}s que a pgina mudou
   * 
   * @param pageNumber nmero da pgina iniciada em 0
   * @param totalPages total de nmero de pginas do documento
   */
  protected void notifyPageChanged(int pageNumber, int totalPages) {
    for (PDFPageChangedListener pageChangeListener : pageChangeListeners) {
      pageChangeListener.pageChanged(pageNumber, totalPages);
    }
  }

  /**
   * Notifica os {@link PDFZoomFitPolicyChangedListener}s que a pgina mudou
   * 
   * @param percentage nova porcentagem
   */
  protected void notifyZoomChanged(double percentage) {
    for (PDFZoomChangedListener zoomChangedListener : pdfZoomChangedListeners) {
      zoomChangedListener.zoomChanged(percentage);
    }
  }

  /**
   * Notifica os {@link PDFZoomFitPolicyChangedListener}s que a pgina mudou
   * 
   * @param zoomFitPolicy nova poltica de zoom
   */
  protected void notifyZoomFitPolicyChanged(ZoomFitPolicy zoomFitPolicy) {
    for (PDFZoomFitPolicyChangedListener zoomFitPolicyChangedListener : pdfZoomFitChangedListeners) {
      zoomFitPolicyChangedListener.zoomFitPolicyChanged(zoomFitPolicy);
    }
  }

  /**
   * Notifica os {@link PDFDocumentOpenCloseListener}s que o documento foi
   * aberto
   */
  protected void notifyPDFDocumentOpened() {
    for (PDFDocumentOpenCloseListener documentOpenCloseListener : pdfDocumentOpenCloseListeners) {
      documentOpenCloseListener.documentOpened(this);
    }
  }

  /**
   * Notifica os {@link PDFDocumentOpenCloseListener}s que o documento foi
   * aberto
   */
  protected void notifyPDFDocumentClosed() {
    for (PDFDocumentOpenCloseListener documentOpenCloseListener : pdfDocumentOpenCloseListeners) {
      documentOpenCloseListener.documentClosed(this);
    }
  }

  /**
   * Atribui a pgina ser exibida, este nmero  _0_ based, isto , comea em 0.
   * 
   * @param pageNumber nmero da pgina, deve ser maior ou igual a 0 e menor ou
   *        igual ao nmero de pginas total - 1
   */
  public void setPage(int pageNumber) {
    if (pdfFile != null) {
      if (pageNumber >= 0 && pageNumber < pdfFile.getNumPages()) {
        if (pageNumber != this.pageNumber) {
          this.pageNumber = pageNumber;
          showPageFromNumberOneBased();
        }
      }
      else {
        throw new IllegalStateException("Pgina " + pageNumber
          + " est fora do limite, que  " + pageNumber + " >= 0 && "
          + pageNumber + "< " + pdfFile.getNumPages()
          + " (nmero total de pginas)");
      }
    }
    else {
      throw new IllegalStateException("Aquivo PDF no foi carregado");
    }
  }

  /**
   * Retorna verdadeiro se  possvel ir para a prxima pgina, isto , a pgina
   * corrente no  ltima, falso caso contrrio.
   * 
   * @return verdadeiro se  possvel ir para a prxima pgina, isto , a pgina
   *         corrente no  ltima, falso caso contrrio.
   */
  public boolean canGoToNextPage() {
    return pageNumber + 1 < pdfFile.getNumPages();
  }

  /**
   * Retorna verdadeiro se  possvel ir para a pgina anterior, isto , a
   * pgina corrente no  primeira, falso caso contrrio.
   * 
   * @return verdadeiro se  possvel ir para a pgina anterior, isto , a
   *         pgina corrente no  primeira, falso caso contrrio.
   */
  public boolean canGoToPrevPage() {
    return pageNumber - 1 >= 0;
  }

  /**
   * Vai para prxima pgina
   * 
   * @return verdadeiro se foi possvel mudar de pgina, falso caso contrrio
   */
  public boolean goToNextPage() {
    if (canGoToNextPage()) {
      ++pageNumber;
      showPageFromNumberOneBased();
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Vai para a pgina anterior
   * 
   * @return verdade se foi possvel mudar de pgina, falso caso contrrio
   */
  public boolean goToPrevPage() {
    if (canGoToPrevPage()) {
      --pageNumber;
      showPageFromNumberOneBased();
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Atribui e aplica a poltica de zoom
   * 
   * @param fitPolicy {@link ZoomFitPolicy} que representa a poltica de zoom
   */
  public void setZoomFitPolicy(ZoomFitPolicy fitPolicy) {
    if (fitPolicy != zoomFitPolicy) {
      zoomFitPolicy = fitPolicy;
      applyZoomFitPolicy();
      notifyZoomFitPolicyChanged(fitPolicy);
    }
  }

  /**
   * Consulta tamanho da pgina do cdocumento.
   * 
   * @return tamanho
   */
  public Dimension getDocumentPageSize() {
    final Dimension size = rendererPanel.getDefaultPageSize();
    return size;
  }

  /**
   * Aplica a poltica de zoom
   */
  protected void applyZoomFitPolicy() {
    if (zoomFitPolicy == ZoomFitPolicy.FREE) {
      return;
    }

    final Dimension oneHundredPercentZoom = rendererPanel.getDefaultPageSize();

    if (oneHundredPercentZoom.width > 0 && oneHundredPercentZoom.height > 0) {
      /*
       * ATENO!! Ao fazer o fit de WIDTH, HEIGHT existe a possbilidade de
       * aparecer uma scrollbar.
       * 
       * No fit de WIDTH  esperado uma scroll bar vertical, no entanto, ao se
       * colocar a scrollbar vertical, o viewport perder tamanho horizontal e
       * assim colocar uma outra scroll bar horizontal, e para isso 
       * necessrio refazer a conta levando em considerao a largura da
       * scrollbar vertical, para que nunca aparea a scroll bar horizontal. No
       * entanto existe uma constante mgica "4" nas contas que deve ser
       * adicionada ao tamanho da largura, pois se no houver, a scroll bar
       * horizontal aparece de todas as formas.
       * 
       * No fit de HEIGHT o mesmo comportamento descrito acima ocorre para a
       * scroll bar horizontal.
       * 
       * No WHOLE_PAGE, a lgica do fit de WIDTH e de HEIGHT so re-utilizados.
       * Se a altura do painel for maior ou igual a largura,  utilizado a
       * lgica do fit de WIDTH. Se a altura for menor que a largura  utilizado
       * a lgica do fit de HEIGHT.
       */

      double rendererPanelHeight = scrollPane.getHeight() - 4;
      double rendererPanelWidth = scrollPane.getWidth() - 4;

      if (zoomFitPolicy == ZoomFitPolicy.WIDTH
        || (zoomFitPolicy == ZoomFitPolicy.WHOLE_PAGE && rendererPanelHeight >= rendererPanelWidth)) {
        double zoomFactor = rendererPanelWidth / oneHundredPercentZoom.width;
        if (oneHundredPercentZoom.getHeight() * zoomFactor > rendererPanelHeight) {
          double verticalScrollBarWidth = getVerticalScrollBarWidth();
          zoomFactor =
            (rendererPanelWidth - verticalScrollBarWidth)
              / oneHundredPercentZoom.width;
        }
        rendererPanel.setZoom(zoomFactor * 100);
      }
      else if (zoomFitPolicy == ZoomFitPolicy.HEIGHT
        || (zoomFitPolicy == ZoomFitPolicy.WHOLE_PAGE && rendererPanelHeight < rendererPanelWidth)) {
        double zoomFactor = rendererPanelHeight / oneHundredPercentZoom.height;
        if (oneHundredPercentZoom.getWidth() * zoomFactor > rendererPanelWidth) {
          double horizontalScrollBarHeight = getHorizontalScrollBarHeight();
          zoomFactor =
            (rendererPanelHeight - horizontalScrollBarHeight)
              / oneHundredPercentZoom.height;
        }
        rendererPanel.setZoom(zoomFactor * 100);
      }
      notifyZoomChanged(rendererPanel.getZoom());
    }
  }

  /**
   * Retorna a largura da barra de scroll vertical
   * 
   * @return a largura da barra de scroll vertical
   */
  protected int getVerticalScrollBarWidth() {
    /*
     * Aqui  utilizado o MaximumSize pois se pegar o Width direto,  possvel
     * que ele retorne zero caso no esteja visvel
     */
    return (int) scrollPane.getVerticalScrollBar().getMaximumSize().getWidth();

  }

  /**
   * Retorna a altura da barra de scroll horizontal
   * 
   * @return altura da barra de scroll horizontal
   */
  protected int getHorizontalScrollBarHeight() {
    /*
     * Aqui  utilizado o MaximumSize pois se pegar o Height direto,  possvel
     * que ele retorne zero caso no esteja visvel
     */
    return (int) scrollPane.getHorizontalScrollBar().getMaximumSize()
      .getHeight();
  }

  /**
   * Retorna o nmero total de pginas do documento corrente
   * 
   * @return o nmero total de pginas do documento corrente
   */
  public int getTotalPageNumber() {
    if (pdfFile != null) {
      return pdfFile.getNumPages();
    }
    else {
      throw new IllegalStateException(
        "No h como pegar o nmero total de pginas pois no existe arquivo PDF aberto (pdfFile == null)");
    }
  }

  /**
   * Mostra o PDF no viewer, habilitando todo o necessrio para seu
   * funcionamento correto
   */
  protected void openPDF() {
    rendererPanel.addComponentListener(applyZoomFitPolicyListerner);
    notifyPDFDocumentOpened();
    setPage(0);
    zoomFitPolicy = ZoomFitPolicy.FREE;
    setZoomFitPolicy(ZoomFitPolicy.WIDTH);
  }

  /**
   * Fecha o PDF
   */
  public void closePDF() {
    rendererPanel.removeComponentListener(applyZoomFitPolicyListerner);
    notifyPDFDocumentClosed();
    rendererPanel.showPage(null);
    pdfFile = null;
    pageNumber = -1;
  }

  /**
   * Retorna o nmero da pgina corrente iniciando em 0
   * 
   * @return o nmero da pgina corrente iniciando em 0
   */
  public int getPageNumber() {
    return pageNumber;
  }

  /**
   * Retorna o PDFFile aberto no momento. Nota, esse mtodo  package visible e
   * no deve ter sua visibilidade alterada, pois expe a biblioteca usada para
   * implementao
   * 
   * @return o PDFFile aberto no momento.
   */
  PDFFile getPDFFile() {
    return pdfFile;
  }

  /**
   * Retorna true se exitir algum arquivo aberto no momento da chamada, false
   * caso contrrio
   * 
   * @return true se exitir algum arquivo aberto no momento da chamada, false
   *         caso contrrio
   */
  public boolean hasOpenFile() {
    return getPDFFile() != null;
  }

  /**
   * Retorna uma String internacionalizada para dada a chave
   * 
   * @param key chave da string de
   * @return uma String internacionalizada para dada a chave
   */
  public String getString(String key) {
    return resourceBunble.getString(key);
  }
}
