package br.pucrio.tecgraf.soma.job.infrastructure.persistence.message;

import br.pucrio.tecgraf.soma.job.JobHistoryEvent;
import br.pucrio.tecgraf.soma.job.SomaJobHistoryConsumer;
import br.pucrio.tecgraf.soma.job.application.appservice.JobHistoryEventService;
import br.pucrio.tecgraf.soma.job.application.appservice.LostEventAppService;
import br.pucrio.tecgraf.soma.job.domain.model.LostEvent;
import io.confluent.kafka.serializers.KafkaAvroDeserializer;
import io.confluent.kafka.serializers.KafkaAvroDeserializerConfig;
import io.confluent.kafka.serializers.KafkaAvroSerializerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.persistence.RollbackException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

import static org.apache.kafka.clients.consumer.ConsumerConfig.*;

@Service
public class JobHistoryEventReader {

	private final Logger logger = LoggerFactory.getLogger(SomaJobHistoryConsumer.class);

	private JobHistoryEventService jobHistoryEventService;
	private LostEventAppService lostEventAppService;
	private RecordReader recordReader;

	@Autowired
	public JobHistoryEventReader(JobHistoryEventService jobHistoryEventService,
			LostEventAppService lostEventAppService) {
		this.jobHistoryEventService = jobHistoryEventService;
		this.lostEventAppService = lostEventAppService;
		this.recordReader = new RecordReader();
	}

	public void run(String kafkaServer, String schemaRegistryUrl, String statusTopic, String progressTopic,
			String group) throws IOException, InterruptedException {
		KafkaConsumer<String, JobHistoryEvent> consumer = buildKafkaConsumer(kafkaServer, schemaRegistryUrl, group);
		List<String> topics = new ArrayList<>();
		topics.add(statusTopic);
		topics.add(progressTopic);
		consumer.subscribe(topics);
		while (!Thread.currentThread().isInterrupted()) {
			readRecords(consumer, statusTopic);
		}
	}

	protected void readRecords(KafkaConsumer<String, JobHistoryEvent> consumer, String statusTopic)
			throws InterruptedException, IOException {
		ConsumerRecords<String, JobHistoryEvent> records = consumer.poll(Duration.ofMillis(100));
		// TODO Confirmar se os eventos serão processados em lote ou individualmente,
		// isso depende de como será o commit no banco. Aqui eles estão sendo
		// processados individualmente
		logger.debug("Got {} records from Kafka", records.count());
		for (TopicPartition partition : records.partitions()) {
			List<ConsumerRecord<String, JobHistoryEvent>> partitionRecords = records.records(partition);
			for (ConsumerRecord<String, JobHistoryEvent> record : partitionRecords) {
				recordReader.readRecord(consumer, partition, statusTopic, partitionRecords, record);
			}
		}
	}

	protected KafkaConsumer<String, JobHistoryEvent> buildKafkaConsumer(String kafkaServer, String schemaRegistryUrl,
			String group) {
		return new KafkaConsumer<>(buildProperties(kafkaServer, schemaRegistryUrl, group));
	}

	private Properties buildProperties(String kafkaServer, String schemaRegistryUrl, String group) {
		Properties properties = new Properties();
		// kafka bootstrap server
		properties.setProperty(BOOTSTRAP_SERVERS_CONFIG, kafkaServer);
		properties.setProperty(KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class.getName());
		properties.setProperty(VALUE_DESERIALIZER_CLASS_CONFIG, KafkaAvroDeserializer.class.getName());
		properties.setProperty(KafkaAvroSerializerConfig.SCHEMA_REGISTRY_URL_CONFIG, schemaRegistryUrl);
		properties.setProperty(KafkaAvroDeserializerConfig.SPECIFIC_AVRO_READER_CONFIG, "true");

		properties.setProperty(GROUP_ID_CONFIG, group);
		properties.setProperty(ENABLE_AUTO_COMMIT_CONFIG, "false");
		// properties.setProperty(AUTO_COMMIT_INTERVAL_MS_CONFIG, "1000");
		properties.setProperty(AUTO_OFFSET_RESET_CONFIG, "earliest");
		return properties;
	}

	private class RecordReader {

		public void readRecord(KafkaConsumer<String, JobHistoryEvent> consumer, TopicPartition partition,
				String statusTopic, List<ConsumerRecord<String, JobHistoryEvent>> partitionRecords,
				ConsumerRecord<String, JobHistoryEvent> record) throws InterruptedException, IOException {
			while (!Thread.currentThread().isInterrupted()) {
				try {
					long lastOffset = partitionRecords.get(partitionRecords.size() - 1).offset();
					int partitionId = partition.partition();
					logger.info("Job {} at Kafka topic {}:{}:{}", record.key(), record.topic(), partitionId,
							lastOffset);
					jobHistoryEventService.process(record.value(), partitionId, lastOffset);
					consumer.commitSync(Collections.singletonMap(partition, new OffsetAndMetadata(lastOffset + 1)));
					break;
				} catch (org.hibernate.exception.JDBCConnectionException ce) {
					long waitingTime = 1000 * 60 * 5;
					logger.warn("No DB connection. Retrying in {} seconds.", waitingTime);
					Thread.sleep(waitingTime);
				} catch (IOException | RollbackException ex) {
					long lastOffset = partitionRecords.get(partitionRecords.size() - 1).offset();
					int partitionId = partition.partition();
					if (record.topic().equals(statusTopic)) {
						String eventId = getEventId(record.value().getEvent());
						LostEvent lostEvent = new LostEvent(eventId, partitionId, lastOffset, record.value().toString());
						lostEventAppService.saveLostEvent(lostEvent, partitionId, lastOffset);
						logger.error(
								"Error trying to process a kafka message. A lost event with id {} was persisted in the database for further inspection.",
								lostEvent.getId(), ex);
					} else {
						logger.error(
								"Error trying to process a kafka message. The event was not persisted.", ex);
					}
					consumer.commitSync(Collections.singletonMap(partition, new OffsetAndMetadata(lastOffset + 1)));
					break;
				} catch (Exception e) {
					logger.error("Unrecoverable error.", e);
					break;
				}
			}
		}

		private String getEventId(Object event) {
			java.lang.reflect.Method method;
			try {
				method = event.getClass().getMethod("getEventId");
			} catch (SecurityException | NoSuchMethodException e) {
				return null;
			}
			try {
				return (String) method.invoke(event);
			} catch (IllegalArgumentException | IllegalAccessException | InvocationTargetException e) {
				return null;
			}
		}
	}

}
