package br.pucrio.tecgraf.soma.job.application.controller;

import java.util.List;

import jakarta.persistence.NoResultException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.ForbiddenException;
import jakarta.ws.rs.ServiceUnavailableException;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.SecurityContext;

import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.transaction.CannotCreateTransactionException;

import br.pucrio.tecgraf.soma.job.api.MultiflowApi;
import br.pucrio.tecgraf.soma.job.api.model.Multiflow;
import br.pucrio.tecgraf.soma.job.api.model.MultiflowBasicResponse;
import br.pucrio.tecgraf.soma.job.api.model.MultiflowFullResponse;
import br.pucrio.tecgraf.soma.job.api.model.MultiflowPatch;
import br.pucrio.tecgraf.soma.job.application.appservice.MultiflowAppService;

@Component
public class MultiflowController extends MultiflowApi {
  private static final Logger LOG = Logger.getLogger(MultiflowController.class);

  @Autowired private HttpServletRequest request;
  @Autowired private MultiflowAppService multiflowAppService;

  public MultiflowController() {
    super(null);
  }

  @Override
  public Response multiflowPost(Multiflow multiflow, String locale, SecurityContext securityContext) {
    MultiflowBasicResponse multiflowResponse;

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    try {
      multiflowResponse = multiflowAppService.createMultiflow(multiflow);
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error creating a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error creating a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (Exception e) { // Demais erros
      String errorMsg = "Error creating a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.CREATED.value()).entity(multiflowResponse).build();
  }

  @Override
  public Response multiflowGet(String projectId, String locale, SecurityContext securityContext) {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    List<MultiflowFullResponse> multiflows;

    try {
      multiflows = multiflowAppService.findMultiflowsByProjectId(projectId);
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (Exception e) {
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.OK.value()).entity(multiflows).build();
  }

  @Override
  public Response multiflowMultiflowIdGet(Long multiflowId, String locale, SecurityContext securityContext) {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    MultiflowFullResponse multiflow;

    try {
      multiflow = multiflowAppService.findMultiflowById(multiflowId);
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (NoResultException e) {
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.NOT_FOUND.value()).entity(errorMsg).build();
    } catch (Exception e) {
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.OK.value()).entity(multiflow).build();
  }

  @Override
  public Response multiflowMultiflowIdPatch(Long multiflowId, MultiflowPatch multiflowPatch, String locale, SecurityContext securityContext) {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    // TODO: Usar propriedade do OpenApi para configurar uma quantidade mínima de propriedades
    if (!validatePatchProperties(multiflowPatch)) {
      LOG.info("Missing arguments");
      return Response.status(HttpStatus.BAD_REQUEST.value()).build();
    }

    MultiflowFullResponse multiflow;

    try {
      multiflow = multiflowAppService.updateMultiflow(multiflowId, multiflowPatch.getMultiflowName(), multiflowPatch.getReplicaFilePath(), multiflowPatch.getReplicaFileName(), multiflowPatch.getParameterMapping());
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (NoResultException e) {
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.NOT_FOUND.value()).entity(errorMsg).build();
    } catch (Exception e) {
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.OK.value()).entity(multiflow).build();
  }

  @Override
  public Response multiflowMultiflowIdDelete(Long multiflowId, String locale, SecurityContext securityContext) {
      String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
      if (accessToken == null || accessToken.isEmpty()) {
        LOG.info("Empty or null token");
        return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
      }

      try {
        multiflowAppService.deleteMultiflow(multiflowId);
      } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
        String errorMsg = "Error deleting Multiflow: " + e.getMessage();
        LOG.info(errorMsg);
        return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
      } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
        String errorMsg = "Error deleting Multiflow: " + e.getMessage();
        LOG.info(errorMsg);
        return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
      } catch (NoResultException e) {
        String errorMsg = "Error deleting Multiflow: " + e.getMessage();
        LOG.info(errorMsg);
        return Response.status(HttpStatus.NOT_FOUND.value()).entity(errorMsg).build();
      } catch (Exception e) {
        String errorMsg = "Error deleting Multiflow: " + e.getMessage();
        LOG.info(errorMsg);
        return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
      }

      return Response.status(HttpStatus.NO_CONTENT.value()).build();
  }

  private boolean validatePatchProperties(MultiflowPatch multiflowPatch) {
    if (multiflowPatch.getReplicaFileName() == null &&
        multiflowPatch.getReplicaFilePath() == null &&
        multiflowPatch.getParameterMapping() == null &&
        multiflowPatch.getMultiflowName() == null) {
          return false;
        }
    return true;
  }
}
