package br.pucrio.tecgraf.soma.job.application.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.github.tennaito.rsql.misc.SimpleMapper;

import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.application.JsonUndefined;
import br.pucrio.tecgraf.soma.job.domain.dto.JobMapper;
import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.domain.model.JobAlgorithm;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobAlgorithmRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobGroupValCount;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobByIdSpecification;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobsInListSpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.impl.RSQLSpecFactory;

@Service
public class JobService {
  private static final Logger LOG = Logger.getLogger(JobService.class);
  private static final SimpleMapper mapper = new SimpleMapper(2);

  static {
    Map<String, String> jobAttribAliases = new HashMap<>(11);
    jobAttribAliases.put("jobStatus", "statusHistory.status");
    jobAttribAliases.put("jobStatusTimestamp", "statusHistory.timestamp");
    jobAttribAliases.put("algorithmId", "algorithms.algorithmId");
    jobAttribAliases.put("algorithmVersion", "algorithms.algorithmVersion");
    jobAttribAliases.put("algorithmName", "algorithms.algorithmName");
    jobAttribAliases.put("flowNodeId", "algorithms.flowNodeId");
    jobAttribAliases.put("parameterId", "algorithms.parameters.parameterId");
    jobAttribAliases.put("paramLabel", "algorithms.parameters.label");
    jobAttribAliases.put("paramType", "algorithms.parameters.type");
    jobAttribAliases.put("paramValue", "algorithms.parameters.value");
    jobAttribAliases.put("jobDependency", "jobDependencies.jobDependencyRaw");
    mapper.addMapping(Job.class, jobAttribAliases);

    Map<String, String> jobJobAlgorithmAttribAliases = new HashMap<>(4);
    jobJobAlgorithmAttribAliases.put("jobStatus", "job.statusHistory.status");
    jobJobAlgorithmAttribAliases.put("exitStatus", "job.exitStatus");
    jobJobAlgorithmAttribAliases.put("currentStatus", "job.currentStatus");
    jobJobAlgorithmAttribAliases.put("jobDependency", "job.jobDependencies");
    jobJobAlgorithmAttribAliases.put("groupId", "job.groupId");
    mapper.addMapping(JobAlgorithm.class, jobJobAlgorithmAttribAliases);
  }

  @Autowired private JobRepository jobRepository;
  @Autowired private JobAlgorithmRepository jobAlgorithmRepository;

  JobService(JobRepository jobRepository, JobAlgorithmRepository jobAlgorithmRepository) {
    this.jobRepository = jobRepository;
    this.jobAlgorithmRepository = jobAlgorithmRepository;
  }

  @Transactional
  public List<JobAlgorithm> findDistinctAlgorithms(String rsqlQuery) {
    LOG.info("Querying distinct algorithms using the query: %s".formatted(rsqlQuery));

    RSQLSpecFactory<JobAlgorithm> jobAlgoRsqlBuilder = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
    JPASpecification<JobAlgorithm> query = jobAlgoRsqlBuilder.create(rsqlQuery);
    return jobAlgorithmRepository.findDistinct(query);
  }

  @Transactional
  public Job findJobByStringId(String jobId) {
    return this.jobRepository.first(new JobByIdSpecification(jobId));
  }

  @Transactional
  public void editJobComment(String jobId, String newComment) {
    Job job = this.findJobByStringId(jobId);
    job.setDescription(newComment);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobAsDeleted(String jobId) {
    Job job = this.jobRepository.first(new JobByIdSpecification(jobId));
    job.setDeleted(true);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobsAsDeleted(List<String> jobIds) {
    JobsInListSpecification spec = new JobsInListSpecification(jobIds);
    List<Job> jobs = this.jobRepository.find(spec);
    for (Job job : jobs) {
      job.setDeleted(true);
      this.jobRepository.update(job);
    }
  }

  private int adjustOffset(int offset, int limit, int total) {
    // ajusta para última página, caso esteja além dela
    if (offset > 0) {
      if (offset >= total) {
        if (total % limit == 0) {
          offset = total - limit;
        } else {
          offset = total - (total % limit);
        }
      }
    }
    return offset;
  }

  private Sort[] toSortArray(String sortAttribute, boolean ascending) {
    if (sortAttribute == null) {
      return new Sort[0];
    }
    return new Sort[] {new Sort(sortAttribute, ascending)};
  }

  @Transactional
  public void findJobs(
      String rsqlQuery,
      int limit,
      int offset,
      boolean ascending,
      String sortAttribute,
      JobResponse response) {
    LOG.info("Querying jobs using the query: %s".formatted(rsqlQuery));

    RSQLSpecFactory<Job> jobRsqlBuilderCount = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
    JPASpecification<Job> queryCount = jobRsqlBuilderCount.create(rsqlQuery);

    // Consulta para obter o total geral, desconsiderando limit e offset
    int total = Math.toIntExact(jobRepository.countJobs(queryCount));
    JobPagination pagination = response.getPagination();
    pagination.setTotal(total);

    if (total == 0) {
      pagination.setOffset(0);
      response.getData().setJobs(Collections.emptyList());
    } else {
      offset = adjustOffset(offset, limit, total);
      Sort[] sorting = toSortArray(sortAttribute, ascending);
      RSQLSpecFactory<Job> jobRsqlBuilderList = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
      JPASpecification<Job> queryCondition = jobRsqlBuilderList.create(rsqlQuery);

      // Consulta para obter os ids dos jobs filtrados pela condicional RSQL, considerando limit e offset
      List<Long> jobIds = jobRepository.findJobIds(queryCondition, limit, offset, sorting);

      // Consulta para obter os dados ordenados dos jobs filtrados
      List<Job> domainJobList = jobRepository.getJobs(jobIds, sorting);

      pagination.setOffset(offset);
      response
          .getData()
          .setJobs(convertDomainJobListToApiJobList(domainJobList));
    }
  }

  @Transactional
  public void findGroupedJobs(
      String rsqlQuery,
      int limit,
      int offset,
      boolean ascending,
      String sortAttribute,
      JobResponse response) {
    LOG.info("Querying grouped jobs using the query: %s".formatted(rsqlQuery));

    RSQLSpecFactory<Job> jobRsqlBuilderCount = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
    JPASpecification<Job> queryCount = jobRsqlBuilderCount.create(rsqlQuery);
    JobPagination pagination = response.getPagination();

    // Consulta para obter o total geral de grupos + jobs singleton filtrados, desconsiderando limit e offset
    int total = Math.toIntExact(jobRepository.countGroups(queryCount));
    pagination.setTotal(total);

    if (total == 0) {
      pagination.setOffset(0);
      response.getData().setJobs(Collections.emptyList());
    } else {
      offset = adjustOffset(offset, limit, total);
      Sort[] sorting = toSortArray(sortAttribute, ascending);

      RSQLSpecFactory<Job> jobRsqlBuilderList = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
      JPASpecification<Job> queryCondition = jobRsqlBuilderList.create(rsqlQuery);

      // Consulta para obter os jobs filtrados pela condicional RSQL assumindo que todos são grupos e considerando limit e offset
      List<JobGroupValCount> groups = jobRepository.findGroupedJobs(queryCondition, limit, offset, sorting);

      List<String> groupIds = new ArrayList<>(groups.size());
      Map<Long, Integer> jobId2Pos = new HashMap<>(groups.size());
      for (int pos = 0; pos < groups.size(); pos++) {
        JobGroupValCount group = groups.get(pos);
        jobId2Pos.put(group.firstJobId, pos);
        groupIds.add(group.groupId);
      }

      // Consulta a lista de grupos que contém descrições variadas. Remove a necessidade de por count distinct para strings grandes
      List<String> groupsWithDescriptionVariation = jobRepository.findGroupIdsWithDescriptionVariation(groupIds);

      // Consulta para obter os ids dos grupos que não são singleton ao desconsiderar a condicional RSQL
      // (ou seja, possuem mais de um job)
      groupIds = jobRepository.findNotSingletonGroupIds(groupIds);

      // Consulta para obter os dados do 1º job de cada grupo e dos jobs singleton já filtrados anteriormente
      List<Job> jobList = jobRepository.getJobs(jobId2Pos.keySet());
      br.pucrio.tecgraf.soma.job.api.model.Job[] apiJobs =
          new br.pucrio.tecgraf.soma.job.api.model.Job[groups.size()];

      for (Job job : jobList) {
        br.pucrio.tecgraf.soma.job.api.model.Job apiJob = this.convertDomainJobToApiJob(job);
        int i = jobId2Pos.get(job.getId());
        apiJobs[i] = apiJob;

        // Se está na lista de grupos que não são singleton, trata-se do 1º job de um grupo,
        // então transforma dados do job em dados de grupo
        if (groupIds.contains(job.getGroupId())) {
          JobGroupValCount group = groups.get(i);
          apiJob.setIsGroup(true);
          apiJob.setJobId(JsonUndefined.STRING);
          apiJob.setEndTime(JsonUndefined.STRING);
          apiJob.setCpuTime(JsonUndefined.DOUBLE);
          apiJob.setWallclockTime(null);
          apiJob.setRamMemory(JsonUndefined.DOUBLE);
          apiJob.setAlgorithms(JsonUndefined.typedList());
          apiJob.setStatusHistory(JsonUndefined.typedList());
          apiJob.setReplica(null);
          apiJob.setJobDependencies(null);

          apiJob.setSubmissionTime(group.submissionTime.toString());
          apiJob.setLastModifiedTime(group.lastModifiedTime.toString());

          // clear heterogeneous values.
          if (group.projectId > 1) apiJob.setProjectId(JsonUndefined.STRING);
          if (group.jobOwner > 1) apiJob.setJobOwner(JsonUndefined.STRING);
          if (group.automaticallyMachineSelection > 1)
            apiJob.setAutomaticallyMachineSelection(null);
          if (group.numberOfProcesses > 1) apiJob.setNumberOfProcesses(null);
          if (group.numberOfProcessesByMachine > 1)
            apiJob.setNumberOfProcessesByMachine(null);
          if (groupsWithDescriptionVariation.contains(job.getGroupId())) {
              apiJob.setDescription(JsonUndefined.STRING);
          }
          if (group.priority > 1) apiJob.setPriority(null);
          if (group.multipleExecution > 1) apiJob.setMultipleExecution(null);
          if (group.jobType > 1) apiJob.setJobType(null);
          if (group.executionMachine > 1) apiJob.setExecutionMachine(JsonUndefined.STRING);
          if (group.exitCode > 1) apiJob.setExitCode(null);
          if (group.guiltyNodeId > 1) apiJob.setGuiltyNodeId(JsonUndefined.STRING);
          if (group.exitStatus > 1) apiJob.setExitStatus(null);
          if (group.flowId > 1) apiJob.setFlowId(JsonUndefined.STRING);
          if (group.flowVersion > 1) apiJob.setFlowVersion(JsonUndefined.STRING);
          if (group.flowName > 1) apiJob.setFlowName(JsonUndefined.STRING);
        }
      }
      pagination.setOffset(offset);
      response.getData().setJobs(Arrays.asList(apiJobs));
    }
  }

  private List<br.pucrio.tecgraf.soma.job.api.model.Job> convertDomainJobListToApiJobList(List<Job> domainJobList) {
    return domainJobList.stream().map(j -> convertDomainJobToApiJob(j)).collect(Collectors.toList());
  }

  private br.pucrio.tecgraf.soma.job.api.model.Job convertDomainJobToApiJob(Job domainJob) {
    return JobMapper.getInstance().map(domainJob);
  }
}
