package br.pucrio.tecgraf.soma.job.log.monitor.impl;

import br.pucrio.tecgraf.soma.job.log.monitor.event.FileChunk;
import br.pucrio.tecgraf.soma.job.log.monitor.event.FileChunkEvent;
import br.pucrio.tecgraf.soma.job.log.monitor.model.LogFileMonitoredResource;
import br.pucrio.tecgraf.soma.job.log.reader.FileReader;
import br.pucrio.tecgraf.soma.job.log.watcher.event.FileEvent;
import br.pucrio.tecgraf.soma.job.log.watcher.impl.DefaultFileWatcher;
import br.pucrio.tecgraf.soma.job.log.watcher.impl.JobLogFileWatcher;
import br.pucrio.tecgraf.soma.job.log.watcher.impl.PollingFileWatcher;
import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatchEventListener;
import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatcher;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitor;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitorListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Flow;

public class JobLogMonitor implements ResourceMonitor {

  public static final String TIMESTAMP_PARAMETER = "timestamp";
  public static final String ENCODING_PARAMETER = "encoding";

  private final Logger LOG = LoggerFactory.getLogger(JobLogMonitor.class);
  private final Map<String, LogFileMonitoredResource> monitoredResourcesByPath;
  private final Map<String, JobLogFileWatcher> watchersByParentPath;
  private final Map<String, Set<ResourceMonitorListener<ResourceMonitorEvent>>> listenersByPath;

  private final ExecutorService threadPool;
  private final FileReader fileReader;

  private boolean enableWatcherPolling;
  private Integer watcherThreadPoolSize;
  private Integer watcherPollingIntervalMillis;

  private boolean allListenersWereRemoved = false;

  /** */
  public JobLogMonitor(
      Integer maxLengthSize,
      boolean enableWatcherPolling,
      Integer watcherThreadPoolSize,
      Integer watcherPollingIntervalMillis,
      Charset defaultCharset,
      boolean enableCharsetDetection) {
    this(
        Executors.newCachedThreadPool(),
        new FileReader(maxLengthSize, defaultCharset, enableCharsetDetection),
        enableWatcherPolling,
        watcherThreadPoolSize,
        watcherPollingIntervalMillis,
        new ConcurrentHashMap<>(),
        new ConcurrentHashMap<>(),
        new ConcurrentHashMap<>());
  }

  /**
   * Constructor with arguments.
   *
   * @param pool
   * @param monitoredResourcesByPath
   * @param watchersByParentPath
   * @param listenersByPath
   */
  public JobLogMonitor(
      ExecutorService pool,
      FileReader fileReader,
      boolean enableWatcherPolling,
      Integer watcherThreadPoolSize,
      Integer watcherPollingIntervalMillis,
      Map<String, LogFileMonitoredResource> monitoredResourcesByPath,
      Map<String, JobLogFileWatcher> watchersByParentPath,
      Map<String, Set<ResourceMonitorListener<ResourceMonitorEvent>>> listenersByPath) {
    this.monitoredResourcesByPath = monitoredResourcesByPath;
    this.watchersByParentPath = watchersByParentPath;
    this.listenersByPath = listenersByPath;
    this.fileReader = fileReader;
    this.threadPool = pool;
    this.enableWatcherPolling = enableWatcherPolling;
    this.watcherThreadPoolSize = watcherThreadPoolSize;
    this.watcherPollingIntervalMillis = watcherPollingIntervalMillis;
  }

  @Override
  public synchronized void addListener(
      String filePath,
      ResourceMonitorListener<ResourceMonitorEvent> listener,
      Map<String, Object> args) {
    LOG.info(
        "Starting monitor with: [filepath={}, timestamp={}, encoding={}]",
        filePath,
        getTimestampFromArgs(args),
        getEncodingFromArgs(args));
    // filePath é absolute path.
    createMonitoredResource(
        filePath, listener, getTimestampFromArgs(args), getEncodingFromArgs(args));

    try {
      createFileWatcher(filePath);

    } catch (FileNotFoundException e) {
      LOG.error("File {} not found", filePath);
      listener.onError(e);
    } catch (IOException e) {
      LOG.error("Cannot create watch service!");
      // TODO: Rever o que temos que remover quando o erro
      listener.onError(e);
      // removeListener(filePath, listener);
    }
    listener.onSubscribe(createSubscription(filePath, listener));
  }

  /**
   * Cria um subscription para remover o listener.
   *
   * @param filePath Path para o arquivo.
   * @param listener O listener que será removido.
   * @return a subscription com a implementação para cancelamento do listener.
   */
  private Flow.Subscription createSubscription(
      String filePath, ResourceMonitorListener<ResourceMonitorEvent> listener) {
    return new Flow.Subscription() {
      @Override
      public void request(long n) {
        // ignore.
      }

      @Override
      public void cancel() {
        removeListener(filePath, listener);
      }
    };
  }

  @Override
  public synchronized void removeListener(
      String filePath, ResourceMonitorListener<ResourceMonitorEvent> listener) {
    LOG.info("Stopping monitor for ID [{}]", filePath);
    this.allListenersWereRemoved = false;
    this.listenersByPath.get(filePath).remove(listener);

    if (this.listenersByPath.get(filePath).isEmpty()) {
      this.listenersByPath.remove(filePath);
      this.monitoredResourcesByPath.remove(filePath);
      final String parentPath = getParentDirFromPath(filePath);
      final JobLogFileWatcher watcher = this.watchersByParentPath.get(parentPath);
      if (watcher != null) {
        watcher.removeMonitoredFilePath(filePath);
        if (watcher.isMonitoredFilePathsEmpty()) {
          try {
            LOG.debug("Closing watcher for events...");
            watcher.close();
          } catch (IOException e) {
            LOG.error("Cannot close the watcher for directory: {}", parentPath, e);
            e.printStackTrace();
          }
          LOG.debug("WATCHER: removing watcher from MAP");
          this.watchersByParentPath.remove(parentPath);
        }
      } else {
        LOG.debug("Ignoring non existent watcher removal for file path {}!", filePath);
      }
      this.allListenersWereRemoved = true;
    }
  }

  @Override
  public List<ResourceMonitorEvent> getEvents(String filePath, Long startSeqnum, Long endSeqnum) {
    final LogFileMonitoredResource resource = this.monitoredResourcesByPath.get(filePath);
    List<ResourceMonitorEvent> events = new LinkedList<>();

    if (startSeqnum != null && endSeqnum != null) {

      Long currSeqnum = startSeqnum;
      while (currSeqnum < endSeqnum) {
        final FileChunk chunk =
            this.readFile(
                filePath, currSeqnum, (int) (endSeqnum - currSeqnum), resource.getEncoding());

        if (chunk == null) {
          break;
        } else {
          ResourceMonitorEvent event = new FileChunkEvent(chunk);
          currSeqnum = event.getEndSeqnum();
          events.add(event);
        }
      }
    }

    return events;
  }

  public void finishMonitoring() {
    LOG.info("Shutting down executor threads...");
    this.threadPool.shutdown();
  }

  public boolean isAllListenersWereRemoved() {
    return this.allListenersWereRemoved;
  }

  private synchronized void createFileWatcher(String filePath) throws IOException {
    String parentPath = getParentDirFromPath(filePath);
    File file = Paths.get(filePath).toFile();
    LOG.debug(
        "File Name and Parent Dir Name: {} : {} ({})",
        new File(filePath).getName(),
        new File(parentPath).getName(),
        filePath);
    if (!file.exists() || !file.isFile()) {
      LOG.debug("File {} not found", filePath);
      throw new FileNotFoundException("File not found");
    } else if (!file.canRead()) {
      LOG.debug("File {} is not readable", filePath);
      throw new FileNotFoundException("File is not readable");
    }

    final JobLogFileWatcher jobLogFileWatcher;
    if (!this.watchersByParentPath.containsKey(parentPath)) {
      jobLogFileWatcher = createJobLogFileWatcher();
      jobLogFileWatcher.addMonitoredFilePath(filePath);
      this.watchersByParentPath.put(parentPath, jobLogFileWatcher);
      threadPool.execute(
          () -> {
            LOG.debug("Thread is executing for filePath: [{}]", filePath);
            try {
              LOG.debug("Adding FileWatchEventListener...");
              jobLogFileWatcher.addFileWatchEventListener(createFileWatchEventListener());
              LOG.debug("Registering WatchService...");
              jobLogFileWatcher.register();
              LOG.debug("Start watch service to take for events...");
              jobLogFileWatcher.startWatch();
            } catch (InterruptedException ie) {
              LOG.error("Watcher was interrupted:", ie);
              notifyErrorsToListeners(filePath, ie);
            } catch (IOException ioe) {
              LOG.error("Watch was not registered", ioe);
              notifyErrorsToListeners(filePath, ioe);
            }
            LOG.debug("Thread is stopping for path: [{}]", filePath);
          });
    } else {
      LOG.debug("Add file Path to watcher: {}", filePath);
      jobLogFileWatcher = this.watchersByParentPath.get(parentPath);
      jobLogFileWatcher.addMonitoredFilePath(filePath);
    }
  }

  protected JobLogFileWatcher createJobLogFileWatcher() throws IOException {
    final IFileWatcher fileWatcher;
    if (this.enableWatcherPolling) {
      fileWatcher = new PollingFileWatcher(this.watcherPollingIntervalMillis);
      LOG.debug("Creating PollingFileWatcher...");
    } else {
      fileWatcher = new DefaultFileWatcher(this.watcherThreadPoolSize);
      LOG.debug("Creating DefaultFileWatcher...");
    }
    return new JobLogFileWatcher(fileWatcher);
  }

  private IFileWatchEventListener createFileWatchEventListener() {
    return new IFileWatchEventListener() {

      @Override
      public void onFileModified(FileEvent event) {
        final File source = event.getSource();
        Path fullPath = Paths.get(source.getAbsolutePath());
        final String fileFullPath = fullPath.toFile().getAbsolutePath();

        final LogFileMonitoredResource resource = monitoredResourcesByPath.get(fileFullPath);
        FileChunk readChunk =
            readFile(fileFullPath, resource.getCurrentTimestamp(), null, resource.getEncoding());
        if (readChunk != null) {
          LOG.debug(
              "Subscription details [file: {}, current timestamp: {}, new timestamp: {}] ",
              source.getName(),
              resource.getCurrentTimestamp(),
              readChunk.getFileLength());
          for (ResourceMonitorListener l : listenersByPath.get(fileFullPath)) {
            LOG.debug("Got monitor listener for {}", fileFullPath);
            resource.setCurrentTimestamp(readChunk.getFileLength());
            threadPool.execute(
                () -> {
                  LOG.debug("Calling monitor listener onNext");
                  l.onNext(new FileChunkEvent(readChunk));
                });
          }
        }
      }
    };
  }

  private void createMonitoredResource(
      String filePath,
      ResourceMonitorListener<ResourceMonitorEvent> listener,
      Long timestamp,
      Charset encoding) {
    if (!monitoredResourcesByPath.containsKey(filePath)) {
      this.monitoredResourcesByPath.put(
          filePath, new LogFileMonitoredResource(filePath, timestamp, encoding));
    }
    if (!this.listenersByPath.containsKey(filePath)) {
      this.listenersByPath.put(filePath, new HashSet<>());
    }
    if (!this.listenersByPath.get(filePath).add(listener)) {
      LOG.debug("Listener for path {} already registered!", filePath);
    }
  }

  private void notifyErrorsToListeners(String filePath, Throwable throwable) {
    final Set<ResourceMonitorListener<ResourceMonitorEvent>> listeners =
        this.listenersByPath.get(filePath);
    if (listeners != null && !listeners.isEmpty()) {
      listeners.forEach(l -> l.onError(throwable));
    }
  }

  private FileChunk readFile(
      String filePath, Long timestamp, Integer length, Charset forcedCharset) {
    try {
      final FileChunk chunk = fileReader.readFile(filePath, timestamp, length, forcedCharset);
      // Verifica se algo foi lido
      if (chunk == null || chunk.getData().length() == 0) {
        return null;
      }
      LOG.debug(
          "FileChunk [file: {}, fileLength:{}, offset:{}, data length: {}] ",
          chunk.getPath().toFile().getName(),
          chunk.getFileLength(),
          chunk.getOffset(),
          chunk.getLength());
      return chunk;
    } catch (IOException e) {
      LOG.error(
          "Cannot read file chunk from file [{}, {}, {}, {}]",
          filePath,
          timestamp,
          fileReader.getMaxLengthSize(),
          e.getMessage());
      notifyErrorsToListeners(filePath, e);
    }
    return null;
  }

  private Long getTimestampFromArgs(Map<String, Object> params) {
    return (Long) params.get(TIMESTAMP_PARAMETER);
  }

  private Charset getEncodingFromArgs(Map<String, Object> params) {
    return (Charset) params.get(ENCODING_PARAMETER);
  }

  private String getParentDirFromPath(String fileAbsPath) {
    return Paths.get(fileAbsPath).getParent().toString();
  }
}
