package br.pucrio.tecgraf.soma.job.log.watcher.impl;

import br.pucrio.tecgraf.soma.job.log.watcher.event.FileEvent;
import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatchEventListener;
import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatcher;
import com.sun.nio.file.SensitivityWatchEventModifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.*;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static java.lang.String.format;
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY;

public class DefaultFileWatcher implements IFileWatcher {

  public static final WatchEvent.Modifier[] DEFAULT_MODIFIERS = {
    SensitivityWatchEventModifier.HIGH
  };
  private final Logger LOG = LoggerFactory.getLogger(DefaultFileWatcher.class);
  private final List<IFileWatchEventListener> listeners;
  private final WatchService watchService;
  private volatile boolean isWatching = false;
  /** O diretório monitorado */
  private Path watchedDirectoryPath;

  private ExecutorService taskThreadPool;
  /** Usado para filtar as notificações */
  private FileFilter fileFilter;

  public DefaultFileWatcher(Integer threadPoolSize) throws IOException {
    this(FileSystems.getDefault().newWatchService(), new CopyOnWriteArrayList<>(), threadPoolSize);
  }

  /**
   * Construtor para os testes.
   *
   * @param watchService
   * @param listeners
   */
  public DefaultFileWatcher(
      WatchService watchService, List<IFileWatchEventListener> listeners, Integer threadPoolSize) {
    this.watchService = watchService;
    this.listeners = listeners;
    taskThreadPool = Executors.newFixedThreadPool(threadPoolSize);
  }

  @Override
  public void register(final String dirStrPath, final FileFilter fileFilter) throws IOException {
    LOG.debug("[Default Watcher] registering path [{}]", dirStrPath);
    Path watchedDirectoryPath = Paths.get(dirStrPath);
    if (!Files.isDirectory(watchedDirectoryPath)) {
      final String msg = format("The path %s is not a directory!", dirStrPath);
      LOG.error(msg);
      throw new IOException(msg);
    }
    LOG.info(
        "Registering Default Watcher for change events on directory {} [Real path={}]",
        dirStrPath,
        getRealPath(watchedDirectoryPath));
    // Só é necessário ouvir os eventos de modificação dos arquivos (eventos de criação/remoção são ignorados)
    WatchEvent.Kind<?>[] events = { StandardWatchEventKinds.ENTRY_MODIFY };
    watchedDirectoryPath.register(watchService, events, DEFAULT_MODIFIERS);
    this.watchedDirectoryPath = watchedDirectoryPath;
    this.fileFilter = fileFilter;
  }

  @Override
  public void startWatch() throws InterruptedException {
    LOG.info("Default Watcher Service started...");
    isWatching = true;
    while (isWatching) {
      WatchKey key;
      try {
        key = watchService.take();
      } catch (ClosedWatchServiceException e) {
        LOG.debug("Default Watcher was closed on watch service close...");
        // isWatching = false;
        break;
      }
      if (key != null) {
        for (WatchEvent<?> watchEvent : key.pollEvents()) {
          LOG.debug(
              "Got file changed event: {} with context: {}",
              watchEvent.kind(),
              watchEvent.context());
          taskThreadPool.execute(() -> processEvent(watchEvent));
        }
        key.reset();
      }
    }
  }

  @Override
  public void stopWatch() {
    isWatching = false;
  }

  public Path getWatchedDirectoryPath() {
    return watchedDirectoryPath;
  }

  @Override
  public void addFileWatchEventListener(IFileWatchEventListener listener) {
    this.listeners.add(listener);
  }

  @Override
  public void removeFileWatchEventListener(IFileWatchEventListener listener) {
    this.listeners.remove(listener);
  }

  @Override
  public void close() throws IOException {
    LOG.debug("Default Watcher was closed...");
    stopWatch();
    this.watchService.close();
  }

  public boolean isWatching() {
    return isWatching;
  }

  protected synchronized void processEvent(WatchEvent<?> watchEvent) {
    WatchEvent.Kind<?> kind = watchEvent.kind();
    WatchEvent<Path> pathEvent = (WatchEvent<Path>) watchEvent;
    // Relativo a watchedDirectoryPath. Ver @WatchEvent#context
    Path changedFilePath = pathEvent.context();
    File resolvedChangedFile= watchedDirectoryPath.resolve(changedFilePath).toFile();
    if(this.fileFilter == null || this.fileFilter.accept(resolvedChangedFile)) {
      FileEvent fe = new FileEvent(resolvedChangedFile);
      LOG.info(
          "Default Watcher Service notifying change event for file {} [Real path={}]",
          resolvedChangedFile,
          getRealPath(resolvedChangedFile.toPath()));
      for (IFileWatchEventListener listener : this.listeners) {
        if (ENTRY_MODIFY == kind) {
          listener.onFileModified(fe);
        }
      }
    }
  }

  private Path getRealPath(Path path) {
    Path realPath = null;
    try {
      realPath = path.toRealPath();
    } catch (IOException e) {
      LOG.error("Erro while getting real path for {}", path);
    }

    return realPath;
  }
}
