package br.pucrio.tecgraf.soma.job.log.watcher.impl;

import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatchEventListener;
import br.pucrio.tecgraf.soma.job.log.watcher.interfaces.IFileWatcher;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.*;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/** The Job Logs file watcher that watches for changes in files. */
public class JobLogFileWatcher implements IFileWatcher {

  /** Regular file paths to be monitored. */
  private List<String> monitoredFilePaths;

  private final IFileWatcher fileWatcher;

  /**
   * Constructor with a list of listeners for test purposes.
   *
   * @param watchService
   * @param listeners
   */
  public JobLogFileWatcher(List<IFileWatchEventListener> listeners) {
    this(new PollingFileWatcher(listeners, 1000));
  }

  /**
   * Constructor with a file watcher.
   *
   * @param watchService
   * @param listeners
   */
  public JobLogFileWatcher(IFileWatcher watcher) {
    if(watcher == null) {
      throw new IllegalArgumentException("File Watcher cannot be null");
    }
    this.fileWatcher = watcher;
    initFilePaths();
  }

  /**
   * Watch a list of file for changes.
   *
   * @return the watched resource.
   * @throws IOException Exception in register
   */
  public void register() throws IOException {
    String parentFileStrPath = getParentFilePath(monitoredFilePaths);
    this.fileWatcher.register(parentFileStrPath, new FileFilter() {
      @Override
      public boolean accept(File file) {
        Path fileChanged = file.toPath();
        for (String fileStrPath : monitoredFilePaths) {
          String fileName = Paths.get(fileStrPath).toFile().getName();
          if (fileChanged.toFile().getName().endsWith(fileName)) {
            return true;
          }
        }
        return false;
      }
    });
  }

  /**
   * Get and check parent files from file list.
   *
   * @param fileStrPaths file list to be watched.
   * @return The parent file string path or an exception in case of any files are not siblings.
   */
  protected String getParentFilePath(List<String> fileStrPaths) {
    final String fileStrPath = fileStrPaths.get(0);
    Path parentFilePath = Paths.get(fileStrPath).getParent();
    String parentFileStrAbsPath = parentFilePath.toFile().getAbsolutePath();
    for (int i = 1; i < fileStrPaths.size() - 1; i++) {
      String childStrFilePath = fileStrPaths.get(i);
      Path parentChildFilePath = Paths.get(childStrFilePath).getParent();
      if (!parentFileStrAbsPath.equals(parentChildFilePath.toFile().getAbsolutePath())) {
        throw new IllegalArgumentException("Monitoring files have not the same parent directory!");
      }
    }
    return parentFileStrAbsPath;
  }

  public List<String> getMonitoredFilePaths() {
    return Collections.unmodifiableList(this.monitoredFilePaths);
  }

  public synchronized void addMonitoredFilePath(String filePath) {
    if (!this.monitoredFilePaths.contains(filePath)) {
      this.monitoredFilePaths.add(filePath);
    }
  }

  public synchronized void addAllMonitoredFilePath(List<String> filePaths) {
    if (!this.monitoredFilePaths.containsAll(filePaths)) {
      this.monitoredFilePaths.addAll(filePaths);
    }
  }

  public synchronized void removeMonitoredFilePath(String filePath) {
    this.monitoredFilePaths.remove(filePath);
  }

  public boolean isMonitoredFilePathsEmpty() {
    return this.monitoredFilePaths.isEmpty();
  }

  private void initFilePaths() {
    this.monitoredFilePaths = new CopyOnWriteArrayList<>();
  }

  @Override
  public void startWatch() throws InterruptedException {
    this.fileWatcher.startWatch();
  }

  @Override
  public void stopWatch() throws Exception {
    this.fileWatcher.stopWatch();
  }

  @Override
  public void close() throws IOException {
      this.fileWatcher.close();
  }

  @Override
  public void register(String dirStrPath, FileFilter filter) throws IOException {
    this.fileWatcher.register(dirStrPath, filter);
  }

  @Override
  public void addFileWatchEventListener(IFileWatchEventListener listener) {
    this.fileWatcher.addFileWatchEventListener(listener);
  }

  @Override
  public void removeFileWatchEventListener(IFileWatchEventListener listener) {
    this.fileWatcher.removeFileWatchEventListener(listener);
  }

  @Override
  public boolean isWatching() {
    return this.fileWatcher.isWatching();
  }
}
