package br.pucrio.tecgraf.soma.logsmonitor.flow;

import br.pucrio.tecgraf.soma.logsmonitor.model.Topic;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.ErrorType;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.ResourceErrorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitorListener;
import br.pucrio.tecgraf.soma.logsmonitor.websocket.WebSocketNotificatioErrorService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Flow;
import java.util.concurrent.SubmissionPublisher;

/*
O Uso do escopo "prototype" faz com que o Spring crie uma nova instância dessa classe a cada vez que fizer a injeção
dela. E esse é o comportamento desejado ao chamar o método PublisherManager#subscribe.

Mais detalhes em:
https://docs.spring.io/spring-framework/docs/5.3.3/reference/html/core.html#beans-factory-scopes-prototype
 */
@Component
@Scope("prototype")
public class TopicPublisher {
  private static final Log logger = LogFactory.getLog(TopicPublisher.class);

  private final SubmissionPublisher<ResourceMonitorEvent> delegate;

  private final Map<String, TopicSubscriber> subscribers;

  private final Topic topic;
  private final Integer topicUUID;

  private final ResourceMonitorListener<ResourceMonitorEvent> listener;
  private Flow.Subscription monitorSubscription;

  @Autowired private WebSocketNotificatioErrorService webSocketErrorService;

  public TopicPublisher(Topic topic) {
    logger.debug(String.format("Topic [%s]: new TopicPublisher", topic));
    this.topic = topic;
    this.topicUUID = topic.getUUID();
    this.delegate = new SubmissionPublisher<>();
    this.subscribers = new HashMap<>();
    this.listener =
        new ResourceMonitorListener<>() {
          @Override
          public void onSubscribe(Flow.Subscription subscription) {
            logger.debug(
                String.format(
                    "MonitorListener: onSubscribe activated for topic [%s] on its MonitorListener",
                    topic));
            monitorSubscription = subscription;
          }

          @Override
          public void onNext(ResourceMonitorEvent event) {
            logger.debug(
                String.format(
                    "MonitorListener: onNext activated for topic [%s] on its MonitorListener",
                    topic));
            if (!delegate.isClosed()) {
              delegate.submit(event);
            }
          }

          @Override
          public void onError(Throwable throwable) {
            logger.debug(
                String.format(
                    "MonitorListener: onError activated for topic [%s] on its MonitorListener",
                    topic));
            notifyError(throwable.getMessage());
            delegate.close();
          }

          @Override
          public void onComplete() {}
        };
  }

  public Topic getTopic() {
    return topic;
  }

  public ResourceMonitorListener<ResourceMonitorEvent> getListener() {
    return listener;
  }

  public synchronized boolean hasSubscribers() {
    return !subscribers.isEmpty();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    TopicPublisher that = (TopicPublisher) o;
    return topicUUID.equals(that.topicUUID);
  }

  @Override
  public int hashCode() {
    return Objects.hash(topicUUID);
  }

  public synchronized boolean isSubscribed(String sessionId) {
    return subscribers.containsKey(sessionId);
  }

  public synchronized void subscribe(String sessionId, TopicSubscriber subscriber) {
    logger.debug(String.format("Topic [%s]: subscribing session [%s]", topic, sessionId));
    if (!subscribers.containsKey(sessionId)) {
      delegate.subscribe(subscriber);
      subscribers.put(subscriber.getSessionId(), subscriber);
    }
    logger.debug(
        String.format(
            "Topic [%s]: active sessions numbers [%d]", topic, subscribers.keySet().size()));
  }

  public synchronized void unsubscribe(String sessionId) {
    logger.debug(String.format("Topic [%s]: unsubscribing session [%s]", topic, sessionId));
    TopicSubscriber subscriber = subscribers.remove(sessionId);
    subscriber.onComplete();
    logger.debug(
        String.format(
            "Topic [%s]: active sessions numbers [%d]", topic, subscribers.keySet().size()));
    if (!hasSubscribers()) {
      logger.debug(String.format("Topic [%s]: removing listener from monitor", topic));
      if (monitorSubscription != null) {
        monitorSubscription.cancel();
      }
      // indirect onComplete subscriber call
      delegate.close();
    }
  }

  private synchronized void notifyError(String detailsErrorMsg) {
    Map<String, Object> map = new HashMap<>();
    map.put(ResourceErrorEvent.JSON_PROPERTY_DETAILS, detailsErrorMsg);
    for (String sessionId : subscribers.keySet()) {
      TopicSubscriber subscriber = subscribers.get(sessionId);
      webSocketErrorService.onErrorNotify(
          sessionId,
          subscriber.getSubscriptionId(),
          ErrorType.RESOURCE_ERROR,
          "Internal error in monitor",
          map);
    }
  }
}
