package br.pucrio.tecgraf.soma.logsmonitor.service;

import br.pucrio.tecgraf.soma.job.log.monitor.impl.JobLogMonitor;
import br.pucrio.tecgraf.soma.logsmonitor.flow.TopicPublisher;
import br.pucrio.tecgraf.soma.logsmonitor.manager.PublisherManager;
import br.pucrio.tecgraf.soma.logsmonitor.model.JobLogsTopic;
import br.pucrio.tecgraf.soma.logsmonitor.model.Topic;
import br.pucrio.tecgraf.soma.logsmonitor.model.TopicType;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.ErrorType;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.ProjectPermissionErrorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.ResourceErrorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitor;
import br.pucrio.tecgraf.soma.logsmonitor.monitor.ResourceMonitorEvent;
import br.pucrio.tecgraf.soma.logsmonitor.utils.CodeUtils;
import br.pucrio.tecgraf.soma.logsmonitor.utils.ConstantsUtils;
import br.pucrio.tecgraf.soma.logsmonitor.utils.FileUtils;
import br.pucrio.tecgraf.soma.logsmonitor.websocket.WebSocketNotificatioErrorService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.socket.WebSocketSession;

import java.nio.file.Paths;
import java.util.*;

@Service
public class JobLogsTopicService implements TopicService {
  private static final Log logger = LogFactory.getLog(JobLogsTopicService.class);

  @Value("${job.logs.monitor.rootdir}")
  public String rootDir;

  @Value("${job.logs.monitor.logsdir.pattern}")
  public String logsDirPattern;

  @Autowired private PublisherManager publisherManager;

  @Autowired
  @Qualifier("JobLogsResourceMonitor")
  private ResourceMonitor monitor;

  @Autowired private ProjectService projectService;
  @Autowired private WebSocketNotificatioErrorService webSocketErrorService;

  @Override
  public TopicType getTopicType() {
    return TopicType.JOBLOGS;
  }

  @Override
  public boolean subscribe(
      Topic topic, WebSocketSession session, String subscriptionId, Long seqnum) {
    String sessionId = session.getId();
    String token = getToken(session);
    JobLogsTopic jobLogsTopic = (JobLogsTopic) topic;

    // Decodifica o JobId e ProjectId
    String decodedJobId = CodeUtils.decode(jobLogsTopic.getJobId());
    String decodedProjectId = CodeUtils.decode(jobLogsTopic.getProjectId());

    // Converte o jobId e projectId para paths
    String jobsIdDecodedFixed = FileUtils.fixDirectoryName(decodedJobId);

    String resourceId =
        getLogsDirPath(jobsIdDecodedFixed, decodedProjectId, jobLogsTopic.getLogName());

    try {
      if (!userHasProjectPermission(token, decodedProjectId)) {
        notifyProjectPermissionError(
            subscriptionId, sessionId, "User has no project permission", decodedProjectId);
        return false;
      }
    } catch (ServiceException e) {
      notifyJobLogFileError(
          subscriptionId,
          sessionId,
          "Error while querying for user project permissions",
          e.getMessage());
      return false;
    }

    TopicPublisher publisher = null;
    try {
      publisher = publisherManager.subscribe(jobLogsTopic, seqnum, sessionId, subscriptionId);
    } catch (IllegalArgumentException e) {
      notifyJobLogFileError(
          subscriptionId, sessionId, "Error while subscribing to a topic", e.getMessage());
      return false;
    }

    // Adiciona os argumentos ao iniciar o monitor
    Map<String, Object> params = new HashMap<>();
    params.put(JobLogMonitor.TIMESTAMP_PARAMETER, seqnum);
    params.put(JobLogMonitor.ENCODING_PARAMETER, jobLogsTopic.getEncoding());

    monitor.addListener(resourceId, publisher.getListener(), params);

    return true;
  }

  private boolean userHasProjectPermission(String token, String decodedProjectId) {
    return projectService.hasUserProjectPermission(token, decodedProjectId);
  }

  private void notifyProjectPermissionError(
      String subscriptionId, String sessionId, String shortErrorDescr, String projectId) {
    Map<String, Object> map = new HashMap<>();
    map.put(ProjectPermissionErrorEvent.JSON_PROPERTY_PROJECT_ID, projectId);
    webSocketErrorService.onErrorNotify(
        sessionId, subscriptionId, ErrorType.PROJECT_PERMISSION_ERROR, shortErrorDescr, map);
  }

  private void notifyJobLogFileError(
      String subscriptionId, String sessionId, String shortErrorDescr, String detailsErrorMsg) {
    Map<String, Object> map = new HashMap<>();
    map.put(ResourceErrorEvent.JSON_PROPERTY_DETAILS, detailsErrorMsg);
    webSocketErrorService.onErrorNotify(
        sessionId, subscriptionId, ErrorType.RESOURCE_ERROR, shortErrorDescr, map);
  }

  @Override
  public boolean unsubscribe(String sessionId, Topic topic) {
    JobLogsTopic jobLogsTopic = (JobLogsTopic) topic;

    Optional<TopicPublisher> publisherOptional =
        publisherManager.unsubscribe(jobLogsTopic, sessionId);

    return true;
  }

  @Override
  public List<ResourceMonitorEvent> getEvents(Topic topic, Long startSeqnum, Long endSeqnum) {
    JobLogsTopic jobLogsTopic = (JobLogsTopic) topic;

    // Decodifica o JobId e ProjectId
    String decodedJobId = CodeUtils.decode(jobLogsTopic.getJobId());
    String decodedProjectId = CodeUtils.decode(jobLogsTopic.getProjectId());

    // Converte o jobId e projectId para paths
    String jobsIdDecodedFixed = FileUtils.fixDirectoryName(decodedJobId);

    String resourceId =
        getLogsDirPath(jobsIdDecodedFixed, decodedProjectId, jobLogsTopic.getLogName());

    return this.monitor.getEvents(resourceId, startSeqnum, endSeqnum);
  }

  private String getLogsDirPath(String jobId, String projectId, String fileName) {
    String logsDirPath = logsDirPattern;
    logsDirPath = logsDirPath.replaceAll("__ROOT__", rootDir);
    logsDirPath = logsDirPath.replaceAll("__PROJECT__", projectId);
    logsDirPath = logsDirPath.replaceAll("__JOB__", jobId);
    logger.debug(String.format("Reading file data from path: %s", logsDirPath));
    return Paths.get(logsDirPath, fileName).toAbsolutePath().toString();
  }

  private String base64Decode(String s) throws IllegalArgumentException {
    return new String(Base64.getUrlDecoder().decode(s));
  }

  private String getToken(WebSocketSession session) {
    return (String)
        session.getAttributes().get(ConstantsUtils.WEBSOCKET_SESSION_ATTRIBUTES_TOKEN_KEY);
  }
}
